package ru.yandex.chemodan.app.stat.limits.whitelist;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.Zk;
import ru.yandex.commune.zk2.primitives.ZkPathClient;
import ru.yandex.commune.zk2.primitives.group.GroupWatcherListener;
import ru.yandex.commune.zk2.primitives.group.ZkGroupWatcher;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Lev Tolmachev
 */
public class WhitelistRegistry extends ZkPathClient implements GroupWatcherListener {

    private static final Logger logger = LoggerFactory.getLogger(WhitelistRegistry.class);

    private SetF<WhitelistEntry> whitelist = Cf.set();
    private SetF<DownloadStatId> ids = Cf.set();

    private final ZkGroupWatcher groupWatcher;

    public WhitelistRegistry(ZkPath zkPath) {
        super(zkPath);
        this.groupWatcher = new ZkGroupWatcher(zkPath, this);
    }

    public boolean isInWhitelist(DownloadStatId id) {
        return getIds().containsTs(id);
    }

    public void addToWhiteList(DownloadStatId id, String description) {
        saveWhitelist(whitelist
                .filter(WhitelistEntry.getIdF().andThenEquals(id).notF())
                .plus(new WhitelistEntry(id, description)));
    }

    public void removeFromWhiteList(DownloadStatId id) {
        saveWhitelist(whitelist.filter(WhitelistEntry.getIdF().andThenEquals(id).notF()));
    }

    private void saveWhitelist(SetF<WhitelistEntry> whitelist) {
        ZkPath path = this.path.child("list");
        zk().deleteIfExists(path);
        zk().createOrUpdate(path,
                WhitelistEntry.PS.getSerializer().serializeListJson(whitelist.toList()));
    }

    public SetF<WhitelistEntry> getWhitelist() {
        return whitelist;
    }

    public SetF<DownloadStatId> getIds() {
        return ids;
    }

    @Override
    public void initialize(Zk zk) {
        super.initialize(zk);
        groupWatcher.initialize(zk);
    }

    @Override
    public void expired() {
        super.expired();
        groupWatcher.expired();
    }

    @Override
    public void groupChanged(ListF<byte[]> members) {
        whitelist = members.flatMap(bytes -> WhitelistEntry.PS.getParser().parseListJson(new ByteArrayInputStreamSource(bytes))).unique();
        ids = whitelist.map(WhitelistEntry.getIdF()).unique();
    }
}
