package ru.yandex.chemodan.app.stat.log;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.DiskStatManager;
import ru.yandex.chemodan.app.stat.DownloadStatChannels;
import ru.yandex.chemodan.app.stat.storage.OneChannelStats;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.CacheUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Meter;
import ru.yandex.misc.monica.core.blocks.Statistic;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

import static ru.yandex.chemodan.app.stat.storage.OneChannelStats.of;

/**
 * @author Lev Tolmachev
 */
@ListenerTskvFormat("ydisk-downloader-access-log")
public class DownloadStatLogListener implements LogListener, MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(DownloadStatLogListener.class);
    private final DiskStatManager diskStatManager;
    private final LogLineParser logLineParser;

    private final Cache<String, Boolean> tokensCache = CacheUtils.newLru(100000);

    @MonicaMetric
    @GroupByDefault
    private final Meter views = new Meter();

    @MonicaMetric
    @GroupByDefault
    private final Meter bytesMeter = new Meter();

    @MonicaMetric
    @GroupByDefault
    private Statistic speed = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic speedLimited = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private final Meter sessionsNotLimited = new Meter();

    @MonicaMetric
    @GroupByDefault
    private final Meter sessionsLimited = new Meter();

    @MonicaMetric
    @GroupByDefault
    private Statistic time1MbLimited = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic time1MbNotLimited = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic time1MbLimitedSasulca = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic time1MbNotLimitedSasulca = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic time1MbLimitedNotSasulca = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic time1MbNotLimitedNotSasulca = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic timingsLimited = new Statistic();

    @MonicaMetric
    @GroupByDefault
    private Statistic timingsNotLimited = new Statistic();

    public DownloadStatLogListener(DiskStatManager diskStatManager, LogLineParser logLineParser) {
        this.diskStatManager = diskStatManager;
        this.logLineParser = logLineParser;
    }

    public static boolean isMulca(String hostname) {
        return hostname.contains("sasulca");
    }

    @Override
    public void processLogLine(String line) {
        Option<DownloadParameters> parametersO = logLineParser.parse(line);
        if (!parametersO.isPresent()) {
            return;
        }
        DownloadParameters parameters = parametersO.get();

        if (!HttpStatus.is2xx(parameters.code)) {
            return;
        }

        int views = tokensCache.getFromCache(parameters.rtoken).isPresent() ? 0 : 1;
        tokensCache.putInCache(parameters.rtoken, true);

        OneChannelStats statsDelta = of(!parameters.hash.isPresent(), parameters.traffic, views);

        boolean needIncrementMetric = false;
        if (parameters.hid.isPresent()) {
            diskStatManager.incFileDownloadStat(parameters.hid.get(), parameters.hash, parameters.name,
                    parameters.mediaType, DownloadStatChannels.SIMPLE, statsDelta);
            needIncrementMetric = true;
        }

        if (parameters.hash.isPresent()) {
            diskStatManager.incHashDownloadStat(parameters.hash.get(), statsDelta);
            needIncrementMetric = true;
        }

        if (needIncrementMetric) {
            incrementMetrics(parameters, views);
        }
    }

    private void incrementMetrics(DownloadParameters parameters, int views) {
        bytesMeter.inc(parameters.traffic.toBytes());

        if (views > 0) {
            this.views.inc();
        }

        if (parameters.limited) {
            timingsLimited.update(parameters.requestTimeSeconds);
        } else {
            timingsNotLimited.update(parameters.requestTimeSeconds);
        }

        if (parameters.requestTimeSeconds != 0) {
            double speedKb = parameters.traffic.toKiloBytes() / parameters.requestTimeSeconds;
            if (parameters.limited) {
                speedLimited.update(speedKb);
                sessionsNotLimited.inc();
            } else {
                speed.update(speedKb);
                sessionsLimited.inc();
            }

            if (speedKb != 0) {
                double time = DataSize.fromMegaBytes(1).toKiloBytes() / speedKb;
                if (parameters.limited) {
                    time1MbLimited.update(time);
                } else {
                    time1MbNotLimited.update(time);
                }
                if (isMulca(parameters.downloaderHost)) {
                    if (parameters.limited) {
                        time1MbLimitedSasulca.update(time);
                    } else {
                        time1MbNotLimitedSasulca.update(time);
                    }
                } else {
                    if (parameters.limited) {
                        time1MbLimitedNotSasulca.update(time);
                    } else {
                        time1MbNotLimitedNotSasulca.update(time);
                    }
                }
            }
        }
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "downloader-salr",
                new MetricName("salr", "downloader"),
                "Downloader log reader"
        );
    }
}
