package ru.yandex.chemodan.app.stat.log;

import javax.annotation.Nonnull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Lev Tolmachev
 */
public class TskvLogLineParser implements LogLineParser {

    private static final String REQUEST = "request";
    private static final String STATUS = "status";
    private static final String BYTES_SEND = "bytes_sent";
    private static final String REQUEST_TIME = "request_time";
    private static final String IP = "ip";
    private static final String VHOST = "vhost";

    private static final String HID = "hid";
    private static final String HASH = "hash";
    private static final String NAME = "filename";
    private static final String RTOKEN = "rtoken";
    private static final String MEDIA_TYPE = "media_type";
    private static final String LIMITED = "limit";


    private static final SetF<String> filesNecessaryParameters = Cf.set(HASH, RTOKEN, MEDIA_TYPE);
    private static final SetF<String> zipNecessaryParameters = Cf.set(HASH, RTOKEN);

    @Override
    public Option<DownloadParameters> parse(String line) {
        MapF<String, String> m = TskvUtils.extractTskv(line);
        if (!m.containsKeyTs(REQUEST) || !m.containsKeyTs(STATUS) || !m.containsKeyTs(BYTES_SEND)) {
            return Option.empty();
        }
        String request = m.getTs(REQUEST);

        int question = request.indexOf('?');
        if (question == -1 || question == request.length() - 1) {
            return Option.empty();
        }
        String path = request.substring(0, question);
        String parameters = request.substring(question + 1);
        if (path.startsWith("/rdisk")) {
            MapF<String, String> parametersMap = UrlUtils.urlParametersToTuple2List(parameters).toMap();
            if (parametersMap.keys().containsAllTs(filesNecessaryParameters)) {
                Option<MpfsHid> hid = parametersMap.getOrElse(HID, "None").equals("None")
                                      ? Option.empty()
                                      : Option.of(new MpfsHid(parametersMap.getTs(HID)));

                Option<String> hash = parametersMap.getTs(HASH).isEmpty()
                                      ? Option.empty()
                                      : Option.of(transformHash(parametersMap.getTs(HASH)));

                Option<String> name = parametersMap.getTs(NAME).isEmpty()
                                      ? Option.empty()
                                      : Option.of(parametersMap.getTs(NAME));

                Option<String> mediaType = parametersMap.getTs(MEDIA_TYPE).isEmpty()
                                           ? Option.empty()
                                           : Option.of(parametersMap.getTs(MEDIA_TYPE));

                String rtoken = parametersMap.getTs(RTOKEN);

                int code = Cf.Integer.parse(m.getTs(STATUS));
                DataSize traffic = new DataSize(Cf.Long.parse(m.getTs(BYTES_SEND)));

                Option<IpAddress> ip = m.getO(IP).flatMap(IpAddress.parseIpAddressOF()).firstO();

                double requestTimeSeconds = Cf.Double.parse(m.getTs(REQUEST_TIME));

                boolean limited = parametersMap.getO(LIMITED).filter(Cf.String.equalsF("1")).isNotEmpty();

                String downloaderHost = m.getTs(VHOST);

                return Option.of(new DownloadParameters(code, hid, traffic, hash, name, mediaType, rtoken, ip,
                        requestTimeSeconds, limited, downloaderHost));
            }
        }

        if (path.startsWith("/rzip/")) {
            MapF<String, String> parametersMap = UrlUtils.urlParametersToTuple2List(parameters).toMap();
            if (parametersMap.keys().containsAllTs(zipNecessaryParameters)) {
                Option<String> hash = parametersMap.getTs(HASH).isEmpty()
                                      ? Option.empty()
                                      : Option.of(transformHash(parametersMap.getTs(HASH)));

                Option<String> name = parametersMap.getTs(NAME).isEmpty()
                                      ? Option.empty()
                                      : Option.of(parametersMap.getTs(NAME));

                String rtoken = parametersMap.getTs(RTOKEN);

                int code = Cf.Integer.parse(m.getTs(STATUS));
                DataSize traffic = new DataSize(Cf.Long.parse(m.getTs(BYTES_SEND)));

                Option<IpAddress> ip = m.getO(IP).flatMap(IpAddress.parseIpAddressOF()).firstO();

                double requestTimeSeconds = Cf.Double.parse(m.getTs(REQUEST_TIME));

                boolean limited = parametersMap.getO(LIMITED).filter(Cf.String.equalsF("1")).isNotEmpty();

                String downloaderHost = m.getTs(VHOST);

                return Option.of(new DownloadParameters(code, Option.empty(), traffic, hash, name, Option.empty(),
                        rtoken, ip, requestTimeSeconds, limited, downloaderHost));
            }
        }

        return Option.empty();
    }

    private static String transformHash(@Nonnull String hash) {
        return StringUtils.substringBefore(hash, ":/");
    }
}
