package ru.yandex.chemodan.app.stat.storage;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.ReadPreference;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.admin.web.AdminApp;
import ru.yandex.commune.mongo.MongoUtils;
import ru.yandex.commune.mongo.admin.browser.MongoBrowserAdminPage;
import ru.yandex.inside.admin.conductor.Conductor;
import ru.yandex.inside.admin.conductor.GroupOrHost;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Lev Tolmachev
 */
@Configuration
public class DiskStatMongoContextConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(DiskStatMongoContextConfiguration.class);

    @Bean
    public MongoClient statMongoClient(
            @Value("${meta-mongo.hosts:-}")
            GroupOrHost hosts,
            @Value("${meta-mongo.port:-27028}")
            IpPort port,
            @Value("${mongo.connections.per.host:-50}")
            int connectionsPerHost,
            @Value("${mongo.connect.timeout.ms:-5000}")
            int connectTimeoutMs,
            @Value("${mongo.socket.timeout.ms:-5000}")
            int socketTimeoutMs,
            @Value("${mongo.pool.timeout.ms:-5000}")
            int poolWaitTimeoutMs,
            @Value("${mongo.threads.block.multiplier:-20}")
            int threadsAllowedToBlockForConnectionMultiplier,
            @Value("${mongo.w:-2}")
            int w,
            @Value("${mongo.fsync:-false}")
            boolean fsync,
            @Value("${mongo.wtimeout.ms:-2500}")
            int wtimeoutMs,
            Conductor conductor)
    {
        ListF<ServerAddress> hostList = conductor.hostsFromString(hosts).map(MongoUtils.consServerAddressF(port));

        MongoClientOptions mongoOptions = MongoClientOptions.builder()
                .connectionsPerHost(connectionsPerHost)
                .connectTimeout(connectTimeoutMs)
                .socketTimeout(socketTimeoutMs)
                .maxWaitTime(poolWaitTimeoutMs)
                .threadsAllowedToBlockForConnectionMultiplier(threadsAllowedToBlockForConnectionMultiplier)
                .writeConcern(new WriteConcern(w, wtimeoutMs, fsync))
                .readPreference(ReadPreference.primaryPreferred())
                .build();

        logger.info("Starting mongo client to hosts {} with options: {}", hostList,
                YandexToStringBuilder.reflectionToStringValueObject(mongoOptions));

        return new MongoClient(hostList, mongoOptions);
    }

    @Bean
    public MongoBrowserAdminPage diskStatMongoBrowserAdminPage(
            @Qualifier("statMongoClient") MongoClient mongoClient,
            AdminApp adminApp)
    {
        MongoBrowserAdminPage mongoBrowserAdminPage = new MongoBrowserAdminPage(mongoClient);
        adminApp.registerAdminPage("Stat mongo browser", Option.of("Db"), mongoBrowserAdminPage);
        return mongoBrowserAdminPage;
    }
}
