package ru.yandex.chemodan.app.stat.storage;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.stat.TimeUtils;
import ru.yandex.chemodan.app.stat.storage.util.DbUtils;
import ru.yandex.chemodan.app.stat.storage.util.DownloadStatIdMarshaller;
import ru.yandex.chemodan.app.stat.storage.util.DownloadStatIdUnmarshaller;
import ru.yandex.commune.mongo.MongoCollection;
import ru.yandex.commune.mongo.MongoDefaultInterceptors;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.impl.LruCache;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.clock.Clock;

public class DownloadStatDao {
    private static final Logger logger = LoggerFactory.getLogger(DownloadStatDao.class);

    private static final BenderConfiguration benderConfiguration = new BenderConfiguration(
            BenderConfiguration.defaultSettings(),
            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                    .add(DownloadStatId.class, new DownloadStatIdUnmarshaller())
                    .add(DownloadStatId.class, new DownloadStatIdMarshaller())
                    .build()
    );

    private final DB db;
    private final Duration period;
    private final Clock clock = Clock.DEFAULT;

    private final Cache<InstantInterval, MongoCollection<DownloadStatId, DownloadStat>> collectionsCache =
            new LruCache<>(100);

    public DownloadStatDao(DB db, Duration period) {
        this.db = db;
        this.period = period;
    }

    public Option<DownloadStat> getDownloadStats(DownloadStatId id, Instant time) {
        return getPeriodCollection(time).find(DbUtils.getIdQuery(id)).firstO();
    }

    public Option<DownloadStat> getDownloadStats(DownloadStatId id) {
        return getDownloadStats(id, clock.now());
    }

    public void removePeriodStatsCollection(Instant time) {
        getPeriodCollection(time).dropCollection();
    }

    public static String getCollectionName(InstantInterval interval) {
        return "download_" + TimeUtils.intervalToString(interval);
    }

    public DownloadStat incStats(DownloadStatId id, String channel, OneChannelStats stats) {
        return incStats(id, Cf.map(channel, stats));
    }

    private void addToInc(BasicDBObject inc, String prefix, ViewsAndTraffic viewsAndTraffic) {
        if (viewsAndTraffic.getTraffic().toBytes() != 0) {
            inc.append(prefix + ".t", viewsAndTraffic.getTraffic().toBytes());
        }
        if (viewsAndTraffic.getViews() != 0) {
            inc.append(prefix + ".c", viewsAndTraffic.getViews());
        }
    }

    public DownloadStat incStats(DownloadStatId statId, MapF<String, OneChannelStats> stats)
    {
        MongoCollection<DownloadStatId, DownloadStat> collection = getCurrentCollection();

        BasicDBObject inc = new BasicDBObject();
        for (Tuple2<String, OneChannelStats> t : stats.entries()) {
            addToInc(inc, "cs." + t._1 + ".a", t._2.getAuthStats());
            addToInc(inc, "cs." + t._1 + ".p", t._2.getPublicStats());
        }

        return collection.findAndUpdate(
                DbUtils.getIdQuery(statId), false, new BasicDBObject("$inc", inc), true, true)
                .firstO().get();
    }

    private MongoCollection<DownloadStatId, DownloadStat> getCollection(Instant start, Instant end) {
        return getCollection(new InstantInterval(start, end));
    }

    private MongoCollection<DownloadStatId, DownloadStat> getCollection(final InstantInterval interval) {
        return collectionsCache.getFromCacheSome(interval,
                () -> new MongoCollection<>(DbUtils.getCollection(db, interval, "download"),
                        DownloadStat.class, benderConfiguration, MongoDefaultInterceptors.defaultInterceptors()));
    }

    private MongoCollection<DownloadStatId, DownloadStat> getCurrentCollection() {
        return getPeriodCollection(clock.now());
    }

    private MongoCollection<DownloadStatId, DownloadStat> getPeriodCollection(Instant now) {
        Instant start = TimeUtils.roundTo(now, period);
        return getCollection(new InstantInterval(start, start.plus(period)));
    }

}
