package ru.yandex.chemodan.app.stat.storage;

import java.nio.charset.StandardCharsets;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import com.google.common.primitives.Bytes;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.lang.DefaultObject;

import static java.util.function.Function.identity;
import static ru.yandex.chemodan.app.stat.storage.DownloadStatType.HID;

@Bendable
public class DownloadStatId extends DefaultObject {
    private final byte[] idBytes;
    private final DownloadStatType type;

    private static final MapF<String, DownloadStatType> PREFIX_TO_TYPES = Cf.x(Stream.of(DownloadStatType.values())
            .collect(Collectors.toMap(type -> type.name() + ":", identity())));

    public DownloadStatId(@Nonnull MpfsHid hid) {
        this.idBytes = hid.data;
        this.type = HID;
    }

    private DownloadStatId(byte[] idBytes, DownloadStatType type) {
        this.idBytes = idBytes;
        this.type = type;
    }

    public DownloadStatId(String id, DownloadStatType type) {
        this.idBytes = id.getBytes(StandardCharsets.UTF_8);
        this.type = type;
    }

    public DownloadStatId(@Nonnull String hash) {
        this.idBytes = hash.getBytes(StandardCharsets.UTF_8);
        this.type = DownloadStatType.HASH;
    }

    public DownloadStatType getType() {
        return type;
    }

    public byte[] toMongoId() {
        if (type == HID) {
            return idBytes;
        }
        return Bytes.concat(
                type.name().getBytes(StandardCharsets.UTF_8),
                ":".getBytes(StandardCharsets.UTF_8),
                idBytes
        );
    }

    public static DownloadStatId fromMongoId(byte[] idBytes) {
        String representation = new String(idBytes, StandardCharsets.UTF_8);
        for (Tuple2<String, DownloadStatType> entry : PREFIX_TO_TYPES.entries()) {
            if (StringUtils.startsWithAny(representation, entry.get1())) {
                return new DownloadStatId(
                        StringUtils.removeStart(representation, entry.get1()).getBytes(StandardCharsets.UTF_8),
                        entry.get2());
            }
        }
        return new DownloadStatId(idBytes, HID);
    }

    public boolean isHid() {
        return type.equals(HID);
    }

    @Nonnull
    public MpfsHid toHid() {
        return new MpfsHid(idBytes);
    }

    @Nonnull
    public String getIdString() {
        return new String(idBytes, StandardCharsets.UTF_8);
    }

    @Override
    public String toString() {
        return DownloadStatId.class.getSimpleName() +
                "[" + type + ":" + new String(idBytes, StandardCharsets.UTF_8) + ']';
    }

}
