package ru.yandex.chemodan.app.stat.storage;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.stat.TimeUtils;
import ru.yandex.commune.mongo.MongoCollection;
import ru.yandex.commune.mongo.MongoDefaultInterceptors;
import ru.yandex.commune.mongo.MongoUtils;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Lev Tolmachev
 */
public class PeriodSummaryStatsDao {
    private static final Logger logger = LoggerFactory.getLogger(PeriodSummaryStatsDao.class);

    private final Duration period;
    private final MongoCollection<Instant, PeriodStats> collection;

    public PeriodSummaryStatsDao(DBCollection collection, Duration period) {
        this.period = period;
        this.collection = new MongoCollection<>(collection, PeriodStats.class,
                BenderConfiguration.defaultConfiguration(), MongoDefaultInterceptors.defaultInterceptors());
    }

    public Instant getCurrentPeriodStart() {
        return TimeUtils.getCurrentPeriod(period).getStart();
    }

    public void inc(MapF<String, OneChannelStats> channelStats) {
        inc(channelStats, getCurrentPeriodStart());
    }

    private void addToInc(BasicDBObject inc, String prefix, ViewsAndTraffic viewsAndTraffic) {
        if (viewsAndTraffic.getTraffic().toBytes() != 0) {
            inc.append(prefix + ".t", viewsAndTraffic.getTraffic().toBytes());
        }
        if (viewsAndTraffic.getViews() != 0) {
            inc.append(prefix + ".c", viewsAndTraffic.getViews());
        }
    }

    private void inc(MapF<String, OneChannelStats> stats, Instant periodStart) {
        logger.info("Increasing period stats. period: {}, stats: {}", periodStart, stats);
        BasicDBObject inc = new BasicDBObject();
        BasicDBObject update = new BasicDBObject("$inc", inc);

        for (Tuple2<String, OneChannelStats> t : stats.entries()) {
            addToInc(inc, "cs." + t._1 + ".a", t._2.getAuthStats());
            addToInc(inc, "cs." + t._1 + ".p", t._2.getPublicStats());
        }

        collection.update(new BasicDBObject("_id", MongoUtils.toMongoValue(periodStart)), update, true, false);
    }

    public PeriodStats getCurrentPeriodStats() {
        return getPeriodStats(getCurrentPeriodStart());
    }

    public PeriodStats getPeriodStats(Instant periodStart) {
        return collection.findById(periodStart).getOrElse(new PeriodStats(periodStart,
                Cf.map()));
    }

    public void clearPeriodStatsForTests(Instant periodStart) {
        collection.remove(new BasicDBObject("_id", MongoUtils.toMongoValue(periodStart)));
    }
}
