# Zaberoon v2 by Sergey Sytnik and Sergei Vavinov
# Description for Zaberoon v1 is available here:
# svn cat svn+ssh://svn.yandex.ru/mail/trunk/meta/yandex-disk-downloader/src/disk-secure-download.pm@85288 | head -n 27
# Documentation: http://wiki.yandex-team.ru/Pochta/ya.disk/downloader

package YandexDisk;

use nginx;
use strict;
use warnings;

use Digest::MD5 'md5_hex';
use Digest::SHA 'hmac_sha256_hex';
use URI::Escape;
use URI::Split 'uri_split', 'uri_join';
use MIME::Base64;

use Encode;


use constant {
    # request handler types
    REDIRECT => 1,
    GET_FILE => 2,

    # http://lxr.evanmiller.org/http/source/http/modules/perl/nginx.pm
    HTTP_UNPROCESSABLE      => 422,

    OK_INTERRUPT_AND_RETURN => "OK_INTERRUPT_AND_RETURN_",
};


# external (nginx public)

sub handleRedirect {
    return handle(redirectOrGetFileHandler(REDIRECT, @_));
}

sub handleGetFile {
    return handle(redirectOrGetFileHandler(GET_FILE, @_));
}


# internal

sub redirectOrGetFileHandler() {
    my ($type, $r) = @_;

    return sub {
        nginxLog($r, "== Starting (request type: $type): uri=" . $r->uri . ", params=" . $r->args);
        my $params = parseParams($r);
        #logParams($r, $params);
        logRangeQuery($r);

        validateToken($r, $params);
        validateTimestamp($r, $params);
        validatePreviewSize($r, $params);

        my $queryType = getParam($r, $params, 'query_type');
        if ($type == GET_FILE) {
            my ($get_file_location, $add_args) = ('', '');
            if ($queryType eq 'disk') {
                $get_file_location =
                    isPrivateFile($r, $params) ? 'int_get_private_file' : # always unlimited
                    getParam($r, $params, 'limit') == 1 ? 'int_get_limited_file' : 'int_get_unlimited_file';
            } elsif ($queryType eq 'zip') {
                $get_file_location =
                    isPrivateFile($r, $params) ? 'int_get_private_zip' : # always unlimited
                    getParam($r, $params, 'limit') == 1 ? 'int_get_limited_zip' : 'int_get_unlimited_zip';
                $add_args = 'path=' . uri_escape_utf8 getParamRequired($r, $params, 'file_path');
            } elsif ($queryType eq 'zip-files') {
                $get_file_location = 'int_get_private_zip_files';
                $add_args = 'oid=' . uri_escape_utf8 getParamRequired($r, $params, 'file_path');
            } elsif ($queryType eq 'zip-album') {
                $get_file_location = getParam($r, $params, 'limit') == 1 ? 'int_get_limited_zip_album' : 'int_get_unlimited_zip_album';
            } elsif ($queryType eq 'share') {
                $get_file_location = 'int_get_unlimited_file';
            } elsif ($queryType eq 'preview') {
                $get_file_location = isAlbumPreview($params) ? 'int_get_album_preview' : 'int_get_preview';
            } else {
                finalize($r, HTTP_BAD_REQUEST, "Unknown query type", ": $queryType");
            }
            my $uri = $r->uri . '?' . $r->args . ($add_args ? '&' . $add_args : '');
            finalize($r, HTTP_BAD_REQUEST, "Cannot match query type", ": /r$queryType") unless $uri =~ s|^/r$queryType|/$get_file_location|;

            setZaberunHostVariable($r) unless $queryType eq 'preview';
            internalRedirect($r, $uri);
        } else {
            # protection against infinite redirection with wrong config
            finalize($r, HTTP_BAD_REQUEST, "Request contains rtoken", ", check nginx config") if getParam($r, $params, 'rtoken') ne '';

            my $rToken = calcRToken();
            my $forceDefault = isPrivateFile($r, $params) ? 'yes' : 'no';
            my $ycrid = $r->variable('ycrid');
            my $uri = $r->uri . '?' . $r->args;
            if ($queryType eq 'share' && needToPatchInstaller($r, $params)) {
                # redirect directly to internal handle
                $uri =~ s|^/share|/int_patch_installer|;
            } else {
                $uri = $uri . "&rtoken=$rToken&force_default=$forceDefault&ycrid=$ycrid";
                finalize($r, HTTP_BAD_REQUEST, "Cannot match query type", ": /$queryType") unless $uri =~ s|^/$queryType|/r$queryType|;
            }
            internalRedirect($r, $uri);
        }
    };
}

sub handle {
    my ($handler) = @_;

    eval {
        $handler->();
        1;
    } or do {
        die $@ unless $@ =~ /^OK_INTERRUPT_AND_RETURN_(\d+)/;
        return $1;
    };
    return OK;
}

sub isPrivateFile {
    my ($r, $params) = @_;
    if (getParamRequired($r, $params, 'uid') eq '0') {
        return 0;
    } else {
        return 1;
    }
}

sub isAlbumPreview {
    my ($params) = @_;
    return defined($params->{'album_name'});
}

sub setZaberunHostVariable {
    my ($r) = @_;
    my (@parts) = uri_split $r->variable('upstream_http_location');
    $r->variable('zaberun_host', $parts[1]);
}

sub validateToken {
    my ($r, $params) = @_;
    doValidateToken($r, $params, 'token', [calcToken($r, $params, getUrlSecret())]);
}

sub doValidateToken {
    my ($r, $params, $inputTokenParamName, $allowedCalcTokens) = @_;
    my $inputToken = getParamRequired($r, $params, $inputTokenParamName, 0);
    if (! grep { $_ eq $inputToken } @$allowedCalcTokens) {
        finalize($r, HTTP_FORBIDDEN, "Invalid $inputTokenParamName: $inputToken", ", expected: @$allowedCalcTokens");
    }
}

sub calcToken {
    my ($r, $params, $secret) = @_;
    finalize($r, HTTP_BAD_REQUEST, 'Unsupported disposition')
        unless getParamRequired($r, $params, 'disposition', 0) =~ /^(inline|attachment)$/i;
    my @rNames = qw(timestamp file_path uid filename content_type disposition);
    my @rValues = map { getParamRequired($r, $params, $_, 0) } @rNames;
    my @oNames = qw(hash limit);
    my @oValues = map { getParam($r, $params, $_, 0) } @oNames;
    my $valuesForToken = [@rValues, @oValues];
    my $version = getParam($r, $params, 'tknv');
    return doCalcToken( $r, $version, $secret, $valuesForToken );
}

sub calcRToken {
    # In token v1 this token was calculated by cookie and timestamp,
    # but now it just random value, because anti-fo used this parameter
    # to identify unique requests
    # Now this parameter is used only by anti-fo
    my @set = ('0' ..'9', 'A' .. 'Z', 'a' .. 'z');
    return join '' => map $set[rand @set], 1 .. 12;
}

sub getSessionIdCookie {
    my ($r) = @_;
    return getCookie($r, 'Session_id');
}

sub getCookie {
    my ($r, $name) = @_;
    return getHeader($r, 'Cookie') =~ /\b$name=(.*?)(?:;|$)/ ? $1 : '';
}

sub doCalcToken {
    my ($r, $version, $secret, $valuesForToken) = @_;
    if ($version eq 'v2') {
        return hmac_sha256_hex(join('-', @$valuesForToken), $secret);
    } else {
        finalize($r, HTTP_FORBIDDEN, "Unsupported token format");
    }
}


sub validateTimestamp {
    my ($r, $params) = @_;
    doValidateTimestamp($r, $params, 'timestamp');
}

sub doValidateTimestamp {
    my ($r, $params, $tsParamName) = @_;

    my $tsHex = getParamRequired($r, $params, $tsParamName);
    return if $tsHex eq 'mpfs'; # CHEMODAN-6034
    return if $tsHex eq 'inf'; # unlimit link case

    unless ($tsHex =~ /[0-9a-f]+/) {
        finalize($r, HTTP_UNPROCESSABLE, "Failed to parse $tsParamName: $tsHex", "");
    }

    my $tsSec = int hex $tsHex;
    my $curTime = time();
    if ($curTime > $tsSec) {
        my $message = "Invalid $tsParamName: $tsHex";
        my $details = sprintf(" ttl = (%08x), current time = %08x", $tsSec, $curTime);
        finalize($r, HTTP_GONE, $message, $details);
    }
}


sub validatePreviewSize {
    my ($r, $params) = @_;
    if ((getParamRequired($r, $params, 'query_type') eq 'preview') and !isAlbumPreview($params)) {
        my $size = getParamRequired($r, $params, 'size');
        if (!isValidPreviewSizeParam($size)) {
            finalize($r, HTTP_BAD_REQUEST, "Incorrect size parameter", "")
        }
    }
}

sub isValidPreviewSizeParam {
    my ($size) = @_;
    if($size =~ /^(\d*x\d*)$|^(X{0,3}(L|S))$|^(x{0,3}(l|s))$|^(M|m)$/ && $size ne "x") {
        return 1;
    } else {
        return 0;
    }
}

sub getHeader {
    my ($r, $name) = @_;
    my $value = $r->header_in($name);
    return (defined($value) ? $value : '');
}

sub parseParams {
    my ($r, $singleNameO) = @_;
    my %params = ();

    my ($queryType, $token, $timestamp, $filePath) = ($r->uri =~ m|/r?([\w\-]+)/(\w+)/(\w+)/(.+)/?$|);
    $queryType = ($queryType or '');
    # this includes inner queries like 'int_get_file_unlimited', not only 'disk', 'preview' etc.
    $params{'query_type'} = $queryType;

    if (! defined($singleNameO) || grep { $singleNameO eq $_ } ('token', 'timestamp', 'file_path')) {
        finalize($r, HTTP_UNPROCESSABLE, "Undefined file path", "") unless defined($filePath);

        $filePath =~ tr|-_|+/|;
        my $decodedBase64 = decode_base64($filePath);
        if ($queryType =~ /^zip/ || $decodedBase64 =~ /^[\w\.\:]+$/) {
            $filePath = $decodedBase64;
        } else {
            $filePath = decrypt($filePath);
        }

        $params{'token'} = $token;
        $params{'timestamp'} = $timestamp;
        $params{'file_path'} = $filePath;
    }

    foreach my $pair (split '&', $r->args) {
        if ($pair =~ /^(.*?)=(.*)$/) {
            my ($name, $value) = ($1, $2);
            next if defined($singleNameO) && $singleNameO ne $name;
            if (not exists $params{$name}) {
                $value = uriUnescape($r, $value);
                $params{$name} = $value;
            }
        }
    }

    if ($queryType =~ /^zip/ && (! defined($singleNameO) || $singleNameO eq 'content_type')) {
        finalize($r, HTTP_BAD_REQUEST, "Request contains content_type", "") if getParam($r, \%params, 'content_type') ne '';
        $params{'content_type'} = 'application/zip';
    }

    return \%params;
}

sub parseGetParam {
    my ($r, $singleName) = @_;
    return getParam($r, parseParams($r, $singleName), $singleName);
}

sub getParam {
    my ($r, $params, $name, $rmEndLineSymbols) = @_;
    $rmEndLineSymbols = defined($rmEndLineSymbols) ? $rmEndLineSymbols : 1;
    my $value = $params->{$name};
    $value = defined($value) ? $value : '';
    # Remove endline symbols to avoid http splitting (CHEMODAN-34482)
    if ($rmEndLineSymbols) {
        $value = decode('utf8', $value);
        $value =~ s/\R//g;
    }
    return $value;
}

sub hasParam {
    my ($r, $params, $name) = @_;
    my $value = $params->{$name};
    return defined($value);
}

sub getParamRequired {
    my ($r, $params, $name, $rmEndLineSymbols) = @_;
    my $value = getParam($r, $params, $name, $rmEndLineSymbols);
    finalize($r, HTTP_UNPROCESSABLE, "No parameter: $name", "") if $value eq '';
    return $value;
}


sub uriUnescape {
    my ($r, $value) = @_;
    $value =~ s/\+/ /g;
    return uri_unescape($value);
}

sub logParams {
    my ($r, $params) = @_;
    nginxLog($r, "Params: " . (join ' ', map { $_ . '=' . $params->{$_} } keys %$params));
}

sub logRangeQuery {
    my ($r) = @_;
    my $rangeHeader = getHeader($r, 'Range');
    nginxLog($r, "Range: $rangeHeader") if $rangeHeader;
}


sub nginxLog {
    my ($r, $msg) = @_;
    $r->log_error(0, $msg);
}

sub internalRedirect {
    my ($r, $uri) = @_;
    nginxLog($r, "<- Redirecting: uri=$uri");
    $r->internal_redirect($uri);
    die OK_INTERRUPT_AND_RETURN . OK;
}

sub needToPatchInstaller {
    my ($r, $params) = @_;
    my $autoLogin = getParam($r, $params, "al");
    my $sessionId = getSessionIdCookie($r);
    return (($autoLogin eq "1") && ($sessionId ne "")) || hasParam($r, $params, "src") || hasParam($r, $params, "ouai");
}

# 'message + innerDetails' goes to inner log;
# 'message' goes to public http output
sub finalize {
    my ($r, $status, $message, $innerDetails) = @_;
    nginxLog($r, "<- Finalizing: status=$status, message=$message$innerDetails");
    if (grep {$_ == $status} (403, 404, 410, 500)) {
        die OK_INTERRUPT_AND_RETURN . $status;
    } else {
        $r->status($status);
        $r->header_out("Cache-Control", "no-store, no-cache");
        $r->send_http_header("text/plain");
        $r->print("HTTP $status\n$message\n");
        die OK_INTERRUPT_AND_RETURN . OK;
    }
}

return 1;
