# Zaberoon v2 by Sergey Sytnik and Sergei Vavinov
# Description for Zaberoon v1 is available here:
# svn cat svn+ssh://svn.yandex.ru/mail/trunk/meta/yandex-disk-downloader/src/disk-secure-download.pm@85288 | head -n 27
# Documentation: http://wiki.yandex-team.ru/Pochta/ya.disk/downloader

package YandexDisk;

use nginx;
use strict;
use warnings;
use Digest::MD5 qw(md5_hex);
use Digest::SHA 'hmac_sha256_hex';
use URI::Escape;
use Sys::Hostname;
use JSON;
use LWP::UserAgent;

# external (nginx public)

sub cat($) {
    my ($filename) = @_;
    open(my $fh, '<', $filename) or die "Could not open file '$filename' $!";
    my $type = <$fh>;
    close($fh);
    return $type;
}

my $TVMTOOL_AUTH = cat '/var/lib/tvmtool/local.auth';
my $ENVIRONMENT_TYPE = cat '/etc/yandex/environment.type';
my $SHORT_HOST_NAME = getShortHostName(hostname);
my @contentTypes = ("image/jpeg", "image/png", "image/gif", "image/x-ms-bmp", "image/tiff", "image/bmp", "image/x-png", "image/psd", "image/webp", "application/octet-stream", "application/ogg", "application/epub+zip", "text/json");

sub getRandomKey {
    my @set = ('0' ..'9', 'A' .. 'Z', 'a' .. 'z');
    return join '' => map $set[rand @set], 1 .. 8;
}

sub parseAvatarsStid {
    my ($r) = @_;
    if (varMulcaId($r) =~ /^ava:([^:]+):([^:]+):([^:]+)$/) {
        return ($1, $2, $3);
    } else {
        return;
    }
}

sub varMdsTicket {
    my $response = LWP::UserAgent->new->get(
        'http://localhost:1488/tvm/tickets?dsts=mds',
        'Authorization' => $TVMTOOL_AUTH,
    );
    die 'tvmtool failed' unless $response->is_success;
    return JSON->new->decode($response->decoded_content)->{mds}->{ticket};
}

sub varPreviewCacheKey {
    my ($r) = @_;
    if (parseAvatarsStid($r)) {
        return join '-',
            time,
            getRandomKey;
    } else {
        return join '-',
            'x',
            varMulcaId($r),
            varSize($r),
            varCrop($r),
            varLogo($r),
            varQuality($r),
            varPreviewType($r);
    }
}

sub varTimestamp {
    my ($r) = @_;
    return parseGetParam($r, 'timestamp');
}

sub varCheckAuth {
    my ($r) = @_;
    my $timestamp = parseGetParam($r, 'timestamp');
    my $uid = parseGetParam($r, 'uid');
    my $tknv = parseGetParam($r, 'tknv');
    my $queryType = parseGetParam($r, 'query_type');
    my $autoLogin = parseGetParam($r, 'al');

    if ($queryType eq 'share' and $autoLogin eq '1') {
        # always check auth for installer with auto-login code
        # https://st.yandex-team.ru/CHEMODAN-28487
        return "1";
    } elsif ($uid eq "0" or ($tknv eq "v2" and $timestamp ne "inf")) {
        # for token v2 check only infinite links (it should be preview)
        return "0";
    } else {
        return "1";
    }
}

sub varSize {
    my ($r) = @_;
    return parseGetParam($r, 'size');
}

sub varQuality {
    my ($r) = @_;
    return parseGetParam($r, 'quality');
}

sub varFileSize {
    my ($r) = @_;
    my $fileSize = parseGetParam($r, 'fsize');
    if ($fileSize ne '') {
        return $fileSize;
    } else {
        # hard-coded default value
        return 1000000;
    }
}

sub varCrop {
    my ($r) = @_;
    if (parseGetParam($r, 'crop') eq "1") {
        return "true";
    } else {
        return "false";
    }
}

sub varLogo {
    my ($r) = @_;
    my $logo = parseGetParam($r, 'logo');
    if ($logo eq "1") {
        return "north_west";
    } elsif ($logo eq "2") {
        return "center_new";
    } else {
        return "none";
    }
}

sub varMulcaId {
    my ($r) = @_;
    return parseGetParam($r, 'file_path');
}

sub varPreviewType {
    my ($r) = @_;
    return getPreviewType($r) || 'default';
}

sub varEncodedContentType {
    my ($r) = @_;
    return uri_escape 'image/webp' if getPreviewType($r) eq 'webp';
    return uri_escape parseGetParam($r, 'content_type');
}

sub varDecodedContentDisp {
    my ($r) = @_;
    return parseGetParam($r, 'disposition');
}

sub varLocation {
    my ($r) = @_;
    return getHeader($r, 'Location');
}

sub varAlbumName {
    my ($r) = @_;
    return uri_escape parseGetParam($r, 'album_name');
}

sub varUserName {
    my ($r) = @_;
    return uri_escape parseGetParam($r, 'user_name');
}

sub varFilenameMimeParam {
    my ($r) = @_;
    my $filename = parseGetParam($r, 'filename');
    my $ua = getHeader($r, 'User-agent');

    # https://jira.yandex-team.ru/browse/CHEMODAN-3681
    # http://greenbytes.de/tech/tc2231/
    # http://stackoverflow.com/questions/93551/how-to-encode-the-filename-parameter-of-content-disposition-header-in-http/6745788#6745788
    # UPDATE Mar 2013: http://stackoverflow.com/questions/7967079/special-characters-in-content-disposition-filename
    # Chrome does not like "'"
    # Opera does not like "/" and "\"
    # IE8 replaces "|" and spaces with "_"
    # Safari cannot handle ";" as it is not escaped, and separates header parameters
    $filename =~ s|\'|\"|g;
    $filename =~ s|[\\\/;]||g;
    if ($ua =~ /MSIE (7|8)\.0/) {
        my $escaped = uri_escape_utf8($filename);
        $escaped =~ s/%20/ /g;
        $escaped =~ s/%25/%/g;
        return "filename=$escaped";
    } elsif ($ua =~ /Android/ && $ua !~ /Firefox/) {
        return "filename=\"$filename\"";
    } else {
        return "filename*=UTF-8''" . uri_escape_utf8($filename);
    }
}

sub varDiskDoRedirect {
    my ($r) = @_;
    my $disposition = varDecodedContentDisp($r);
    my $contentType = uri_unescape(varEncodedContentType($r));
    if (($disposition ne 'inline') || (grep { $contentType eq $_ } @contentTypes) || ($contentType =~ /^(audio|video)/x)) {
        return "true"
    }

    return "false";
}

sub varResizerParams {
    my ($r) = @_;
    return getRandomKey() . '?' . getResizerParams($r);
}

sub varYcrid {
    my ($r) = @_;

    my $ycrid = getHeader($r, 'Yandex-Cloud-Request-ID');
    return $ycrid if $ycrid ne '';

    $ycrid = parseGetParam($r, 'ycrid');
    return $ycrid if $ycrid ne '';

    return join '-',
        getClientNameByUserAgent(getHeader($r, 'User-agent')),
        getYcridRandomPart(),
        $SHORT_HOST_NAME;
}

sub varAutoLogin {
    my ($r, $params) = @_;
    my $sessionId = getSessionIdCookie($r);
    my $autoLogin = parseGetParam($r, 'al');
    return (($autoLogin eq "1") && ($sessionId ne ""))
        ? "true"
        : "false";
}

sub varSource {
    my ($r, $params) = @_;
    return parseGetParam($r, 'src');
}

sub varEncodedOUAI {
    my ($r) = @_;
    return uri_escape parseGetParam($r, 'ouai');
}

sub getYcridRandomPart {
    my @set = ('0' ..'9', 'a' .. 'f');
    return join '' => map $set[rand @set], 1 .. 32;
}

sub getPreviewType {
    my ($r) = @_;
    my $previewType = parseGetParam($r, 'preview_type');
    $previewType ||= 'webp' if isWebpRequired($r);
    return $previewType;
}

sub varPreviewSource {
    my ($r) = @_;
    if (varMulcaId($r) =~ /^./ and varSize($r)) {
        return parseAvatarsStid($r) ? 'avatars' : 'resizer';
    } else {
        return "";
    }
}

sub varPreviewUrl {
    my ($r) = @_;
    my $stid = varMulcaId($r);
    $r->log_error(0, "previewStid=$stid");
    my $url;
    if (my @parts = parseAvatarsStid($r)) {
        $r->log_error(0, 'previewSource=avatars');
        $url = 'http://avatars/' . getAvatarsParams($r, @parts);
    } else {
        $r->log_error(0, 'previewSource=resizer');
        $url = 'http://resizer/' . varResizerParams($r);
    }
    $r->log_error(0, "previewUrl=$url");
    return $url;
}

sub getAvatarsThumbnail {
    my ($r) = @_;

    my ($quality, $crop, $watermark) = (varQuality($r), parseGetParam($r, 'crop'), getWatermarkType($r));
    my ($width, $height, $type) = (parseSize(varSize($r), $crop), parseGetParam($r, 'preview_type'));

    my $limit = getAvatarsSizeLimit();
    if (0 < $limit and parseGetParam($r, 'allow_big_size') ne '1') {
        $width = $limit if $width > $limit;
        $height = $limit if $height > $limit;
    }

    my $params = {};
    $params->{'typemap'} = sprintf '"*:%s"', $type if $type;
    $params->{'watermark'} = sprintf '"%s"', $watermark if $watermark ne '';
    $params->{'width'} = sprintf('"%s"', $width || "inf");
    $params->{'height'} = sprintf('"%s"', $height || "inf");
    $params->{'quality'} = $quality eq '' ? 85 : $quality;
    if ($crop eq '1') {
        $params->{'command'} = '"gravity"';
        $params->{'gravity-type'} = '"center"';
    } else {
        $params->{'command'} = '"resize"';
    }

    return uri_escape sprintf('{%s}', join ', ', map { sprintf '"%s":%s', $_, $params->{$_} } sort keys %$params), '=&.{}":, ;';
}

sub getAvatarsParams {
    my ($r, $ns, $group, $uid) = @_;
    my $thumbnail = getAvatarsThumbnail($r);
    my $sign = signAvatarsUrl($ns, $group, $uid, $thumbnail, getAvatarsSecret());
    return sprintf 'get-%s/%s/%s?thumbnail=%s&sign=%s', $ns, $group, $uid, $thumbnail, $sign;
}

sub signAvatarsUrl {
    my ($ns, $group, $uid, $thumbnail, $secret) = @_;
    my $message = sprintf(
        '/namespace=%s/group_id=%s/uid=%s/thumbnail=%s',
        $ns,
        $group,
        $uid,
        $thumbnail
    );
    return hmac_sha256_hex($message, $secret);
}

sub getResizerParams {
    my ($r) = @_;
    my $secret = getResizerSecret();

    my $mulcaId = varMulcaId($r);
    my $quality = varQuality($r);
    my $crop = parseGetParam($r, 'crop');
    my $watermark = getWatermarkType($r);
    my $host = getStorageHost();
    my $previewType = getPreviewType($r);
    $r->log_error(0, 'previewType=' . ($previewType || 'default'));

    my $url_path = "/gate/get/" . $mulcaId;
    my $mulca_sign = hmac_sha256_hex($url_path, getStorageSecret());

    my $encoded_url = uri_escape("http://" . $host . $url_path . "?sign=" . $mulca_sign);
    my $typemap = uri_escape($previewType ? "*:$previewType" : "gif:gif;png:png;*:jpg;", "/:;");

    (my $width, my $height) = parseSize(varSize($r), $crop);
    my @params = ("url=" . $encoded_url, "typemap=" . $typemap);

    push(@params, "watermark=" . $watermark) if $watermark ne '';
    push(@params, "width=" . $width) if $width ne '';
    push(@params, "height=" . $height) if $height ne '';
    push(@params, "quality=" . ($quality eq '' ? 85 : $quality));
    push(@params, "crop=yes") if $crop eq "1";
    #push(@params, "preserve_animation=yes");

    my $key= md5_hex(join('/', @params) . $secret);
    return join('&', @params) . "&key=" . $key;
}

sub getStorageHost {
    if (isProdOrPrestable()) {
        return 'storage.mail.yandex.net:8080';
    } else {
        return 'storage.stm.yandex.net:8080';
    }
}

sub isProdOrPrestable {
    return (($ENVIRONMENT_TYPE  =~ /production/) or ($ENVIRONMENT_TYPE =~ /prestable/));
}

sub parseSize {
    my ($size, $crop) = @_;
    my %previewSizes = (
        'XXXL', '1280x1280',
        'XXL',  '1024x1024',
        'XL',   '800x800',
        'L',    '500x500',
        'M',    '300x300',
        'S',    '150x150',
        'XS',   '100x100',
        'XXS',  '75x75',
        'XXXS', '50x50');

    my $normalized_size = uc($size);
    if (exists $previewSizes{$normalized_size}) {
        $size = $previewSizes{$normalized_size};
    }

    (my $width, my $height) = split('x', $size);
    if ($crop and ($crop eq "1") and ($width eq '' or $height eq '')) {
        $width = $height if $width eq '';
        $height = $width if $height eq '';
    }

    return ($width, $height);
}

sub varPreviewSize {
    my ($r) = @_;
    return join 'x', parseSize varSize($r), varCrop($r) if varSize($r);
}

sub getWatermarkType {
    my ($r) = @_;
    my $logo = parseGetParam($r, 'logo');
    if ($logo eq "1") {
        return 'disk';
    } elsif ($logo eq "2") {
        return 'disk-center';
    } else {
        return '';
    }
}

sub getShortHostName {
    my ($host) = @_;

    my $shortName = (split( '\.', $host))[0];
    if ($shortName =~ /downloader-default/) {
        $shortName = "downloader" . (split('downloader-default', $shortName))[1];
    }

    return $shortName;
}

sub getClientNameByUserAgent {
    my ($ua) = @_;

    if ($ua =~ /^Yandex.Disk [{(]"os":"([^"]*)".*/) {
        if ($1 =~ /^windows|win8|winrt/) {
            return 'win';
        } elsif ($1 =~ /^mac|macstore/) {
            return 'mac';
        } elsif ($1 =~ /^cli/) {
            return 'lnx';
        } elsif ($1 =~ /^android/) {
            return 'android';
        } elsif ($1 =~ /^iOS/) {
            return 'ios';
        } elsif ($1 =~ /^wp/) {
            return "wp";
        }
    }

    return 'na';
}

sub isWebpRequired {
    my ($r) = @_;
    my ($width) = parseSize(varSize($r), parseGetParam($r, 'crop'));
    return 0 unless isAutowebpEnabledForSize($width);
    return 0 unless isAutowebpEnabledForUid(parseGetParam($r, 'uid'));
    return 0 if parseGetParam($r, 'webp') eq 'false';
    return 1 if getHeader($r, 'Accept') =~ qw'image/webp';
    return 0;
}

# internal

return 1;
