use strict;
use warnings;
use Test::More 'no_plan';

use lib '.'; # for local nginx.pm
use_ok('t::NginxRequestMock');
use_ok('DownloaderHandlers');
use_ok('DownloaderVars');
use_ok('DownloaderSecrets');
use_ok('DownloaderCipher');

use utf8;
use LWP::Simple;

my $r = undef;

$r = new NginxRequestMock("", "size=XL&crop=0&logo=0");
is( YandexDisk::varSize($r) , "XL" );
is( YandexDisk::varCrop($r), "false" );
is( YandexDisk::varLogo($r), "none" );

$r = new NginxRequestMock("", "size=XL");
is( YandexDisk::varCrop($r), "false" );
is( YandexDisk::varLogo($r), "none");

$r = new NginxRequestMock("", "crop=1");
is( YandexDisk::varCrop($r), "true" );

$r = new NginxRequestMock("", "logo=1");
is( YandexDisk::varLogo($r), "north_west" );

$r = new NginxRequestMock("", "logo=2");
is( YandexDisk::varLogo($r), "center_new" );

$r = new NginxRequestMock("", "fsize=321");
is( YandexDisk::varFileSize($r), "321" );

$r = new NginxRequestMock("", "");
is( YandexDisk::varFileSize($r), "1000000" );

# mulcaId parameter
my $uri = "/rdisk/07206d78bc2f7d8977cdd4e9bb7bb90a/5149ef9e/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==?limit=0";
$r = new NginxRequestMock($uri, "");

is( YandexDisk::varMulcaId($r), "6658.yadisk:1260145.4209040196165925644914183409545" );

$r = new NginxRequestMock("", "quality=50");
is( YandexDisk::varQuality($r), "50" );

# public file
$uri = "/rdisk/07206d7/5149ef9e/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==";
$r = new NginxRequestMock($uri, "uid=0");
is( YandexDisk::varCheckAuth($r), "0");

# private file, v2
$uri = "/rdisk/07206d7/5149ef9e/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==";
$r = new NginxRequestMock($uri, "uid=123&tknv=v2");
is( YandexDisk::varCheckAuth($r), "0" );
is(YandexDisk::varMulcaId($r), '6658.yadisk:1260145.4209040196165925644914183409545');
is(YandexDisk::varPreviewSource($r), '');
is(YandexDisk::varPreviewSize($r), '');

# private infinite file, v2
$uri = "/preview/07206d7/inf/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==";
$r = new NginxRequestMock($uri, "uid=123&tknv=v2");
is( YandexDisk::varCheckAuth($r), "1" );
is(YandexDisk::varMulcaId($r), '6658.yadisk:1260145.4209040196165925644914183409545');
is(YandexDisk::varPreviewSource($r), '');
is(YandexDisk::varPreviewSize($r), '');

# private file, v2
$uri = "/rdisk/07206d7/asdf/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==";
$r = new NginxRequestMock($uri, "uid=123");
is( YandexDisk::varCheckAuth($r), "1" );

# share file with autologin, uid=1
$uri = "/share/07206d7/asdf/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==";
$r = new NginxRequestMock($uri, "uid=123&tknv=v2&al=1");
is( YandexDisk::varCheckAuth($r), "1" );

# share file with autologin, uid=0
$uri = "/share/07206d7/asdf/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==";
$r = new NginxRequestMock($uri, "uid=0&tknv=v2&al=1");
is( YandexDisk::varCheckAuth($r), "1" );

# check timestamp
$uri = "/rdisk/07206d7/5149ef9e/_OZl24llA2Jy7qnu32cmBc-P54JbwSrGD33QLUtwssw31mEA-ztRT1sr5VkMEiz07tEwf8x-ddgj9YxNS4XtkA==";
$r = new NginxRequestMock($uri, "");
is( YandexDisk::varTimestamp($r), "5149ef9e" );

##############################################
# Resizer urls tests
my $uri2 = "/rdisk/07206d7/asdf/5M-iSLYEfDMZ--tasDOlGcAmWPGfpRVeXkio8wcq0_fcgPsHoM-iaY_Kc8w5xlDh-dOlJzGh5tgpdPU2jVDFAA==";
my $baseParams =
    "url=http%3A%2F%2Fstorage.stm.yandex.net%3A8080%2Fgate%2Fget%2F1000005.yadisk%3A173337688.3983296384203508725332872709265%3Fsign%3Dd50a3b5142ce8c997be5de994d34f1f42ac845b358dc771a33b03a73a3afe893"
    . "&typemap=gif%3Agif%3Bpng%3Apng%3B*%3Ajpg%3B";

# check resizer base params
$r = new NginxRequestMock($uri2, "size=240x240");
is( YandexDisk::getResizerParams($r), $baseParams . "&width=240&height=240&quality=85&key=3c3daa967a5ac39dd017c1b1d51b1c9d" );

# check resizer quality and crop params
$r = new NginxRequestMock($uri2, "size=240x300&quality=50&crop=1");
is( YandexDisk::getResizerParams($r), $baseParams . "&width=240&height=300&quality=50&crop=yes&key=4cb3e3bc338730d6a958d47cf02fb27d" );

# check resizer with sizes XXL, XS, M & etc
$r = new NginxRequestMock($uri2, "size=XL");
is( YandexDisk::getResizerParams($r), $baseParams . "&width=800&height=800&quality=85&key=1b0cdb1928d76aba676b93830c8c9426" );

# check resizer with lowercase sizes
$r = new NginxRequestMock($uri2, "size=xl");
is( YandexDisk::getResizerParams($r), $baseParams . "&width=800&height=800&quality=85&key=1b0cdb1928d76aba676b93830c8c9426" );

# check resizer crop=1 and size=x200
$r = new NginxRequestMock($uri2, "size=x200&crop=1");
is( YandexDisk::getResizerParams($r), $baseParams . "&width=200&height=200&quality=85&crop=yes&key=29c762d9618b371f615875105fbb1c97" );

# check resizer size 200x
$r = new NginxRequestMock($uri2, "size=200x");
is( YandexDisk::getResizerParams($r), $baseParams . "&width=200&quality=85&key=3cbfb527c5d7de123d9996418d743718" );

# check resizer watermark
$r = new NginxRequestMock($uri2, "size=200x100&logo=1");
is( YandexDisk::getResizerParams($r), $baseParams . "&watermark=disk&width=200&height=100&quality=85&key=a113ac71243194e081b0aa41d0d3d432" );

# check resizer preview_type
$r = new NginxRequestMock($uri2, "size=200x100&preview_type=pjpeg");
my $expectedResult =
    "url=http%3A%2F%2Fstorage.stm.yandex.net%3A8080%2Fgate%2Fget%2F1000005.yadisk%3A173337688.3983296384203508725332872709265%3Fsign%3Dd50a3b5142ce8c997be5de994d34f1f42ac845b358dc771a33b03a73a3afe893"
    . "&typemap=*%3Apjpeg&width=200&height=100&quality=85&key=f9b4860f3b0339a2fbc2e13d220097d6";
is( YandexDisk::getResizerParams($r), $expectedResult );

#################################################
# check short hostname
is( YandexDisk::getShortHostName("downloader-default2g.disk.yandex.net"), "downloader2g");
is( YandexDisk::getShortHostName("akirakozov.disk.yandex.net"), "akirakozov");

################################################
# check client name by user agent
is( YandexDisk::getClientNameByUserAgent('Yandex.Disk {"os":"windows","vsn":"1.2.2.4524","id":"fake-fake-fake","dstr":"browser"}'), "win");
is( YandexDisk::getClientNameByUserAgent('Yandex.Disk {"os":"mac","vsn":"1.2.2.4524","id":"fake-fake-fake","dstr":"browser"}'), "mac");
is( YandexDisk::getClientNameByUserAgent('Yandex.Disk {"os":"cli","vsn":"1.2.2.4524","id":"fake-fake-fake","dstr":"browser"}'), "lnx");
is( YandexDisk::getClientNameByUserAgent('Yandex.Disk {"os":"android 4.2.2","vsn":"1.2.2.4524","id":"fake-fake-fake","dstr":"browser"}'), "android");
is( YandexDisk::getClientNameByUserAgent('Yandex.Disk {"os":"iOS","vsn":"1.2.2.4524","id":"fake-fake-fake","dstr":"browser"}'), "ios");
is( YandexDisk::getClientNameByUserAgent('Yandex.Disk {"os":"wp","vsn":"1.2.2.4524","id":"fake-fake-fake","dstr":"browser"}'), "wp");


################################################
# check file-disposition by user-agent

# Chrome user-agent
my %headers = ("User-agent", "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/52.0.2743.116 Safari/537.36");
$r = new NginxRequestMock("", "filename=test.apk", \%headers);
is(YandexDisk::varFilenameMimeParam($r), "filename*=UTF-8''test.apk");

# Chrome user-agent & Cyrillic
%headers = ("User-agent", "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/52.0.2743.116 Safari/537.36");
$r = new NginxRequestMock("", "filename=%D1%82%D0%B5%D1%81%D1%82%D0%BE%D0%B2%D1%8B%D0%B9%20%D1%84%D0%B0%D0%B9%D0%BB.apk", \%headers);
is(YandexDisk::varFilenameMimeParam($r), "filename*=UTF-8''%D1%82%D0%B5%D1%81%D1%82%D0%BE%D0%B2%D1%8B%D0%B9%20%D1%84%D0%B0%D0%B9%D0%BB.apk");

# Android user-agent
%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "filename=test.apk", \%headers);
is(YandexDisk::varFilenameMimeParam($r), 'filename="test.apk"');

# Android user-agent & Cyrillic
%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "filename=%D1%82%D0%B5%D1%81%D1%82%D0%BE%D0%B2%D1%8B%D0%B9%20%D1%84%D0%B0%D0%B9%D0%BB%20%D1%81%20%D0%B1%D1%83%D0%BA%D0%B2%D0%BE%D0%B9%20%D1%85.apk", \%headers);
is(YandexDisk::varFilenameMimeParam($r), "filename=\"тестовый файл с буквой х.apk\"");

# Android and firefox user-agent & Cyrillic
%headers = ("User-agent", "Mozilla/5.0 (Android 4.4.2; Mobile; rv:48.0) Gecko/48.0 Firefox/48.0");
$r = new NginxRequestMock("", "filename=%D1%82%D0%B5%D1%81%D1%82%D0%BE%D0%B2%D1%8B%D0%B9%20%D1%84%D0%B0%D0%B9%D0%BB.apk", \%headers);
is(YandexDisk::varFilenameMimeParam($r), "filename*=UTF-8''%D1%82%D0%B5%D1%81%D1%82%D0%BE%D0%B2%D1%8B%D0%B9%20%D1%84%D0%B0%D0%B9%D0%BB.apk");

# Android user-agent and endline symbols in filename
%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "filename=%0atest.apk%0a", \%headers);
is(YandexDisk::varFilenameMimeParam($r), 'filename="test.apk"');

is(YandexDisk::isAutowebpEnabledForSize(), 0);
is(YandexDisk::isAutowebpEnabledForSize(0), 0);
is(YandexDisk::isAutowebpEnabledForSize(''), 0);
is(YandexDisk::isAutowebpEnabledForSize(50), 0);
is(YandexDisk::isAutowebpEnabledForSize(100), 1);
is(YandexDisk::isAutowebpEnabledForSize(150), 1);
is(YandexDisk::isAutowebpEnabledForSize(200), 1);
is(YandexDisk::isAutowebpEnabledForSize(250), 0);
is(YandexDisk::isAutowebpEnabledForSize(300), 1);
is(YandexDisk::isAutowebpEnabledForSize(350), 1);
is(YandexDisk::isAutowebpEnabledForSize(400), 1);
is(YandexDisk::isAutowebpEnabledForSize(450), 0);

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "filename=%0atest.apk%0a", \%headers);
is(YandexDisk::isWebpRequired($r), 0);

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=L", \%headers);
is(YandexDisk::isWebpRequired($r), 0);

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=M", \%headers);
#is(YandexDisk::isWebpRequired($r), 1);

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=320x240", \%headers);
#is(YandexDisk::isWebpRequired($r), 1);

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=640x480", \%headers);
is(YandexDisk::isWebpRequired($r), 0);

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=M&webp=false", \%headers);
is(YandexDisk::isWebpRequired($r), 0);

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.4.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=L&webp=true", \%headers);
is(YandexDisk::isWebpRequired($r), 0);
is(YandexDisk::varPreviewSize($r), '500x500');

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.2.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=M", \%headers);
#is(YandexDisk::isWebpRequired($r), '');

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.2.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=x88", \%headers);
is(YandexDisk::getAvatarsThumbnail($r), "%7B%22command%22%3A%22resize%22%2C%20%22height%22%3A%2288%22%2C%20%22quality%22%3A85%2C%20%22width%22%3A%22inf%22%7D");

%headers = (
    "User-agent" => "Mozilla/5.0 (Linux; Android 4.2.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36",
    "Accept" => "image/webp",
);
$r = new NginxRequestMock("", "size=M&content_type=xxx%2Fyyy", \%headers);
is(YandexDisk::isWebpRequired($r), 1);
is(YandexDisk::varEncodedContentType($r), 'image%2Fwebp');

%headers = (
    "User-agent" => "Mozilla/5.0 (Linux; Android 4.2.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36",
    "Accept" => "image/webp",
);
$r = new NginxRequestMock("", "size=L&content_type=xxx%2Fyyy", \%headers);
is(YandexDisk::isWebpRequired($r), 0);
is(YandexDisk::varEncodedContentType($r), 'xxx%2Fyyy');

is(YandexDisk::signAvatarsUrl('deritest', '702', '2a0000014f55c205bb16ac6ebc4331f77032', '%7B%22width%22%3A%20%22300%22%2C%20%22height%22%3A%20%22300%22%2C%20%22command%22%3A%20%22resize%22%7D', '123'), 'd21a13e12e9af558412f682f39d29b5dd1a0fee74d3d92db285df87833a28e60');

my $thumb = YandexDisk::getAvatarsThumbnail($r);
my $sign = YandexDisk::signAvatarsUrl('disk', '69197', 'ufo', $thumb, YandexDisk::getAvatarsSecret());
is(get("http://avatars-int.mdst.yandex.net:13000/genurlsign-disk/69197/ufo?thumbnail=$thumb"), "{\n    \"path\": \"/get-disk/69197/ufo/?thumbnail=$thumb&sign=$sign\",\n    \"sign\": \"$sign\"\n}");

%headers = ("User-agent", "Mozilla/5.0 (Linux; Android 4.2.2; ru-ru; SAMSUNG SM-N900 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Version/1.5 Chrome/28.0.1500.94 Mobile Safari/537.36");
$r = new NginxRequestMock("", "size=M", \%headers);

my $url = "http://avatars.mdst.yandex.net/" . YandexDisk::getAvatarsParams($r, 'disk', '69197', 'ufo');
#is($url, 'http://avatars.mdst.yandex.net/get-disk/69197/ufo?thumbnail=%7B%22command%22%3A%22resize%22%2C%20%22height%22%3A%22300%22%2C%20%22quality%22%3A85%2C%20%22typemap%22%3A%22gif%3Agif%3Bpng%3Apng%3B*%3Ajpg%3B%22%2C%20%22width%22%3A%22300%22%7D&sign=365ea1114b4ecfad664b66d161aeaf5facc61e4f6111e1079b189b12823c905d');
#is(defined(get($url)), 1);