package ru.yandex.chemodan.app.eventloader;

import java.net.URI;
import java.net.URISyntaxException;

import org.apache.http.client.HttpClient;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.eventloader.utils.GenerateExampleJsonMain;
import ru.yandex.chemodan.app.eventloader.utils.TskvEventLogReader;
import ru.yandex.chemodan.disksearch.indexing.EventIndexer;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClient4InputStreamSource;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class EventIndexerImplTest {
    private static final Logger logger = LoggerFactory.getLogger(EventIndexerImplTest.class);

    private static final BenderMapper mapper = new BenderMapper();

    private final MultiSearchChecker checker = new MultiSearchChecker(
            new SearchChecker("searchdb-tst01h.disk.yandex.net"),
            new SearchChecker("searchdb-tst02h.disk.yandex.net")
    );

    private final EventIndexer indexer = new EventIndexerContextConfiguration().indexer(
            parseUri("http://disk-producer-test.n.yandex-team.ru/add?service=ipdd"),
            ApacheHttpClientUtils.singleConnectionClient(Timeout.seconds(20), Option.of("chemodan.test")));

    private URI parseUri(String value) {
        try {
            return new URI(value);
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private final TskvEventLogReader logReader = new TskvEventLogReader();

    @Ignore
    @Test
    public void test() {
        logReader.readAll(new ClassPathResourceInputStreamSource(GenerateExampleJsonMain.EXAMPLE_LOG_PATH), event -> {
            logger.info(event);

            EventLoaderLogListener.indexRequestBuilder
                    .buildUnsafe(event.getEvents())
                    .forEach(indexer::index);
            Assert.isTrue(isIndexed(event), "event is not indexed");
        });

    }

    public boolean isIndexed(AbstractEvent event) {
        for(int i = 0; i < 10; i++) {
            Hits hits = checker.search(event);
            if (hits.notEmpty()) {
                return true;
            }

            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        return false;
    }

    static class MultiSearchChecker {
        private final ListF<SearchChecker> checkers;

        MultiSearchChecker(SearchChecker... checkers) {
            this.checkers = Cf.list(checkers);
        }

        Hits search(AbstractEvent event) {
            Hits result = new Hits();
            for(SearchChecker checker : checkers) {
                result = result.plus(checker.search(event));
            }
            return result;
        }
    }

    static class SearchChecker {
        private final HttpClient httpClient = ApacheHttpClientUtils.Builder.create()
                .multiThreaded()
                .withMaxConnections(20)
                .withTimeout(Timeout.seconds(200))
                .withUserAgent("test")
                .build();

        private final URI uri;

        SearchChecker(String host) {
            try {
                this.uri = new URI("http://" + host + ":8088/search?length=0");
            } catch (URISyntaxException e) {
                throw new RuntimeException(e);
            }
        }

        Hits search(AbstractEvent event) {
            URI eventUri = UriBuilder.cons(uri)
                    .addParam("prefix", event.getUid())
                    .addParam("text", "id:" + event.keys.getId())
                    .build();

            String hitJson = new ApacheHttpClient4InputStreamSource(httpClient, eventUri).readText();
            return mapper.parseJson(Hits.class, hitJson);
        }
    }

    @Ignore
    @Test
    public void testHitsBending() {
        Hits hits = mapper.parseJson(Hits.class,
                "{\"hitsCount\": 1, \"hitsArray\": [{\"version\": \"1441324799999\"}]}");
        Assert.equals(new Hits(1, Cf.list(Cf.map("version", "1441324799999"))), hits);
    }

    @Ignore
    @Test
    public void testRequestBending() {
        Request expected = new Request(new PassportUid(1), Cf.list(Cf.map("version", "1441324799999")));
        Request actual = mapper.parseJson(Request.class,
                "{\"prefix\": 1, \"docs\": [{\"version\": \"1441324799999\"}]}");
        Assert.equals(expected, actual);
    }

    @Bendable
    static class Hits extends DefaultObject {
        @BenderPart(name = "hitsCount")
        public int count;

        @BenderPart(name = "hitsArray")
        public ListF<MapF<String, String>> hits;

        Hits() {
            this(0, Cf.list());
        }

        Hits(int count, ListF<MapF<String, String>> hits) {
            this.count = count;
            this.hits = hits;
        }

        Hits plus(Hits another) {
            int count = this.count + another.count;
            ListF<MapF<String, String>> hits = this.hits.plus(another.hits);
            return new Hits(count, hits);
        }

        boolean notEmpty() {
            return count > 0;
        }
    }

    @Bendable
    static class Request extends DefaultObject {
        @BenderPart(name = "prefix")
        public PassportUid uid;

        @BenderPart(name = "docs")
        public ListF<MapF<String, String>> events;

        Request(PassportUid uid, ListF<MapF<String, String>> events) {
            this.uid = uid;
            this.events = events;
        }
    }
}
