package ru.yandex.chemodan.app.eventloader;

import java.math.BigInteger;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.eventloader.serializer.BigIntegerMarshaller;
import ru.yandex.chemodan.app.eventloader.serializer.IntEnumMarshaller;
import ru.yandex.chemodan.app.eventloader.serializer.MediaTypeMarshaller;
import ru.yandex.chemodan.app.eventloader.serializer.MpfsUidMarshaller;
import ru.yandex.chemodan.app.eventloader.serializer.MpfsUidUnmarshaller;
import ru.yandex.chemodan.disksearch.indexing.EventIndexer;
import ru.yandex.chemodan.disksearch.indexing.IndexRequest;
import ru.yandex.chemodan.disksearch.indexing.IndexRequestBuilder;
import ru.yandex.chemodan.eventlog.EventLogListenerSupport;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.MediaType;
import ru.yandex.chemodan.eventlog.events.comment.AbstractCommentEvent;
import ru.yandex.chemodan.eventlog.events.sharing.ShareRights;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.InstantAsMillisMarshaller;
import ru.yandex.misc.bender.custom.InstantAsMillisUnmarshaller;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;
import ru.yandex.misc.monica.core.blocks.Meter;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class EventLoaderLogListener extends EventLogListenerSupport {
    private static final Logger logger = LoggerFactory.getLogger(EventLoaderLogListener.class);

    public static final IndexRequestBuilder<AbstractEvent> indexRequestBuilder =
            new IndexRequestBuilder<>(
                    new BenderMapper(new BenderConfiguration(
                            BenderConfiguration.defaultSettings(),
                            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                                    .add(Instant.class,
                                            new InstantAsMillisMarshaller(),
                                            new InstantAsMillisUnmarshaller())
                                    .add(MpfsUid.class, new MpfsUidMarshaller(), new MpfsUidUnmarshaller())
                                    .add(ShareRights.class, new IntEnumMarshaller())
                                    .add(MediaType.class, new MediaTypeMarshaller())
                                    .add(BigInteger.class, new BigIntegerMarshaller())
                                    .build()
                    )),
                    AbstractEvent::getPassportUid);

    @MonicaMetric(description = "Indexed events")
    @GroupByDefault
    static final MeterMap indexedEvents = new MeterMap();

    @MonicaMetric(description = "Event indexing errors")
    @GroupByDefault
    static final Meter indexErrors = new Meter();

    @MonicaMetric(description = "Mpfs request errors")
    @GroupByDefault
    static final Meter mpfsErrors = new Meter();

    private static MetricGroupName metricGroupName() {
        return new MetricGroupName(
                "event-loader",
                new MetricName("event-loader", "log-processing"),
                "Event history log processing");
    }

    static {
        MonicaStaticRegistry.register(EventLoaderLogListener.class, metricGroupName());
    }
    public static final DynamicProperty<Boolean> disableMpfsRequests =
            DynamicProperty.cons("eventHistoryDisableMpfsRequests", false);

    private final EventIndexer indexer;

    private final MpfsClient mpfsClient;

    private final int retryCount;

    public EventLoaderLogListener(EventIndexer indexer, MpfsClient mpfsClient, int retryCount) {
        this.indexer = indexer;
        this.mpfsClient = mpfsClient;
        this.retryCount = retryCount;
    }

    @Override
    public void processEvent(AbstractEvent event) {
        if (event instanceof AbstractCommentEvent && !event.reject()) {
            try {
                event = obtainCommentMetadata((AbstractCommentEvent) event);
            } catch(IllegalStateException e) {
                logger.warn("Unable to obtain required comment info");
                return;
            }
        }

        for(AbstractEvent rejectedEvent : event.getRejectedEvents()) {
            incSkippedEvents(rejectedEvent.metadata.tskvEventType);
        }

        for(IndexRequest request : indexRequestBuilder.build(event.getAcceptedEvents())) {
            if (request.isNotIndexable()) {
                continue;
            }

            if (event.metadata.cloudRequestId.isPresent()) {
                request = request.withQueryParam("ycrid", event.metadata.cloudRequestId.get().toString());
            }

            indexWithRetries(request, event.metadata.tskvEventType.value());
        }
    }

    @Override
    public Logger loggerForMonitoring() {
        return logger;
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return metricGroupName();
    }

    private AbstractEvent obtainCommentMetadata(AbstractCommentEvent event) {
        event = obtainCommentEventResourceName(event);
        event = propagateCommentEventForShareUsers(event);
        if (!event.entity.resourceName.isPresent() || !event.entity.resourceType.isPresent()) {
            throw new IllegalStateException("Unable to get required comment info");
        }
        return event;
    }

    void indexWithRetries(IndexRequest request, String eventTypeStr) {
        try {
            new RetryManager().withRetryPolicy(retryCount, 1).run(() -> indexer.index(request));
            indexedEvents.inc(eventTypeStr);
            indexedEvents.inc();
        } catch (RuntimeException e) {
            indexErrors.inc();
            logger.error(
                    "Event is not saved to index: event#{}",
                    request.documents.map(d -> d.getO("id").get()).toString()
            );
            throw e;
        }
    }

    private AbstractCommentEvent obtainCommentEventResourceName(AbstractCommentEvent event) {
        return supplementCommentEventByMpfsWithRetries(
                event, "resource name",
                resId -> mpfsClient.getFileInfoOByFileId(MpfsUser.of(resId.owner.getUid()), resId.fileId)
                        .<AbstractCommentEvent>map(event::withFileInfo));
    }

    private AbstractCommentEvent propagateCommentEventForShareUsers(AbstractCommentEvent event) {
        return supplementCommentEventByMpfsWithRetries(
                event, "resource share users",
                resId -> mpfsClient.getShareUidsInGroupO(MpfsUser.of(resId.owner), resId.fileId)
                        .map(users -> event.withExtraRecipients(users.getUids())));
    }

    private <T extends AbstractCommentEvent> T supplementCommentEventByMpfsWithRetries(
            T event, String what, Function<MpfsResourceId, Option<T>> supplier)
    {
        MpfsResourceId res = MpfsResourceId.parse(event.entity.id);
        if (res.owner.isSpecial() || disableMpfsRequests.get()) {
            return event;
        }

        try {
            return RetryUtils.retry(retryCount, () -> supplier.apply(res).getOrElse(event));

        } catch (RuntimeException e) {
            mpfsErrors.inc();
            logger.error(
                    "Failed to obtain {} for comment event#{}: {}",
                    what, event.keys.getId(), ExceptionUtils.getAllMessages(e)
            );
            throw e;
        }
    }
}
