package ru.yandex.chemodan.app.eventloader;

import org.joda.time.Instant;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.disksearch.indexing.EventIndexer;
import ru.yandex.chemodan.disksearch.indexing.IndexRequest;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.EventType;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsGroupUids;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@RunWith(MockitoJUnitRunner.class)
public class EventLoaderLogListenerTest {

    @Mock
    EventIndexer indexerMock;
    @Mock
    MpfsClient mpfsClient;

    @Test
    public void eventsIsRejected() {
        new EventLoaderLogListener(indexerMock, mpfsClient, 1)
                .processEvent(new AbstractEvent(new EventMetadata(new MpfsUid(1L), Instant.now(), Option.empty())) {
                    @Override
                    public EventType getEventType() {
                        return EventType.FS_TRASH_DROP_ALL;
                    }

                    @Override
                    public ListF<Object> getGroupChunks() {
                        return Cf.list("");
                    }

                    @Override
                    public ListF<Object> getNameChunks() {
                        return Cf.list("");
                    }

                    @Override
                    protected boolean rejectInternal() {
                        return true;
                    }
                });

        verify(indexerMock, never())
                .index(any());
    }

    @Test
    public void commentEventPropagation() {
        Mockito.reset(indexerMock);
        MpfsGroupUids groupUids = new MpfsGroupUids(Cf.list(158426306, 97963799, 127117)
                .map(uid -> new MpfsGroupUids.User(PassportUid.cons(uid), MpfsGroupUids.Status.APPROVED)));

        MpfsFileInfo file = MpfsFileInfo.parse("{\"name\":\"name\",\"type\":\"dir\",\"meta\":{}}");

        Mockito.when(mpfsClient.getShareUidsInGroupO(Mockito.any(), Mockito.any())).thenReturn(Option.of(groupUids));
        Mockito.when(mpfsClient.getFileInfoOByFileId(Mockito.any(), Mockito.any())).thenReturn(Option.of(file));

        String line = "tskv\tunixtime=0\t"
                + "event_type=comment-add\tuser_uid=1001197903\t"
                + "entity_type=public_resource\tentity_id=" + groupUids.getUids().first() + ":file_id\t"
                + "comment_id=127\tcomment_author_uid=1001197903";

        ArgumentCaptor<IndexRequest> eventCaptor = ArgumentCaptor.forClass(
                ClassX.wrap(IndexRequest.class).<IndexRequest>uncheckedCast().getClazz()
        );

        new EventLoaderLogListener(indexerMock, mpfsClient, 0)
                .processLogLine(line);

        verify(indexerMock, times(3))
                .index(eventCaptor.capture());

        Assert.forAll(eventCaptor.getAllValues(), req -> req.documents.size() == 1);
    }
}
