package ru.yandex.chemodan.app.fotki;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import org.apache.http.Header;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.fotki.dao.AlbumDao;
import ru.yandex.chemodan.app.fotki.dao.ImageDao;
import ru.yandex.chemodan.app.fotki.dao.model.Album;
import ru.yandex.chemodan.app.fotki.dao.model.Image;
import ru.yandex.chemodan.app.fotki.dao.model.ImageFileType;
import ru.yandex.chemodan.app.fotki.dao.model.ImageFlag;
import ru.yandex.chemodan.app.fotki.dao.model.ImageSize;
import ru.yandex.chemodan.app.fotki.utils.NameUtils;
import ru.yandex.chemodan.app.fotki.utils.Path;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;

public class PhotoProxyManager {

    private static final Logger logger = LoggerFactory.getLogger(PhotoProxyManager.class);

    private static final SetF<String> REQUIRED_META_FIELDS = Cf.set("fotki_image_id", "file_mid", "pmid");
    public static final Path FOTKI_AT_DISK_PATH = Path.get("/attach/YaFotki");
    public static final int INF_EXPIRE_SECONDS = 600;
    public static final String EDORIG_SUFFIX = "_edorig";

    private final ImageDao imageDao;
    private final AlbumDao albumDao;
    private final AlbumManager albumManager;
    private final MpfsClient mpfsClient;
    private final ExecutorService executors;
    private final HttpClient downloaderHttpClient;
    private final Blackbox2 blackbox;

    public PhotoProxyManager(ImageDao imageDao, AlbumDao albumDao,
            AlbumManager albumManager, MpfsClient mpfsClient,
            HttpClient downloaderHttpClient, Blackbox2 blackbox, int saveMpfsPathThreadsCount)
    {
        this.imageDao = imageDao;
        this.albumDao = albumDao;
        this.albumManager = albumManager;
        this.mpfsClient = mpfsClient;
        this.executors = Executors.newFixedThreadPool(saveMpfsPathThreadsCount);
        this.downloaderHttpClient = downloaderHttpClient;
        this.blackbox = blackbox;
    }

    public String getImage(Image imageParsed) {
        final Option<Image> imageO = imageDao.find(imageParsed);
        if (!imageO.isPresent()) {
            logger.error("Could not find information about image with id={}, dirName={}", imageParsed.getImageId(),
                    imageParsed.getDirName());
            throw new PermanentHttpFailureException("Image not found", HttpStatus.SC_404_NOT_FOUND);
        }
        final Image image = imageO.get();
        logger.info("Found information about image: {}", image);
        if (isImageBlocked(image)) {
            throw new PermanentHttpFailureException("Image blocked", HttpStatus.SC_403_FORBIDDEN);
        }
        final MpfsFileInfo fileInfo = getMpfsFileInfoFromSavedPath(image).orElse(() -> getMpfsFileInfo(image))
                .getOrThrow(
                        () -> new PermanentHttpFailureException("MPFS path not found", HttpStatus.SC_404_NOT_FOUND));
        final ImageSize size = imageParsed.getSize();
        Option<String> sizeString =
                Option.when(!ImageSize.isOriginal(size), () -> size.getWidth() + "x" + size.getHeight());
        final ImageFormat originalImageFormat = image.getOriginalImageFormat();
        Option<String> contentType =
                Option.when(originalImageFormat != null, () -> originalImageFormat.getContentType())
                        .orElse(Option.of("image/jpeg"));
        return getDownloaderUrl(fileInfo, sizeString, contentType);
    }

    public String getDownloaderUrl(MpfsFileInfo fileInfo, Option<String> sizeString, Option<String> contentType) {
        final boolean isPreview = sizeString.isPresent();
        final String fileStid = fileInfo.getMeta().getFileMid().get();
        final String previewStid = fileInfo.getMeta().getPmid().getOrElse(fileStid);
        final String stid = isPreview ? previewStid : fileStid;
        final String fileName = fileInfo.name.get();
        String downloaderUrl = mpfsClient.generateZaberunUrl(
                stid,
                fileName,
                isPreview ? "preview" : "file",
                Option.empty(),
                Option.empty(),
                contentType,
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                sizeString,
                Option.empty(),
                Option.empty(),
                Option.of(INF_EXPIRE_SECONDS),
                Option.of(0),
                Option.of(true),
                Option.empty());
        if (!isPreview) {
            downloaderUrl = getRedirectLocation(downloaderUrl);
        }
        logger.info("Downloader url: {} for stid {}", downloaderUrl, stid);
        return downloaderUrl.replace("https://", "/downloader/");
    }

    private String getRedirectLocation(String downloaderUrl) {
        final HttpGet request = new HttpGet(downloaderUrl);
        Option<String> locationO = Option.empty();
        try {
            locationO = ApacheHttpClientUtils.execute(request, downloaderHttpClient, (response) -> {
                final Header location = response.getFirstHeader("Location");
                return location == null ? Option.empty() : Option.of(location.getValue());
            });
        } catch (RuntimeIoException e) {
            logger.info("Failed to connect to downloader to retrieve redirect URL: {}", e.getMessage());
        }
        return locationO.getOrElse(downloaderUrl);
    }

    private void saveMpfsPath(Image image, String mpfsPath) {
        executors.submit(() -> imageDao
                .saveMpfsPath(!mpfsPath.isEmpty(), mpfsPath, image.getImageId(), image.getDirName()));
    }

    private Option<MpfsFileInfo> getMpfsFileInfo(Image image) {
        final MpfsUser uid = MpfsUser.of(image.getAuthorId());
        final int imageId = image.getImageId();
        ListF<String> paths = getAllPaths(image);
        Option<MpfsFileInfo> fileInfoO = getFileInfoWithCorrectImageO(uid, paths, imageId);
        //in case we have not found path, we'll retry up to 10 times appending imageId to Mpfs path
        int repeatCount = 0;
        while (repeatCount++ < 10 && !fileInfoO.isPresent()) {
            paths = paths.map(path -> NameUtils.removeExtension(path) + "_" + imageId + getExtensionSuffix(image));
            fileInfoO = getFileInfoWithCorrectImageO(uid, paths, imageId);
        }
        if (!fileInfoO.isPresent() && isImageEdited(image)) {
            return fallbackForEditedImages(
                    image); //in case there is no edited photo in mpfs we return original instead of 404
        }
        if (fileInfoO.isPresent()) {
            final String mpfsPath = fileInfoO.get().getPath().get();
            logger.info("Found MPFS path: {}", mpfsPath);
            saveMpfsPath(image, mpfsPath);
        } else {
            logger.error("MPFS path not found. Last try was for {}", paths.last());
        }
        return fileInfoO;
    }

    private Option<MpfsFileInfo> fallbackForEditedImages(Image image) {
        logger.info("Fallback for edited image. Trying to find unedited without suffix '_edorig'");
        unsetImageEditedFlag(image);
        return getMpfsFileInfo(image);
    }

    private void unsetImageEditedFlag(Image image) {
        image.setFlags(image.getFlags().unset(ImageFlag.HAS_EDITED_ORIGINAL));
    }

    @Nonnull
    private Option<MpfsFileInfo> getFileInfoWithCorrectImageO(MpfsUser uid, ListF<String> paths, int imageId) {
        final ListF<MpfsFileInfo> fileInfos = mpfsClient.bulkInfoByPaths(uid, REQUIRED_META_FIELDS, paths);
        final IteratorF<MpfsFileInfo> iterator = fileInfos.iterator();
        Option<MpfsFileInfo> fileInfoO = iterator.nextO();
        while (fileInfoO.isPresent() && !isCorrectImageFound(fileInfoO.get(), imageId)) {
            fileInfoO = iterator.nextO();
        }
        return fileInfoO;
    }

    private Option<MpfsFileInfo> getMpfsFileInfoFromSavedPath(Image image) {
        if (image.isMpfsPathFound()) {
            final MpfsUser uid = MpfsUser.of(image.getAuthorId());
            final int imageId = image.getImageId();
            final Option<MpfsFileInfo> fileInfoO =
                    mpfsClient.getFileInfoOByUidAndPath(uid, image.getMpfsPath(), REQUIRED_META_FIELDS.toList()); //check if the path is still valid
            if (fileInfoO.isPresent() && isCorrectImageFound(fileInfoO.get(), imageId)
                    && isCorrectFileForEditedImage(image, fileInfoO))
            {
                return fileInfoO;
            } else {
                saveMpfsPath(image, ""); //if path not valid, update with empty path
            }
        }
        return Option.empty();
    }

    private boolean isCorrectImageFound(MpfsFileInfo fileInfo, int imageId) {
        return fileInfo.getMeta().getFotkiImageId().isSome(imageId);
    }

    private boolean isCorrectFileForEditedImage(Image image, Option<MpfsFileInfo> fileInfoO) {
        return !(isImageEdited(image) && !isFileEdited(fileInfoO.get()));
    }

    private boolean isImageEdited(Image image) {
        return image.getFlags().isSet(ImageFlag.HAS_EDITED_ORIGINAL);
    }

    private boolean isFileEdited(MpfsFileInfo fileInfo) {
        return fileInfo.name.get().contains(EDORIG_SUFFIX);
    }

    private ListF<String> getAllPaths(Image image) {
        return Stream
                .of(getImagePath(image, false, true),
                        getImagePath(image, false, false),
                        getImagePath(image, true, true),
                        getImagePath(image, true, false))
                .map(Path::toString)
                .collect(CollectorsF.toList());
    }

    private Path getImagePath(Image image, boolean addAlbumIdSuffix, boolean addImageIdSuffix) {
        String albumPath = StringUtils.join(
                getAlbumsStartingFrom(image.getAuthorId(), image.getAlbumId())
                        .reverseIterator()
                        .map(NameUtils::extractDirName),
                "/"
        );
        String albumIdSuffix = addAlbumIdSuffix ? "_" + image.getAlbumId() : "";
        final Path path = FOTKI_AT_DISK_PATH.resolve(albumPath + albumIdSuffix)
                .resolve(getFilenameAtDiskAlbum(image, addImageIdSuffix));
        return path;
    }

    private ListF<Album> getAlbumsStartingFrom(PassportUid authorId, int albumId) {
        return albumManager.findPathToRootFrom(authorId, albumId);
    }

    private String getFilenameAtDiskAlbum(Image image, boolean addIdSuffix) {
        String fileTypeSuffix = getFileTypeForDiskAlbum(image)
                .getSuffixO()
                .getOrElse("");
        String idSuffix = addIdSuffix ? "_" + image.getImageId() : "";
        String edorigSuffix = isImageEdited(image) ? EDORIG_SUFFIX : "";
        return NameUtils.extractFilename(image) + fileTypeSuffix + idSuffix + edorigSuffix + getExtensionSuffix(image);
    }

    @Nonnull
    private String getExtensionSuffix(Image image) {
        return StringUtils.notBlankO(image.getExtension())
                .map(ext -> "." + ext)
                .getOrElse("");
    }

    private ImageFileType getFileTypeForDiskAlbum(Image image) {
        if (image.getFlags().isSet(ImageFlag.YA_DISK_FORCED_EDORIG)) {
            return ImageFileType.EDORIG;
        } else if (image.getFlags().isSet(ImageFlag.YA_DISK_PREVIEW_MIGRATED)) {
            return ImageFileType.PREVIEW;
        } else {
            return ImageFileType.ORIG;
        }
    }

    private boolean isImageBlocked(Image image) {
        return image.getFlags().isSet(ImageFlag.BLOCKED);
    }

    public String getAlbum(Image image) {
        final String login = image.getLogin();
        final int parsedAlbumId = image.getAlbumId();
        final Option<PassportUid> uidO =
                blackbox.query().userInfo(LocalhostUtils.localAddress(), login, Cf.list()).getUid();
        if (!uidO.isPresent()) {
            logger.error("Could not get uid for login: {}", login);
            throw new PermanentHttpFailureException("User not found", HttpStatus.SC_404_NOT_FOUND);
        } else {
            PassportUid uid = uidO.get();
            if (image.getImageId() != 0) {
                final Option<Image> imageO = imageDao.find(uid, image.getImageId());
                if (!imageO.isPresent()) {
                    logger.info("Could not find information about album item with imageId={}, albumId={}, uid={}",
                            image.getImageId(), image.getAlbumId(), uid);
                } else {
                    image =
                            imageO.get(); //here we fetch album id from db which overwrites the one we obtained from request
                    Option<String> itemUrlO = getAlbumItemUrl(image, uid);
                    if (!itemUrlO.isPresent()) {
                        logger.info("Could not get URL for album item with imageId={}, albumId={}, uid={}",
                                image.getImageId(), image.getAlbumId(), uid);
                    } else {
                        return itemUrlO.get();
                    }
                }
            }
            //in case we can't get album item URL we'll try to get album URL
            final Option<Album> albumO = albumDao.findAlbumById(uid, image.getAlbumId())
                    .orElse(albumDao.findAlbumById(uid, parsedAlbumId)); //another try with album id we get from request
            if (!albumO.isPresent()) {
                logger.info("Could not find information about album with id={} and also with id={}, uid={}",
                        image.getAlbumId(), parsedAlbumId, uid);
            } else {
                final Option<String> albumUrlO = getAlbumUrl(uid, albumO.get().getAlbumId());
                if (albumUrlO.isPresent()) {
                    return albumUrlO.get();
                }
            }
        }
        logger.error("Could not get URL for album with id={} and also with id={}, uid={}",
                image.getAlbumId(), parsedAlbumId, uidO.get());
        throw new PermanentHttpFailureException("Album not found", HttpStatus.SC_404_NOT_FOUND);
    }

    @Nonnull
    private Option<String> getAlbumItemUrl(Image image, PassportUid userId) {
        final Option<MpfsFileInfo> fileInfoO = getMpfsFileInfoFromSavedPath(image).orElse(() -> getMpfsFileInfo(image));
        if (fileInfoO.isPresent()) {
            return mpfsClient.getFotkiAlbumItemUrl(userId, image.getAlbumId(), fileInfoO.get().path.get());
        }
        return Option.empty();
    }

    @Nonnull
    private Option<String> getAlbumUrl(PassportUid userId, int albumId) {
        return mpfsClient.getFotkiAlbumUrl(userId, albumId);
    }
}
