package ru.yandex.chemodan.app.fotki;

import javax.annotation.Nonnull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.fotki.dao.model.Image;
import ru.yandex.chemodan.app.fotki.dao.model.ImageSize;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;

public class UrlParser {

    private static final Logger logger = LoggerFactory.getLogger(UrlParser.class);

    private static final Pattern2 URL_PATTERN = Pattern2.compile("(?:(?:https?://[^/]+)?(?:/?get))?(.+)");

    public static Image parseUrlLenient(String url) {
        Matcher2 matcher = URL_PATTERN.matcher2(url);
        matcher.find();
        return parseUrl(matcher.group(1).getOrElse(url));
    }

    public static Image parseUrl(String pathInfo) {

        final ListF<String> path = validateAndSplitPath(pathInfo);
        if (path.size() < 2) {
            throw new PermanentHttpFailureException("Invalid path", HttpStatus.SC_400_BAD_REQUEST);
        }

        final IteratorF<String> pathIterator = path.iterator();
        final Image.ImageBuilder imageBuilder = Image.builder();

        if (path.size() > 2) { //sometimes this part is absent
            final Option<String> storageId = pathIterator.nextO(); //unused
        }
        final boolean dirNameParsed = parseDirName(pathIterator, imageBuilder);
        final boolean plainFileNameParsed = parsePlainFileName(pathIterator, imageBuilder);

        if (dirNameParsed && plainFileNameParsed) {
            final Image image = imageBuilder.build();
            logger.info("Image URL parsed: {}", image);
            return image;
        } else {
            logger.error("Could not parse image URL: {}", path);
            throw new PermanentHttpFailureException("Path could not be parsed", HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    public static Image parseAlbumUrl(String pathInfo) {

        final ListF<String> path = validateAndSplitPath(pathInfo);
        if (path.size() < 2) {
            throw new PermanentHttpFailureException("Invalid path", HttpStatus.SC_400_BAD_REQUEST);
        }
        final IteratorF<String> pathIterator = path.iterator();
        final Image.ImageBuilder imageBuilder = Image.builder();
        final boolean userParsed = parseUser(pathIterator, imageBuilder);
        Option<String> keyWord = pathIterator.nextO(); //link might contain "album", "view" or both: /album/12345/view/67890
        boolean mainPartParsed = false;
        if (keyWord.isSome("album")) {
            mainPartParsed = parseAlbum(pathIterator, imageBuilder);
            keyWord = pathIterator.nextO();
        }
        if (keyWord.isSome("view")) {
            mainPartParsed = parseView(pathIterator, imageBuilder);
        }
        if (userParsed && mainPartParsed) {
            final Image image = imageBuilder.build();
            logger.info("Next URL parsed: {}", image);
            return image;
        } else {
            logger.error("Could not parse next URL: {}", path);
            throw new PermanentHttpFailureException("Path could not be parsed", HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    @Nonnull
    private static ListF<String> validateAndSplitPath(String pathInfo) {
        if (!pathInfo.startsWith("/")) {
            return Cf.list();
        }
        return Cf.x(pathInfo.substring(1).split("/"));
    }

    private static boolean parseDirName(IteratorF<String> pathIterator, Image.ImageBuilder imageBuilder) {
        final Option<String> dirName = pathIterator.nextO();
        if (dirName.isPresent()) {
            imageBuilder.dirName(dirName.get());
        }
        return dirName.isPresent();
    }

    private static boolean parsePlainFileName(IteratorF<String> pathIterator, Image.ImageBuilder imageBuilder) {
        final Option<String> plainFileNamePart = pathIterator.nextO();
        if (plainFileNamePart.isPresent()) {
            final ListF<String> plainFileNameParts = Cf.x(plainFileNamePart.get().split("_"));
            if (plainFileNameParts.size() >= 4) {
                final IteratorF<String> partsIterator = plainFileNameParts.iterator();
                final Option<String> accessString = partsIterator.nextO(); //unused
                final boolean imgIdParsed = parseImgId(partsIterator, imageBuilder);
                final Option<String> randomPartString = partsIterator.nextO(); //unused
                partsIterator.nextO().ifPresent(versionAndSize -> parseVersionAndSize(versionAndSize, imageBuilder));
                return imgIdParsed;
            }
        }
        return false;
    }

    private static boolean parseImgId(IteratorF<String> partsIterator, Image.ImageBuilder imageBuilder) {
        final Option<String> imgIdStringO = partsIterator.nextO();
        if (imgIdStringO.isPresent()) {
            try {
                final String imgIdString = imgIdStringO.get().replace("STATIC", ""); //this part should be ignored
                final int imgId = Math.toIntExact(Long.parseLong(imgIdString, 16));
                imageBuilder.imageId(imgId);
            } catch (NumberFormatException | ArithmeticException ex) {
                return false;
            }
        }
        return imgIdStringO.isPresent();
    }

    private static void parseVersionAndSize(String versionAndSize, Image.ImageBuilder imageBuilder) {
        final ListF<String> versionAndSizeList = Cf.x(versionAndSize.split("-")); //version format like "-1-" which is going before size
        if (versionAndSizeList.isNotEmpty()) {
            if (versionAndSizeList.size() == 3) { //it means version is provided and we need to parse it
                try {
                    int version = Integer.parseInt(versionAndSizeList.get(1));
                    imageBuilder.version(version);
                } catch (NumberFormatException ex) {
                    //just not setting version
                }
            }
            final String sizeString =
                    StringUtils.substringBefore(versionAndSizeList.last(), ".") //ignoring extension like ".jpg"
                            .toUpperCase()
                            .replace("GIF", ""); //for cases like "GIFXL"
            try {
                ImageSize size = ImageSize.valueOf(sizeString);
                imageBuilder.size(size);
            } catch (IllegalArgumentException ex) {
                //just not setting size
            }
        }
    }

    private static boolean parseUser(IteratorF<String> pathIterator, Image.ImageBuilder imageBuilder) {
        final Option<String> user = pathIterator.nextO();
        if (user.isPresent()) {
            imageBuilder.login(user.get());
        }
        return user.isPresent();
    }

    private static boolean parseAlbum(IteratorF<String> pathIterator, Image.ImageBuilder imageBuilder) {
        final Option<String> albumStringO = pathIterator.nextO();
        if (albumStringO.isPresent()) {
            try {
                final int albumId = Math.toIntExact(Long.parseLong(albumStringO.get()));
                imageBuilder.albumId(albumId);
            } catch (NumberFormatException | ArithmeticException ex) {
                return false;
            }
        }
        return albumStringO.isPresent();
    }

    private static boolean parseView(IteratorF<String> pathIterator, Image.ImageBuilder imageBuilder) {
        final Option<String> viewStringO = pathIterator.nextO();
        if (viewStringO.isPresent()) {
            try {
                final int imageId = Math.toIntExact(Long.parseLong(viewStringO.get()));
                imageBuilder.imageId(imageId);
            } catch (NumberFormatException | ArithmeticException ex) {
                return false;
            }
        }
        return viewStringO.isPresent();
    }
}
