package ru.yandex.chemodan.app.grelka;

import java.time.Instant;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CompletableFuture;
import java.util.function.LongSupplier;
import java.util.function.Predicate;
import java.util.stream.Stream;

import lombok.AllArgsConstructor;
import lombok.ToString;
import lombok.val;
import org.dom4j.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.eventlog.events.fs.FsEvent;
import ru.yandex.chemodan.eventlog.log.TskvEventLogLine;
import ru.yandex.chemodan.eventlog.log.TskvLogLine;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.dataSize.DataSizeUnit;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author swined
 */
@AllArgsConstructor
class GrelkaEventLogListener implements LogListener {

    private static final Logger logger = LoggerFactory.getLogger(GrelkaEventLogListener.class);

    private final MapF<Predicate<TskvLogLine>, Predicate<TskvLogLine>> handlers =
            Cf.<Predicate<TskvLogLine>, Predicate<TskvLogLine>>map()
                    .plus1(fieldEquals("event_type", "fs-hardlink-copy"), this::handleFsEvent)
                    .plus1(fieldEquals("tskv_format", "mail-mxback-attach-log"), this::handleMailAttachEvent)
                    .plus1(fieldEquals("tskv_format", "mail-mxfront-attach-log"), this::handleMailAttachEvent)
                    .unmodifiable();

    private final Timer timer = new Timer();

    private final DocviewerClient docviewerClient;
    private final MpfsClient mpfsClient;
    private final GrelkaConfig config;
    private final YandexStaffUserRegistry yandexStaffUserRegistry;

    private static Function1B<TskvLogLine> fieldEquals(String name, String value) {
        return tskv -> tskv.getStringO(name).exists(value::equals);
    }

    @Override
    public void processLogLine(String line) {
        try {
            TskvLogLine tskv = TskvLogLine.parse(line);
            if (handlers.filterKeys(k -> k.test(tskv)).filterValues(v -> v.test(tskv)).isNotEmpty()) {
                tskv.getNonEmptyStringO("unixtime").mapToOptionalLong(Long::parseLong).ifPresent(
                        time -> logger.info("startLag= " + (Instant.now().getEpochSecond() - time)));
            }
        } catch (Exception e) {
            logger.error("failed to handle log line: " + line, e);
        }
    }

    @AllArgsConstructor
    @ToString(exclude = "size")
    private class WarmupCandidate {

        private final long unixtime;
        private final PassportUid uid;
        private final String ext;
        private final String url;
        private final LongSupplier size;

        boolean warmup() {
            Option<DataSize> sizeConfig = config.getSize(ext);
            if (yandexStaffUserRegistry.isStaffUser(uid) || (sizeConfig.isPresent() && sizeMatches(sizeConfig.get()))) {
                startConvertToHtml();
                return true;
            } else {
                return false;
            }
        }

        private void startConvertToHtml() {
            logger.info("converting " + this);
            docviewerClient.startConvertToHtml(uid.toUidOrZero(), url, Option.empty(), true);
            scheduleAvailabilityCheck();
        }

        private void scheduleAvailabilityCheck() {
            if (getConversionLag() > config.getAvailabilityCheckTimeout()) {
                logger.warn("conversion timed out " + this);
                logger.info("conversionLag=" + getConversionLag());
                return;
            }
            timer.schedule(new TimerTask() {
                @Override
                public void run() {
                    CompletableFuture.runAsync(WarmupCandidate.this::checkAvailability);
                }
            }, config.getAvailabilityCheckInterval());
        }

        private void checkAvailability() {
            if (isAvailable()) {
                logger.info("conversionLag=" + getConversionLag());
            } else {
                scheduleAvailabilityCheck();
            }
        }

        private long getConversionLag() {
            return Instant.now().getEpochSecond() - unixtime;
        }

        private boolean isAvailable() {
            try {
                Option<String> stateO = parseStateFromUrlInfo(docviewerClient.urlInfo(uid.toUidOrZero(), url));
                if (!stateO.isPresent()) {
                    logger.debug("state is empty");
                    return false;
                }
                String state = stateO.get();
                logger.debug("checking availability of " + uid + "/" + url + ": " + state);
                return Stream.of("AVAILABLE", "CONVERTING_ERROR").anyMatch(state::equalsIgnoreCase);
            } catch (Exception e) {
                logger.error(e);
                return false;
            }
        }

        private boolean sizeMatches(DataSize dataSize) {
            long limit = dataSize.toBytes();
            return limit <= 0 || size.getAsLong() >= limit;
        }

    }

    static Option<String> parseStateFromUrlInfo(Element urlInfo) {
        return Option.ofNullable(urlInfo)
                .map(element -> element.element("convert-state"))
                .filterNotNull()
                .map(element -> element.element("state"))
                .map(Element::getStringValue);
    }

    private static String getExt(String name) {
        return StringUtils.substringAfterLast(name, ".");
    }

    private long getSizeFromMpfs(PassportUid uid, String path) {
        return mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(uid), path, Cf.list("size")).getMeta().getSize().get()
                .to(DataSizeUnit.BYTES);
    }

    private boolean handleFsEvent(TskvLogLine line) {
        val event = FsEvent.parse(new TskvEventLogLine(line));
        val uid = event.getPassportUid();
        return event.resourceChange.targetInfo.targetPath.map(path -> new WarmupCandidate(line.getLong("unixtime"), uid,
                getExt(path), String.format("ya-disk://%s", path), () -> getSizeFromMpfs(uid, path)).warmup()).getOrElse(false);
    }

    private boolean handleMailAttachEvent(TskvLogLine line) {
        val url = String.format("ya-mail://%s/%s", line.getLong("mid"), line.getNonEmptyString("hid"));
        return new WarmupCandidate(line.getLong("unixtime"), new PassportUid(line.getLong("uid")),
                getExt(line.getNonEmptyString("name")), url, () -> line.getLong("size")).warmup();
    }

}
