package ru.yandex.chemodan.app.grelka;

import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsResourceTimes;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyBoolean;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;

/**
 * @author swined
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
        GrelkaEventLogListenerTest.Config.class
})
public class GrelkaEventLogListenerTest extends AbstractTest {

    public final static List<Tuple4<PassportUidOrZero, String, Option<String>, Boolean>> startConvertToHtmlRequests = new ArrayList<>();

    @Configuration
    @Import(GrelkaContextConfiguration.class)
    public static class Config {

        @Bean
        public DocviewerClient docviewerClient() {
            DocviewerClient mock = mock(DocviewerClient.class);
            doAnswer(invocation -> startConvertToHtmlRequests.add(new Tuple4<>(
                    (PassportUidOrZero) invocation.getArguments()[0],
                    (String) invocation.getArguments()[1],
                    (Option<String>) invocation.getArguments()[2],
                    (Boolean) invocation.getArguments()[3]))).when(mock).startConvertToHtml(any(PassportUidOrZero.class), anyString(), any(Option.class), anyBoolean());

            return mock;
        }

        @Bean
        public FakeMpfsClient mpfsClient() {
            return new FakeMpfsClient("wtf");
        }

        @Bean
        public GrelkaConfig grelkaConfig() {
            return new GrelkaConfig() {
                @Override
                public Option<DataSize> getSize(String ext) {
                    return Option.of(DataSize.MEGABYTE);
                }
            };
        }

    }

    private static class FakeMpfsClient extends MpfsClientImpl {

        public FakeMpfsClient(String host) {
            super(host);
        }

        @Override
        public MpfsFileInfo getFileInfoByUidAndPath(MpfsUser uid, String path, ListF<String> metaFields) {
            switch (uid.getUidStr() + " " + path) {
                case "4000475747 /disk/Загрузки/file_from_archive.pptx":
                    return sizeOnly(DataSize.fromMegaBytes(2));
                case "4000475747 /disk/Загрузки/file_from_archive.PPTX":
                    return sizeOnly(DataSize.fromMegaBytes(2));
                case "4000475747 /disk/Загрузки/small_file_from_archive.pptx":
                    return sizeOnly(DataSize.fromKiloBytes(2));
                default:
                    throw new UnsupportedOperationException(uid + " " + path);
            }
        }

        private MpfsFileInfo sizeOnly(DataSize size) {
            return new MpfsFileInfo(
                    Option.empty(),
                    Option.empty(),
                    Option.empty(),
                    MpfsFileMetaDto.builder().size(size).build(),
                    MpfsResourceTimes.ZERO
            );
        }
    }

    @Autowired
    private GrelkaEventLogListener listener;

    @Autowired
    private DocviewerClient docviewer;

    @Before
    public void cleanupDocviewer() {
        startConvertToHtmlRequests.clear();
    }

    @Test
    public void testStoreTxt() throws InterruptedException {
        listener.processLogLine("tskv\tunixtime=1508931248\thostname=mpfs02f\tevent_type=fs-hardlink-copy\treq_id=mpfs-870deb4bb00e562f4ab278a4f7a00637-mpfs02f\ttskv_format=ydisk-event-history-log\ttgt_folder_id=4000475747:1b091b2934d607643ad0301b3b0ef52b9bf829ef8b4f5221e0a601cf5607964d\ttgt_rawaddress=4000475747:/disk/Загрузки/file_from_archive.txt\tuid=4000475747\tresource_file_id=fe0aa6ce0f30c5735a3a593535269f24115e846bde2be25e718a035d5738205a\tchanges=\towner_uid=4000475747\tsubtype=extract-file-from-archive\tlenta_media_type=document\tresource_media_type=document\ttype=store\tresource_type=file");
        Assert.assertHasSize(0, startConvertToHtmlRequests);
    }

    @Test
    public void testStorePptx() throws InterruptedException {
        listener.processLogLine("tskv\tunixtime=1508931248\thostname=mpfs02f\tevent_type=fs-hardlink-copy\treq_id=mpfs-870deb4bb00e562f4ab278a4f7a00637-mpfs02f\ttskv_format=ydisk-event-history-log\ttgt_folder_id=4000475747:1b091b2934d607643ad0301b3b0ef52b9bf829ef8b4f5221e0a601cf5607964d\ttgt_rawaddress=4000475747:/disk/Загрузки/file_from_archive.pptx\tuid=4000475747\tresource_file_id=fe0aa6ce0f30c5735a3a593535269f24115e846bde2be25e718a035d5738205a\tchanges=\towner_uid=4000475747\tsubtype=extract-file-from-archive\tlenta_media_type=document\tresource_media_type=document\ttype=store\tresource_type=file");
        Assert.assertHasSize(1, startConvertToHtmlRequests);
        Assert.equals(startConvertToHtmlRequests.get(0).toString(), "(4000475747, ya-disk:///disk/Загрузки/file_from_archive.pptx, None, true)");
    }

    @Test
    public void testStorePptxUppercase() throws InterruptedException {
        listener.processLogLine("tskv\tunixtime=1508931248\thostname=mpfs02f\tevent_type=fs-hardlink-copy\treq_id=mpfs-870deb4bb00e562f4ab278a4f7a00637-mpfs02f\ttskv_format=ydisk-event-history-log\ttgt_folder_id=4000475747:1b091b2934d607643ad0301b3b0ef52b9bf829ef8b4f5221e0a601cf5607964d\ttgt_rawaddress=4000475747:/disk/Загрузки/file_from_archive.PPTX\tuid=4000475747\tresource_file_id=fe0aa6ce0f30c5735a3a593535269f24115e846bde2be25e718a035d5738205a\tchanges=\towner_uid=4000475747\tsubtype=extract-file-from-archive\tlenta_media_type=document\tresource_media_type=document\ttype=store\tresource_type=file");
        Assert.assertHasSize(1, startConvertToHtmlRequests);
        Assert.equals(startConvertToHtmlRequests.get(0).toString(), "(4000475747, ya-disk:///disk/Загрузки/file_from_archive.PPTX, None, true)");
    }

    @Test
    public void testStorePptxSmall() throws InterruptedException {
        listener.processLogLine("tskv\tunixtime=1508931248\thostname=mpfs02f\tevent_type=fs-hardlink-copy\treq_id=mpfs-870deb4bb00e562f4ab278a4f7a00637-mpfs02f\ttskv_format=ydisk-event-history-log\ttgt_folder_id=4000475747:1b091b2934d607643ad0301b3b0ef52b9bf829ef8b4f5221e0a601cf5607964d\ttgt_rawaddress=4000475747:/disk/Загрузки/small_file_from_archive.pptx\tuid=4000475747\tresource_file_id=fe0aa6ce0f30c5735a3a593535269f24115e846bde2be25e718a035d5738205a\tchanges=\towner_uid=4000475747\tsubtype=extract-file-from-archive\tlenta_media_type=document\tresource_media_type=document\ttype=store\tresource_type=file");
        Assert.assertHasSize(0, startConvertToHtmlRequests);
    }

    @Test
    public void testMailAttachment() {
        listener.processLogLine("tskv\ttskv_format=mail-mxback-attach-log\tuid=4007637622\tsuid=\tmid=164099911422370463\tfidType=Drafts\tfidSpecType=5\thid=1.2\tfileType=text/plain\tname=plain_text_attachment.pptx\tsize=1200000\tstid=320.mail:4007637622.E4192:55610879214148542574726660618\tunixtime=1512982188");
        Assert.assertHasSize(1, startConvertToHtmlRequests);
        Assert.equals(startConvertToHtmlRequests.get(0).toString(), "(4007637622, ya-mail://164099911422370463/1.2, None, true)");
    }

    @Test
    public void testParseStateFromErroneousUrlInfo() {
        String urlInfo = "<state>" +
                "<copy-error>ru.yandex.inside.wmi.WmiResponseParserException: Couldn't parse attachment info, mid: 123, hid: 1.3, data: {\"mimes\":{}}</copy-error>" +
                "<copy-error-code>UNKNOWN_COPY_ERROR</copy-error-code>" +
                "<state>COPY_ERROR</state>" +
                "</state>";
        Option<String> state = GrelkaEventLogListener.parseStateFromUrlInfo(
                Dom4jUtils.readRootElement(urlInfo.getBytes())
        );
        Assert.none(state);
    }
}
