package ru.yandex.chemodan.app.hackathon;

import java.io.IOException;
import java.io.StringReader;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.protobuf.ExtensionRegistry;
import com.google.protobuf.Message;
import com.google.protobuf.util.JsonFormat;

import ru.yandex.alice.megamind.protos.scenarios.RequestProto;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.web.servlet.HttpServletResponseX;
import ru.yandex.passport.tvmauth.CheckedUserTicket;

/**
 * @author tolmalev
 */
public class AliceScenarioServlet extends HttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(AliceScenarioServlet.class);
    private final ExtensionRegistry registry = ExtensionRegistry.newInstance();

    private final DynamicProperty<ListF<String>> stopWords =
            new DynamicProperty<>("hackaton-stop-words", Cf.list("хватит"));

    private final ListF<AliceScenario> scenarios;

    private final Tvm2 tvm2;

    public AliceScenarioServlet(List<AliceScenario> scenarios, Tvm2 tvm2) {
        this.scenarios = Cf.x(scenarios);
        this.tvm2 = tvm2;

        RequestProto.getDescriptor().getExtensions().forEach(this.registry::add);
        ResponseProto.getDescriptor().getExtensions().forEach(this.registry::add);
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        HttpServletRequestX reqX = HttpServletRequestX.wrap(req);
        HttpServletResponseX respX = HttpServletResponseX.wrap(resp);

        Option<Long> uid = extractUid(reqX);
        logger.debug("extracted uid: {}", uid);

        if (!uid.isPresent()) {
            resp.setStatus(401);
            return;
        }

        switch (reqX.getPathInfo()) {
            case "/run": serializeResponse(reqX, respX, processRunRequest(uid.get(), reqX)); break;
            case "/apply": serializeResponse(reqX, respX, processApplyRequest(uid.get(), reqX)); break;
            default: {
                respX.setStatus(400);
                respX.writeContent(("Unknown method " + reqX.getPathInfo()).getBytes(), "text/plain");
                logger.error("Unknown method: {}", reqX.getPathInfo());
            }
        }
    }

    private <T> T parseRequest(HttpServletRequestX reqX, com.google.protobuf.GeneratedMessageV3.Builder builder, Class<T> clazz) {
        try {
            byte[] data = reqX.getInputStreamX().readBytes();

            if ("application/json".equals(reqX.getContentType())) {
                JsonFormat.parser().merge(new StringReader(new String(data)), builder);
                return (T) builder.build();
            } else {
                return (T) builder.mergeFrom(data, registry).build();
            }
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private Message processRunRequest(long uidL, HttpServletRequestX reqX) {
        RequestProto.TScenarioRunRequest request = parseRequest(reqX, RequestProto.TScenarioRunRequest.newBuilder(), RequestProto.TScenarioRunRequest.class);
        logger.info("runRequest: {}", serializeMessageJson(request));

        String requestText = request.getInput().getVoice().getUtterance();
        if (requestText == null || requestText.equals("")) {
            requestText = request.getInput().getText().getUtterance();
        }

        logger.info("request text: {}", requestText);

        ListF<String> words = Cf.x(requestText.replaceAll("[,\\.!?]", " ").split(" "))
                .filter(StringUtils::isNotBlank)
                .map(s -> s.toLowerCase().trim());

        logger.info("request words: {}", words);

        Option<AliceScenario> activeScenario = scenarios.find(AliceScenario::isActive);

        if (activeScenario.isPresent()) {
            if (words.exists(word -> stopWords.get().containsTs(word))) {
                logger.info("terminating active scenario: {}", activeScenario.get());
                activeScenario.get().forceDeactivate();
                return HackatonUtils.createSimpleTextResponse("Хорошо, давай займёмся чем-нибудь ещё");
            }

            logger.info("active scenario: {}", activeScenario.get());
            return activeScenario.get().run(uidL, reqX, words, request);
        }

        Tuple2List<AliceScenario, Integer> scenariosWithMatch = scenarios
                .zipWith(s -> s.match(words))
                .sortedBy2Desc();

        logger.info("no scenario matches: {}", scenariosWithMatch);

        if (scenariosWithMatch.isEmpty() || scenariosWithMatch.first()._2 == 0) {
            logger.info("no scenario matched. total = {}", scenariosWithMatch.size());
            throw new IllegalStateException("No scenarios at all, even default: " + scenariosWithMatch);
        } else {
            Tuple2<AliceScenario, Integer> t2 = scenariosWithMatch.first();
            logger.info("matched scenario: {}, match = {}", t2._1, t2._2);

            return t2._1.run(uidL, reqX, words, request);
        }
    }

    private Message processApplyRequest(long uid, HttpServletRequestX reqX) {
        RequestProto.TScenarioApplyRequest request = parseRequest(reqX, RequestProto.TScenarioApplyRequest.newBuilder(), RequestProto.TScenarioApplyRequest.class);
        logger.info("applyRequest: {}", serializeMessageJson(request));

        return ResponseProto.TScenarioApplyResponse
                .newBuilder()
                .setResponseBody(ResponseProto.TScenarioResponseBody
                        .newBuilder()
                        .setLayout(ResponseProto.TLayout
                                .newBuilder()
                                .addCards(ResponseProto.TLayout.TCard
                                        .newBuilder()
                                        .setText("Метод Apply не поддержан")
                                )
                                .setOutputSpeech("Метод Apply не поддержан")
                        ))
                .build();
    }

    private String serializeMessageJson(Message request) {
        try {
            return JsonFormat.printer().print(request);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private void serializeResponse(HttpServletRequestX reqX, HttpServletResponseX respX, Message message) throws IOException {
        boolean json = "application/json".equals(reqX.getContentType());
        if (message != null) {
            if (json) {
                respX.writeContent(serializeMessageJson(message).getBytes(), "application/json");
            } else {
                respX.writeContent(message.toByteArray(), "application/protobuf");
            }
        }
        respX.flushBuffer();

        logger.debug("Result: {}", serializeMessageJson(message));
    }

    private Option<Long> extractUid(HttpServletRequestX reqX) {
        Option<CheckedUserTicket> userTicketO = reqX.getHeaderO(TvmHeaders.USER_TICKET).map(tvm2::checkUserTicket);
        if (userTicketO.isPresent()) {
            return Option.of(userTicketO.get().getDefaultUid());
        }
        return reqX.getParameterO("uid")
                .flatMapO(Cf.Long::parseSafe)
                .orElse(Option.of(50273844L));
    }
}
