package ru.yandex.chemodan.app.hackathon;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.alice.megamind.protos.scenarios.RequestProto;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author yashunsky
 */
public class DemotivationScenario implements AliceScenario {
    private static final Logger logger = LoggerFactory.getLogger(DemotivationScenario.class);

    private static final String CAPTIONS_FILE_NAME = "demotivations.txt";

    private final DynamicProperty<Double> beautyEdge =
            new DynamicProperty<>("demotivation-beauty-edge", -2.0);
    private final DynamicProperty<Double> matchPercentil =
            new DynamicProperty<>("demotivation-match-percentil", 0.9);
    private final DynamicProperty<Integer> topBest =
            new DynamicProperty<>("demotivation-topBest", 20);
    private final DynamicProperty<Boolean> photoFirst =
            new DynamicProperty<>("demotivation-photo-first", true);
    private final DynamicProperty<ListF<String>> activation =
            new DynamicProperty<>("demotivation-activation", Cf.list("демотивируй", "поумничай про"));

    private final DivTemplateProcessor templateProcessor = new DivTemplateProcessor("demotivated_photo.ftj");

    @Autowired
    private Cvi2tProcessor cvi2tProcessor;
    @Autowired
    private SearchCache searchCache;
    @Autowired
    private HackatonMpfsClient hackatonMpfsClient;

    @Override
    public int match(ListF<String> text) {
        String phrase = text.mkString(" ");
        return activation.get().exists(phrase::contains) ? 100 : 0;
    }

    @Override
    public ResponseProto.TScenarioRunResponse run(long uid, HttpServletRequestX reqX, ListF<String> words,
            RequestProto.TScenarioRunRequest request)
    {
        logger.info("Got demotivation request for {}: {}", uid, words);
        ListF<String> requestWords = getRequestWords(words);

        String caption;
        DiskSearchFileInfo mostMatchingPhoto;

        if (requestWords.isEmpty()) {
            caption = Random2.R.randomElement(getCaptions());
            mostMatchingPhoto = getMostMatchingPhoto(uid, Cf.list(caption));
        }else if (photoFirst.get()) {
            mostMatchingPhoto = getMostMatchingPhoto(uid, requestWords);
            caption = getMostMatchingCaption(mostMatchingPhoto);
        } else {
            caption = getMostMatchingCaption(requestWords);
            mostMatchingPhoto = getMostMatchingPhoto(uid, Cf.list(caption));
        }

        String previewUrl = hackatonMpfsClient.getPreviews(uid, Cf.list(mostMatchingPhoto)).first();
        String divJson = templateProcessor.processTemplate(
                Cf.map("image_url", previewUrl, "text", caption.toUpperCase()));

        logger.info("div json: {}", divJson);

        return HackatonUtils.createResponse(divJson, Option.of(caption));
    }

    public DiskSearchFileInfo getMostMatchingPhoto(long uid, ListF<String> words) {
        ListF<DiskSearchFileInfo> photos = getAllPhotos(uid);

        return getMostMatching(photos, photo -> cvi2tProcessor.dotProduct(photo, words));
    }

    public ListF<DiskSearchFileInfo> getAllPhotos(long uid) {
        return searchCache.getAllPhotos(uid)
                .filter(photo -> photo.beautiful2.exists(beauty -> beauty >= beautyEdge.get()))
                .sortedByDesc(p -> p.etime.get())
                .take(30000);
    }

    public String getMostMatchingCaption(DiskSearchFileInfo info) {
        return getMostMatching(getCaptions(), caption ->
                Cf.list(Cvi2tProcessor.dotProduct(info.geti2tVector(), cvi2tProcessor.getTextVector(caption))));
    }

    public String getMostMatchingCaption(ListF<String> words) {
        ListF<String> captions = getCaptions();
        return getMostMatching(captions,
                caption -> cvi2tProcessor.dotProduct(cvi2tProcessor.getTextVector(caption), words));
    }

    public long getPercentile(ListF<Long> data, double percentile) {
        return data.sorted().get((int) (data.length() * percentile));
    }

    public ListF<String> getCaptions() {
        return new ClassPathResourceInputStreamSource(DemotivationScenario.class, CAPTIONS_FILE_NAME).readLines()
                .filterNot(s -> s.startsWith("#"));
    }

    public <T> T getMostMatching(ListF<T> elements, Function<T, ListF<Long>> getMatches) {
        ListF<T> bestElements = elements.zipWith(element ->
                getPercentile(getMatches.apply(element), matchPercentil.get()))
                .sortedBy2Desc().get1().take(topBest.get());

        return Random2.R.randomElement(bestElements);
    }

    public ListF<String> getRequestWords(ListF<String> words) {
        ListF<String> wordsToIgnore = activation.get().flatMap(phrase -> Cf.x(phrase.split(" ")));
        ListF<String> requestWords = words.filterNot(word -> wordsToIgnore.containsTs(word)
                || activation.get().exists(word::contains));
        logger.info("words left for request: {}", requestWords);
        return requestWords;
    }
}
