package ru.yandex.chemodan.app.hackathon;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import lombok.Data;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.alice.megamind.protos.scenarios.RequestProto;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
public class PhotoStatisticsScenario implements AliceScenario {
    private final DivTemplateProcessor templateProcessor = new DivTemplateProcessor("one_photo.ftj");

    @Autowired
    private SearchCache searchCache;

    @Autowired
    private BlockTitlesGenerator blockTitlesGenerator;

    @Autowired
    private Cvi2tProcessor cvi2tProcessor;

    @Autowired
    private HackatonMpfsClient hackatonMpfsClient;

    private final ListF<FactSelector> selectors = Cf.list(
            new CvStatsSelecctor(),
            new PhotoDateSelector(),
            new PhotosPerYearSelector()
    );

    private final AtomicInteger index = new AtomicInteger(0);

    private ListF<FactSelector> lastSelectors = Cf.arrayList();

    @Override
    public int match(ListF<String> text) {
        return (
                text.mkString(" ").contains("какой я фотограф")
                || text.mkString(" ").contains("какой фотограф")
                || text.mkString(" ").matches("думаешь .*про мои фот")
                || text.mkString(" ").matches("скажешь .*про мои фот")
                || text.mkString(" ").matches("расскажи .*про мои фот")
                ) ? 100 : 0;
    }

    @Override
    public ResponseProto.TScenarioRunResponse run(long uid, HttpServletRequestX reqX, ListF<String> words, RequestProto.TScenarioRunRequest request) {
        Fact fact = getRandomfact(uid);

        if (fact.imageResourceId.isPresent()) {
            String previewUrl = hackatonMpfsClient.getPreviewsByResourceIds(uid, Cf.list(uid + ":" + fact.imageResourceId.get())).first();
            String divJson = templateProcessor.processTemplate(Cf.map("image_url", previewUrl));

            return ResponseProto.TScenarioRunResponse
                    .newBuilder()
                    .setResponseBody(ResponseProto.TScenarioResponseBody
                            .newBuilder()
                            .setLayout(ResponseProto.TLayout
                                    .newBuilder()
                                    .addCards(ResponseProto.TLayout.TCard
                                            .newBuilder()
//                                             .setDivCard(divJson) see: MEGAMIND-378
                                    )
                                    .setOutputSpeech(fact.text)
                            ))
                    .build();
        } else {
            return ResponseProto.TScenarioRunResponse
                    .newBuilder()
                    .setResponseBody(ResponseProto.TScenarioResponseBody
                            .newBuilder()
                            .setLayout(ResponseProto.TLayout
                                    .newBuilder()
                                    .addCards(ResponseProto.TLayout.TCard
                                            .newBuilder()
                                            .setText(fact.text)
                                    )
                                    .setOutputSpeech(fact.text)
                            ))
                    .build();
        }
    }

    public Fact getRandomfact(long uid) {
        for (int i = 0; i < 10; i++) {
            FactSelector selector = selectors.get(index.getAndIncrement() % selectors.size());

            Option<Fact> fact = selector.getFact(uid);
            if (fact.isPresent()) {
                return fact.get();
            }
        }
        return new Fact("Никакой");
    }

    @Data
    public static class Fact {
        public final String text;
        public final Option<String> imageResourceId;

        public Fact(String text) {
            this.text = text;
            this.imageResourceId = Option.empty();
        }

        public Fact(String text, String imageResourceId) {
            this.text = text;
            this.imageResourceId = Option.of(imageResourceId);
        }
    }

    public interface FactSelector {
        Option<Fact> getFact(long uid);
    }

    public ListF<DiskSearchFileInfo> getPhotos(long uid) {
        return searchCache.getAllPhotos(uid)
                .filter(p -> p.etime.isPresent())
                .filter(p -> p.getBeautiful2().isPresent())
                .filter(p -> p.geti2tVector().length > 10);
    }

    public class OldestPhotoSelector implements FactSelector {
        @Override
        public Option<Fact> getFact(long uid) {

            return getPhotos(uid)
                    .minByO(p -> p.etime.get())
                    .map(photo -> {
                        TitleGenerationContext context =
                                new TitleGenerationContext(Random2.R, IntervalType.ONE_DAY, new DateTime(photo.etime.get() * 1000));

                        return new Fact(
                                blockTitlesGenerator.processTemplate("Ты снимаешь фото начиная с {{date.start:d}} {{month.start:gen}} {{date.start:yyyy}} года", Language.RUSSIAN, context)
                        );
                    });
        }
    }

    public class NewestPhotoSelector implements FactSelector {
        @Override
        public Option<Fact> getFact(long uid) {

            return getPhotos(uid)
                    .maxByO(p -> p.etime.get())
                    .map(photo -> {
                        TitleGenerationContext context =
                                new TitleGenerationContext(Random2.R, IntervalType.ONE_DAY, new DateTime(photo.etime.get() * 1000));

                        return new Fact(
                                blockTitlesGenerator.processTemplate("Последнее фото сделано {{date.start:d}} {{month.start:gen}} {{date.start:yyyy}} года", Language.RUSSIAN, context)
                        );
                    });
        }
    }

    public class PhotosPerYearSelector implements FactSelector {

        @Override
        public Option<Fact> getFact(long uid) {
            MapF<Integer, Integer> countByYear = searchCache.getAllPhotos(uid)
                    .flatMap(p -> p.etime)
                    .map(etime -> new DateTime(etime * 1000))
                    .map(dt -> dt.getYear())
                    .groupBy(i -> i)
                    .mapValues(List::size)
                    .filterValues(cnt -> cnt > 50);

            if (countByYear.size() == 0) {
                return Option.empty();
            }

            Tuple2List<Integer, Integer> sorted = countByYear.entries().sortedBy2Desc();
            Tuple2List<Integer, Integer> part = sorted.take(sorted.size() / 4);
            if (part.size() == 0) {
                part = sorted;
            }

            Tuple2<Integer, Integer> t2 = Random2.R.randomElement(part);
            int year = t2._1;
            int count = t2._2;

            ListF<String> variants = Cf.list(
                    year + " - выдался \"фотогодичным\"! " + count + " фоточек нащелкал",
                    "У тебя " + count + " фоток за " + year + "год",
                    "За " + year + " у тебя " + count + " фоток! Это в среднем " + count / 365 + " в день"
            );

            return Option.of(new Fact(Random2.R.randomElement(variants)));
        }
    }

    public class CvStatsSelecctor implements FactSelector {

        ListF<String> word1 = Cf.list("котиков", "горы", "цветы", "напитки", "селфи", "лето", "пиццу", "кофе");
        ListF<String> word2 = Cf.list("собак", "море", "снег", "блюда", "групповое фото", "зиму", "мороженое", "пиво");

        @Override
        public Option<Fact> getFact(long uid) {
            int index = Random2.R.nextInt(word1.size());

            String word1 = this.word1.get(index);
            String word2 = this.word2.get(index);

            Tuple2List<DiskSearchFileInfo, ListF<Long>> photosWithMatch = getPhotos(uid)
                    .sortedByDesc(p -> p.etime.get())
                    .take(30000)
                    .zipWith(pi -> cvi2tProcessor.dotProduct(pi, Cf.list(word1, word2)));

            ListF<ListF<Long>> results = photosWithMatch
                    .get2();

            int cnt1 = results.count(list -> list.get(0) > 6000);
            int cnt2 = results.count(list -> list.get(1) > 6000);

            if (cnt1 == 0 && cnt2 == 0) {
                return Option.empty();
            }

            String maxW = word1;
            String minW = word2;
            String resourceId = photosWithMatch.maxBy(t2 -> t2._2.get(0))._1.id;

            if (cnt2 > cnt1) {
                maxW = word2;
                minW = word1;
                resourceId = photosWithMatch.maxBy(t2 -> t2._2.get(1))._1.id;
            }

            ListF<String> variants = Cf.list(
                    "Кажется, ты любишь " + maxW + " больше, чем " + minW,
                    "Кажется, ты снимаешь " + maxW + " чаще, чем " + minW
            );

            return Option.of(new Fact(Random2.R.randomElement(variants), resourceId));
        }
    }

    public class PhotoDateSelector implements FactSelector {

        @Override
        public Option<Fact> getFact(long uid) {
            return Random2.R
                    .randomElement(Cf.list(new OldestPhotoSelector(), new NewestPhotoSelector()))
                    .getFact(uid);
        }
    }

}
