package ru.yandex.chemodan.app.hackathon;

import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.Service;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
public class SearchCache implements Service {
    private static final Logger logger = LoggerFactory.getLogger(SearchCache.class);

    private final Yt ytClient;
    private final DiskSearchClient diskSearchClient;
    private final MapF<Long, ListF<DiskSearchFileInfo>> cache = Cf.concurrentHashMap();

    private final YTableEntryType<DiskSearchFileInfo> type = YTableEntryTypes.bender(DiskSearchFileInfo.class);

    private final AtomicInteger totalPhotos = new AtomicInteger(0);

    public SearchCache(Yt ytClient, DiskSearchClient diskSearchClient) {
        this.ytClient = ytClient;
        this.diskSearchClient = diskSearchClient;
    }

    public ListF<DiskSearchFileInfo> getAllPhotos(long uid) {
        return cache.getOrElseUpdate(uid, () -> loadFromSearch(uid));
    }

    private ListF<DiskSearchFileInfo> loadFromSearch(long uid) {
        ListF<DiskSearchFileInfo> items = diskSearchClient.findPhotosWithBeauty(PassportUid.cons(uid),
                new InstantInterval(Instant.now().minus(Duration.standardDays(3650)), Instant.now()),
                Option.empty(), 0, 100000
        ).hitsArray;
        totalPhotos.addAndGet(items.size());
        return items;
    }

    @Override
    public void start() throws Exception {
        new Thread() {
            @Override
            public void run() {
                loadDataFromYt();
            }
        }.start();
    }

    public void loadDataFromYt() {
        ExecutorService executorService = Executors.newFixedThreadPool(30, new ThreadNameIndexThreadFactory("yt-loader"));

        Cf.list(50273844, 789428872, 322147061, 122625849).forEach(uidL -> executorService.submit(() -> loadDataFromYt(uidL)));

//        ytClient.cypress().list(YPath.simple("//home/disk-dev/tolmalev/hackathon"))
//                .flatMap(node -> Cf.Long.parseSafe(node.getValue().split("_")[0]))
//                .forEach(uidL -> executorService.submit(() -> loadDataFromYt(uidL)));
    }

    public void loadDataFromYt(long uid) {
        logger.info("Start load from yt: {}", uid);
        Instant start = Instant.now();

        YPath yPath = YPath.simple("//home/disk-dev/tolmalev/hackathon/" + uid + "_all_photos");

        ListF<DiskSearchFileInfo> items = Cf.arrayList();
        ytClient.tables().read(Optional.empty(), false, yPath, type, it -> {
            while (it.hasNext()) {
                items.add(it.next());
            }
            return null;
        });
        logger.info("Loaded data from yt, uid={}, time={}", uid, TimeUtils.secondsStringToNow(start));

        cache.put(uid, items);
        totalPhotos.addAndGet(items.size());

        logger.info("total loaded: {}", totalPhotos.get());
    }

    @Override
    public void stop() throws Exception {
    }
}
