package ru.yandex.chemodan.app.hackathon;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import lombok.Data;
import org.springframework.beans.factory.annotation.Autowired;

// import ru.yandex.alice.megamind.protos.scenarios.DirectivesProto;
import ru.yandex.alice.megamind.protos.scenarios.RequestProto;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author m-smolina
 */
public class WhatSongScenario implements AliceScenario {
    private final DynamicProperty<Double> beautyEdge = new DynamicProperty<>("song-beauty-edge", 1.0);

    private static final String ACTIVATION_WORD = "песн";
    private static final String ACTIVATION_WORD2 = "музык";
    private static final Logger logger = LoggerFactory.getLogger(GetPhotoSelectionsScenario.class);
    private static final String CAPTIONS_FILE_NAME = "songs.txt";
    private static final int LINES_SIZE = 8;
    private static final int SONGS_SIZE = 7;
    private static List<Song> SONGS = new ArrayList<>(SONGS_SIZE);

    static {
        try {
            try (BufferedReader br = new BufferedReader(
                new InputStreamReader(
                    WhatSongScenario.class.getResourceAsStream(CAPTIONS_FILE_NAME),
                    StandardCharsets.UTF_8)))
            {
                for (int j = 0; j < SONGS_SIZE; j++) {
                    Song song = new Song();
                    String title = br.readLine();
                    song.setTitle(title.toLowerCase(Locale.getDefault()));
                    List<String> lines = new ArrayList<>(LINES_SIZE);
                    for (int i = 0; i < LINES_SIZE; i++) {
                        String line = br.readLine();
                        lines.add(line.toLowerCase(Locale.getDefault()));
                    }
                    song.setLines(Cf.x(lines));
                    SONGS.add(song);
                }
            }
        } catch (IOException e) {
            logger.error("Read songs file error", e);
        }
    }

    private final DivTemplateProcessor templateProcessor = new DivTemplateProcessor("songs.ftj");
    @Autowired
    private Cvi2tProcessor cvi2tProcessor;
    @Autowired
    private SearchCache searchCache;
    @Autowired
    private HackatonMpfsClient hackatonMpfsClient;

    @Override
    public int match(ListF<String> text) {
        return text.exists(
            s -> s.startsWith(ACTIVATION_WORD) || s.startsWith(ACTIVATION_WORD2)) ? 10 : 0;
    }

    private Song selectedSong = null;

    @Override
    public ResponseProto.TScenarioRunResponse run(long uidL, HttpServletRequestX reqX, ListF<String> words, RequestProto.TScenarioRunRequest request) {
        if (selectedSong != null) {
            return checkResponse(uidL, words, request);
        }

        ListF<ListF<DiskSearchFileInfo>> photos = Cf.arrayList();
        Song song = Random2.R.randomElement(SONGS);
        selectedSong = song;

        logger.info("SELECTED song : {}", song.title);
        for(String line: song.getLines().take(4)) {
            photos.add(getPhotos(uidL, line));
        }

        ListF<BlockModel> blockModels = getBlockModels(uidL, photos);

        String divJson = templateProcessor.processTemplate(Cf.map("blocks", blockModels));
        logger.info("div json: {}", divJson);

        ListF<Song> variants = Random2.R.randomElements(SONGS, 4).plus(song).unique().shuffle();

        return ResponseProto.TScenarioRunResponse
            .newBuilder()
            .setResponseBody(ResponseProto.TScenarioResponseBody
                .newBuilder()
                .setLayout(ResponseProto.TLayout
                    .newBuilder()
                    .addCards(ResponseProto.TLayout.TCard
                        .newBuilder()
//                         .setDivCard(divJson)
                    )
                    .addCards(ResponseProto.TLayout.TCard
                            .newBuilder()
                            .setTextWithButtons(ResponseProto.TLayout.TTextWithButtons
                                    .newBuilder()
                                    .setText("Угадай какая песня")
                                    .addAllButtons(variants.map(s -> ResponseProto.TLayout.TButton
                                            .newBuilder()
                                            .setTitle(s.title)
                                            //.addDirectives(DirectivesProto.TDirective
                                            //        .newBuilder()
                                            //        .setTypeTextDirective(DirectivesProto.TTypeTextDirective.newBuilder()
                                            //                .setName(s.title)
                                            //                .setText(s.title)
                                            //                .build())
                                            //)
                                    .build()
                                    ))
                                )
                    )
                    .setOutputSpeech("Давайте сыграем. Попробуйте угадать песню по своим фото")
                ))
            .build();
    }

    private ListF<BlockModel> getBlockModels(long uidL, ListF<ListF<DiskSearchFileInfo>> photos) {
        return photos.zipWithIndex().map((block, index) -> {
            String imageUrl = hackatonMpfsClient.getPreviews(uidL, block).shuffle().firstO().getOrElse("");
            return new BlockModel(imageUrl, "", "", index);
        });
    }

    private ResponseProto.TScenarioRunResponse checkResponse(long uidL, ListF<String> words, RequestProto.TScenarioRunRequest request) {
        if (selectedSong.title.toLowerCase().equals(request.getInput().getText().getUtterance().toLowerCase())) {
            ResponseProto.TScenarioRunResponse result = ResponseProto.TScenarioRunResponse
                    .newBuilder()
                    .setResponseBody(ResponseProto.TScenarioResponseBody
                            .newBuilder()
                            .setLayout(ResponseProto.TLayout
                                    .newBuilder()
                                    .addAllCards(Cf.list(ResponseProto.TLayout.TCard.newBuilder().setText(selectedSong.title).build()).plus(selectedSong.lines.map(line -> ResponseProto.TLayout.TCard.newBuilder().setText(line).build())))
                                    .setOutputSpeech("Ты угадал! Это песня " + selectedSong.title)
                            ))
                    .build();
            selectedSong = null;
            return result;
        } else {
            ListF<Song> variants = Random2.R.randomElements(SONGS, 4).plus(selectedSong).unique().shuffle();

            ListF<ListF<DiskSearchFileInfo>> photos = Cf.arrayList();
            for(String line: selectedSong.getLines().drop(4).take(4)) {
                photos.add(getPhotos(uidL, line));
            }

            ListF<BlockModel> blockModels = getBlockModels(uidL, photos);
            String divJson = templateProcessor.processTemplate(Cf.map("blocks", blockModels));
            logger.info("div json: {}", divJson);

            return ResponseProto.TScenarioRunResponse
                    .newBuilder()
                    .setResponseBody(ResponseProto.TScenarioResponseBody
                            .newBuilder()
                            .setLayout(ResponseProto.TLayout
                                    .newBuilder()
                                    .addCards(ResponseProto.TLayout.TCard
                                            .newBuilder()
//                                             .setDivCard(divJson) see: MEGAMIND-378
                                    )
                                    .addCards(ResponseProto.TLayout.TCard
                                            .newBuilder()
                                            .setTextWithButtons(ResponseProto.TLayout.TTextWithButtons
                                                    .newBuilder()
                                                    .setText("Держи еще подсказку")
                                                    .addAllButtons(variants.map(s -> ResponseProto.TLayout.TButton
                                                            .newBuilder()
                                                            .setTitle(s.title)
                                                            //.addDirectives(DirectivesProto.TDirective
                                                            //        .newBuilder()
                                                            //        .setTypeTextDirective(DirectivesProto.TTypeTextDirective.newBuilder()
                                                            //                .setName(s.title)
                                                            //                .setText(s.title)
                                                            //                .build())
                                                            //)
                                                    .build()
                                                    ))
                                            )
                                    )
                                    .setOutputSpeech("Попробуй еще")
                            ))
                    .build();
        }
    }

    @Override
    public boolean isActive() {
        return selectedSong != null;
    }

    @Override
    public void forceDeactivate() {
        selectedSong = null;
    }

    public ListF<DiskSearchFileInfo> getPhotos(long uid, String word) {
        return searchCache.getAllPhotos(uid)
            .filter(p -> p.getBeautiful2().isPresent() && p.getBeautiful2().get() > beautyEdge.get())
            .take(10000)
            .zipWith((f1) -> getWordCoef(f1, word))
            .sortedBy2Desc()
            .get1()
            .take(10)
            .shuffle();
    }

    public long getWordCoef(DiskSearchFileInfo info, String word) {
        return cvi2tProcessor.dotProduct(info.geti2tVector(), word);
    }

    @Data
    public static class BlockModel {
        public final String imageUrl;
        public final String title;
        public final String subtitle;
        public final int index;
    }

    @Data
    public static class Song {
        public ListF<String> lines;
        public String title;
    }
}
