package ru.yandex.chemodan.office.adapter2;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.Duration;
import java.util.Arrays;
import java.util.Date;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Optional;
import java.util.ResourceBundle;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.tf.se.adapter.AdapterMessageException;
import com.tf.se.adapter.impl.AbstractAdapterG3;
import com.tf.se.adapter.impl.TFSIReleaseCallbackWithInputStream;
import com.tf.se.adapter.vo.TFSIFileVO;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.LaxRedirectStrategy;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.Asserts;
import org.apache.http.util.EntityUtils;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author tolmalev
 * @author vpronto
 */
 public class Hancom020DiskAdapter extends AbstractAdapterG3 {

    private static final long serialVersionUID = -64530143845684823L;
    private static final Logger logger = LoggerFactory.getLogger(Hancom020DiskAdapter.class);

    private static final String mpfsHost = "http://mpfs.disk.yandex.net";
    private static final String blackBox = "http://blackbox.yandex.net";
    private static final String downloadUrl = "downloader.disk.yandex.ua";
    private static final String resourceName = "ya_hancom_bundle";
    private static final int defaultConnectTimeout = 100;
    private static final int defaultSocketTimeout = (int) Duration.ofSeconds(10).toMillis();

    private static final int uploadConnectTimeout = 1000;
    private static final int uploadSocketTimeout = (int) Duration.ofSeconds(60).toMillis();

    private static final HttpClient defaultHttpClient = buildHttpClient(defaultConnectTimeout, defaultSocketTimeout);
    private static final HttpClient putHttpClient = buildHttpClient(uploadConnectTimeout, uploadSocketTimeout);

    private volatile String fid;
    private volatile String ownerUid;
    private volatile String uid;

    private volatile String fileName = "";
    private volatile String userName = "";

    private volatile Locale locale;

    private volatile String lockOid;

    private static final ResponseHandler<JsonNode> readJsonNodeHandler = response -> {
        if (response.getStatusLine().getStatusCode() != 200) {
            String ent = EntityUtils.toString(response.getEntity());
            logger.info("Status code from mpfs: {}, {}", response.getStatusLine().getStatusCode(), ent);
            throw new RuntimeException("Status code:" + response.getStatusLine());
        }

        ObjectMapper mapper = new ObjectMapper();
        return mapper.readTree(response.getEntity().getContent());
    };

    @Override
    public boolean start(HttpServletRequest request, HttpServletResponse response, String s)
            throws AdapterMessageException
    {
        setDebug();
        logger.info("Start adapter");
        initLocale(request);

        try {
            fid = request.getParameter("fid");
            ownerUid = request.getParameter("owner_uid");
            Cookie[] cookies = request.getCookies();
            Asserts.notNull(cookies, "Cookies are empty");
            Optional<String> sessionId = findCookie(cookies, "Session_id");
            Optional<String> sslsessionId = findCookie(cookies, "sessionid2");
            if (!sessionId.isPresent() && !sslsessionId.isPresent()) {
                throw new IllegalArgumentException("sessionId and sslsessionId are empty");
            }
            URI uri = buildBlackBoxUri(sessionId, sslsessionId, request.getRemoteAddr(), request.getServerName());
            fetchAuthInfo(uri);
            Asserts.notNull(uid, "uid");
            Asserts.notNull(fid, "fid");
            Asserts.notNull(ownerUid, "ownerUid");
            logger.info("Started for fid: {}, uid: {}, owner_uid: {}, userName: {}", fid, uid, ownerUid, userName);
        } catch (Exception e) {
            logger.warn("Failed to auth user {}", e);
            throw new AdapterMessageException(getLocaleMessage("auth_error", "Ошибка авторизации"));
        }
        return true;
    }

    private void fetchAuthInfo(URI uri) {
        JsonNode jsonNode = executeAndGetJson(uri);
        Asserts.notNull(jsonNode, "Auth error, jsonNode is null, resp: " + jsonNode);
        JsonNode uidNode = jsonNode.get("uid");
        Asserts.notNull(uidNode, "Auth error, uidNode is null, resp: " + jsonNode);
        JsonNode value = uidNode.get("value");
        Asserts.notNull(value, "Auth error, value is null, resp: " + jsonNode);
        uid = value.getTextValue();
        // not necessary fields
        JsonNode displayName = jsonNode.get("display_name");
        if (displayName != null) {
            JsonNode name = displayName.get("name");
            if (name != null) {
                userName = name.getTextValue();
            }
        }
    }

    private Optional<String> findCookie(Cookie[] cookies, String cookie) {
        return Arrays.stream(cookies).filter(c -> c.getName().equals(cookie)).map(Cookie::getValue).findFirst();
    }

    @Override
    public TFSIFileVO info(String path, String connid) throws AdapterMessageException {
        try {
            logger.info("Getting info from {}, connid: {}, fid: {}", path, connid, fid);
            JsonNode json = getFileInfoJson();

            TFSIFileVO info = new TFSIFileVO();
            info.setName(json.get("name").getTextValue());
            fileName = info.getName();
            info.setType("dir".equals(json.get("type").getTextValue()) ? TYPE_FOLDER : TYPE_FILE);
            info.setFid(fid);
            info.setModified(new Date().getTime());
            info.setCreated(json.get("ctime").asInt());
            info.setModifier(userName);
            info.setRead(true);
            info.setWrite(true);
            logger.info("Got info about {} {}", fileName, fid);
            lock(null, null);
            return info;
        } catch (Exception e) {
            logger.error("can't get info: for {}", path, e);
            throw e;
        }
    }

    @Override
    public TFSIReleaseCallbackWithInputStream get(String path) throws AdapterMessageException {
        logger.info("Getting content from {}", path);
        try {
            lock(null, null); // requesting lock again if null
            Asserts.notNull(lockOid, "Lock is null for" + fid);
            JsonNode json = getFileInfoJson();
            String fileUrl = json.get("meta").get("file_url").getTextValue();

            fileUrl = updateDownloadUrl(fileUrl);

            HttpGet request = new HttpGet(fileUrl);
            HttpResponse response = defaultHttpClient.execute(request);
            checkForErrors(response, fileUrl);
            InputStream content = response.getEntity().getContent();
            return new TFSIReleaseCallbackWithInputStream(content) {
                @Override
                public void release() {
                    try {
                        content.close();
                    } catch (IOException e) {
                    }
                }
            };

        } catch (Exception e) {
            logger.error("Can't get stream", e);
            throw new AdapterMessageException(getLocaleMessage("download_error", "Ошибка загрузки файла"));
        }
    }

    private String updateDownloadUrl(String fileUrl) throws MalformedURLException {
        logger.info("File content URL: {}", fileUrl);
        Asserts.notNull(fileUrl, "File content URL is null");

        URL oldUrl = new URL(fileUrl);
        URL newUrl = new URL(oldUrl.getProtocol(),
                downloadUrl,
                oldUrl.getPort(),
                oldUrl.getFile());
        logger.info("Updated URL: {}", newUrl);

        return newUrl.toString();
    }

    private void checkForErrors(HttpResponse response, String fileUrl) throws IOException {
        int statusCode = response.getStatusLine().getStatusCode();
        logger.info("Get {} {} for {}, {}", statusCode, fileUrl, fid, fileName);
        if (statusCode >= 400) {
            String ent = EntityUtils.toString(response.getEntity());
            logger.warn("Get {} {} for {}", response.getStatusLine(), ent, fileUrl);
            throw new RuntimeException("Status code for:" + fileUrl + ", " + response.getStatusLine());
        } else if (statusCode == 302) {
            logger.info("Get redirect {} {} for {}", response.getStatusLine(), response.getAllHeaders(), fileUrl);
        }
    }

    @Override
    public String put(String path, InputStream inputStream, long l) throws AdapterMessageException {
        logger.info("Invoking put fid: {}, {}, {}", fid, path, l);
        lock(null, null);
        String lo = lockOid;
        URI uri = buildMpfsUriWithOid("/json/office_hancom_store", lo);
        return store(uri, lo, inputStream);
    }

    private String store(URI uri, String lo, InputStream inputStream) throws AdapterMessageException {
        logger.info("Call store {}", uri);
        try {
            String uploadUrl = executeAndGetJson(uri).get("upload_url").asText();
            logger.info("Upload URL: {}", uploadUrl);

            HttpPut request = new HttpPut(uploadUrl);
            request.setEntity(new InputStreamEntity(inputStream));

            putHttpClient.execute(request, (ResponseHandler<Void>) response -> {
                int code = response.getStatusLine().getStatusCode();
                logger.info("Store status code {}, for {} {}", code, lo, fid);
                if (code == 200 || code == 201) {
                    return null;
                }
                throw new RuntimeException("Bad status code: " + code);
            });

            logger.info("Store finished OK {}", uri);

            return getFileInfoJson().get("meta").get("file_id").asText();
        } catch (Exception e) {
            logger.error("Can't put stream by {}", uri, e);
            throw new RuntimeException("Failed to get file info from mpfs " + uri, e);
        } finally {
            unlock(null, null);
        }
    }

    @Override
    public boolean lock(String path, String s1) throws AdapterMessageException {
        logger.info("About to lock, passed params {}, {}", path, s1);
        if (lockOid != null) {
            return true;
        }
        lockOid = lockTable.get(fid);
        if (lockOid != null) {
            return true;
        }
        logger.info("About to get lock from mpfs for {} {}", fid, fileName);
        URI uri = buildMpfsUri("/json/office_hancom_lock");
        try {
            JsonNode jsonNode = executeAndGetJson(uri);
            JsonNode oidNode = jsonNode.get("oid");
            if (oidNode == null) {
                logger.warn("Can't lock {}, from {}", jsonNode.toString(), uri);
                return false;
            }
            String oid = oidNode.getTextValue();
            if (oid == null) {
                logger.warn("fetched lock is null {}, from {}", jsonNode.toString(), uri);
                return false;
            }
            lockOid = oid;
            logger.info("Fetched lock {}", oid);
            tryLock(fid, lockOid);
        } catch (Exception e) {
            logger.error("Can't get lock {}", uri, e);
            return false;
        }
        return true;
    }

    @Override
    public boolean unlock(String s, String s1) throws AdapterMessageException {
        logger.info("Unlock {}, {}", s, s1);
        String lo = lockOid;
        logger.info("About to unlock lock: {}, fid: {}", lo, fid);
        if (lo == null) {
            logger.warn("No lock exists. Can't unlock");
            return true;
        }

        URI uri = buildMpfsUriWithOid("/json/office_hancom_unlock", lo);
        try {
            executeAndGetJson(uri);
            logger.info("Unlocked successful {}", lo);
            releaseLock(fid, lockOid);
        } catch (Exception e) {
            logger.warn("failed to unlock file {} {}", uri, e);
        }
        return true;
    }

    private static JsonNode executeAndGetJson(URI uri) {
        try {
            HttpGet request = new HttpGet(uri);
            JsonNode jsonNode = defaultHttpClient.execute(request, readJsonNodeHandler);
            logger.info("Result for {}: {}", jsonNode, request);
            return jsonNode;
        } catch (Exception e) {
            logger.error("Failed {}, {}", uri, e);
            throw new RuntimeException("Failed ", e);
        }
    }

    private JsonNode getFileInfoJson() {
        URI uri = buildMpfsUri("/json/info_by_file_id");
        logger.info("About to get info {}", uri);
        try {
            return executeAndGetJson(uri);
        } catch (Exception e) {
            throw new RuntimeException("Failed to get file info from mpfs:" + uri, e);
        }
    }

    private URI buildMpfsUri(String path) {
        Asserts.notNull(uid, "uid");
        Asserts.notNull(ownerUid, "ownerUid");
        Asserts.notNull(fid, "file_id");
        try {
            URI uri = new URIBuilder(mpfsHost)
                    .setPath(path)
                    .addParameter("uid", uid)
                    .addParameter("owner_uid", ownerUid)
                    .addParameter("file_id", fid)
                    .addParameter("meta", "")
                    .build();

            logger.info("Result uri {}", uri);
            return uri;
        } catch (Exception e) {
            throw new RuntimeException("Failed to build URI for: " + path, e);
        }
    }

    private URI buildMpfsUriWithOid(String path, String lo) {
        try {
            Asserts.notNull(lo, "lockOid is null for: " + fid);
            URI uri = new URIBuilder(mpfsHost)
                    .setPath(path)
                    .addParameter("uid", uid)
                    .addParameter("owner_uid", ownerUid)
                    .addParameter("oid", lo)
                    .addParameter("fid", fid)
                    .addParameter("filename", fileName)
                    .build();
            logger.info("Result uri {}", uri);
            return uri;
        } catch (Exception e) {
            throw new RuntimeException("Failed to build URI", e);
        }
    }

    private URI buildBlackBoxUri(Optional<String> sessionId, Optional<String> sslsessionid, String userIp, String host)
            throws URISyntaxException
    {
        URIBuilder uriBuilder = new URIBuilder(blackBox)
                .setPath("/blackbox")
                .addParameter("method", "sessionid")
                .addParameter("host", host)
                .addParameter("format", "json")
                .addParameter("userip", userIp);
        sessionId.ifPresent(s -> uriBuilder.addParameter("sessionid", s));
        sslsessionid.ifPresent(s -> uriBuilder.addParameter("sslsessionid", s));
        URI uri = uriBuilder.build();
        logger.info("Result uri: {}", uri);
        return uri;
    }

    private static HttpClient buildHttpClient(int connectTimeout, int socketTimeout) {

        RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();
        requestConfigBuilder.setStaleConnectionCheckEnabled(true);
        requestConfigBuilder.setCookieSpec(CookieSpecs.BROWSER_COMPATIBILITY);

        requestConfigBuilder.setConnectTimeout(connectTimeout);
        requestConfigBuilder.setConnectionRequestTimeout(connectTimeout);
        requestConfigBuilder.setSocketTimeout(socketTimeout);

        RegistryBuilder<ConnectionSocketFactory> registryBuilder = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", SSLConnectionSocketFactory.getSocketFactory());

        PoolingHttpClientConnectionManager connectionManager =
                new PoolingHttpClientConnectionManager(registryBuilder.build());

        connectionManager.setDefaultMaxPerRoute(100);
        connectionManager.setMaxTotal(100);

        DefaultHttpRequestRetryHandler retryHandler = new DefaultHttpRequestRetryHandler(3, true);
        return HttpClientBuilder
                .create()
                .setRetryHandler(retryHandler)
                .setConnectionManager(connectionManager)
                .setRedirectStrategy(new LaxRedirectStrategy())
                .setDefaultRequestConfig(requestConfigBuilder.build())
                .build();
    }

    @Override
    public boolean stop() throws AdapterMessageException {
        logger.info("Invoking stop");
        unlock(null, null);
        fid = null;
        uid = null;
        ownerUid = null;
        return true;
    }

    @Override
    public Object list(String s) throws AdapterMessageException {
        logger.info("Invoking list {}", s);
        return null;
    }

    @Override
    public boolean mkdir(String s, String s1) throws AdapterMessageException {
        return true;
    }

    @Override
    public boolean rename(String s, String s1) throws AdapterMessageException {
        return true;
    }

    @Override
    public boolean delete(String s) throws AdapterMessageException {
        return true;
    }

    @Override
    public int level() throws AdapterMessageException {
        return LEVEL_4;
    }

    @Override
    public String author() throws AdapterMessageException {
        return "Yandex.Disk Team";
    }

    private void initLocale(HttpServletRequest request) {
        try {
            Enumeration<Locale> locales = request.getLocales();
            if (locales.hasMoreElements()) {
                locale = locales.nextElement();
            }
        } catch (Exception e) {
            logger.error("error during init locale");
        }
    }

    private String getLocaleMessage(String key, String defaultValue) {
        try {
            ResourceBundle resource = getResource(resourceName, locale);
            return resource.getString(key);
        } catch (Exception e) {
            logger.info("can't get text for {} {}", key, e.getMessage());
        }
        return defaultValue;
    }

}
