package ru.yandex.chemodan.app.lentaloader;

import java.net.URI;

import org.apache.http.client.HttpClient;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.DataApiBaseContextConfiguration;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaUpdateOrDeleteNonExistentRecordException;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.retry.RetryClauses;
import ru.yandex.chemodan.app.lentaloader.blocks.AlbumBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.BlockItemManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ResourceBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockManager;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaConfigurationManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManagerImpl;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaNotificationManager;
import ru.yandex.chemodan.app.lentaloader.lenta.UserBlockGroupKeyAndType;
import ru.yandex.chemodan.app.lentaloader.lenta.limit.LentaLimitManager;
import ru.yandex.chemodan.app.lentaloader.lenta.limit.LentaLimitManagerImpl;
import ru.yandex.chemodan.app.lentaloader.lenta.limit.UserContentPath;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoReminderTaskManager;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.bazinga.ChemodanBazingaClientContextConfiguration;
import ru.yandex.chemodan.blackbox.ProdBlackboxContextConfiguration;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;
import ru.yandex.chemodan.ratelimiter.RateLimiterHttpClientContextConfiguration;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.retry.RetryProxy;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;

/**
 * @author dbrylev
 */
@Configuration
@Import({
        DataApiBaseContextConfiguration.class,
        ChemodanBazingaClientContextConfiguration.class,
        RateLimiterHttpClientContextConfiguration.class,
        ProdBlackboxContextConfiguration.class
})
public class LentaBlocksContextConfiguration {

    private final static Logger logger = LoggerFactory.getLogger(LentaBlocksContextConfiguration.class);

    @Value("${lenta_all.datasource.retry.count}")
    private int retryCount;

    private <T> T retryProxy(T impl, Class<T> iface, Function1B<Throwable> specificRetryClauses) {
        return RetryProxy.proxy(impl, iface, retryCount,
                RetryClauses.dataApiAccess.orF(specificRetryClauses),
                RetryProxy.pushToNdcInterceptor());
    }

    @Bean
    public LentaManager lentaManager(
            DataApiManager dataApiManager, PgBazingaStorage bazingaStorage, ExperimentsManager experimentsManager,
            @Value("${lenta_all.cache.found-blocks.ttl}") Duration cacheTtl,
            @Value("${lenta_all.cache.found-blocks.size}") int cacheSize,
            @Value("${lenta_all.cache.found-blocks.concurrency}") int cacheConcurrency,
            @Value("${lenta_all.cache.db-handles.ttl}") Duration handlesCacheTtl,
            @Value("${lenta_all.cache.db-handles.size}") int handlesCacheSize,
            @Value("${lenta_all.cache.db-handles.concurrency}") int handlesCacheConcurrency)
    {
        MeteredCache<UserBlockGroupKeyAndType, LentaBlockRecord> blocksCache =
                new MeteredCache<>(cacheSize, cacheConcurrency, cacheTtl);

        MeteredCache<DataApiUserId, DatabaseHandle> handlesCache =
                new MeteredCache<>(handlesCacheSize, handlesCacheConcurrency, handlesCacheTtl);

        MonicaStaticRegistry.register(blocksCache, "found-blocks");
        MonicaStaticRegistry.register(handlesCache, "blocks-db-handles");

        LentaManagerImpl impl = new LentaManagerImpl(dataApiManager, bazingaStorage, experimentsManager, blocksCache, handlesCache);

        return retryCount > 0
                ? retryProxy(impl, LentaManager.class, t -> t instanceof DeltaUpdateOrDeleteNonExistentRecordException)
                : impl;
    }

    @Bean
    public LentaLimitManager lentaLimitManager(
            LentaManager lentaManager, DataApiManager dataApiManager,
            MpfsClient mpfsClient, PgBazingaTaskManager bazingaTaskManager,
            @Value("${lenta_all.cache.blocked-folders.ttl}") Duration cacheTtl,
            @Value("${lenta_all.cache.blocked-folders.size}") int cacheSize,
            @Value("${lenta_all.cache.blocked-folders.concurrency}") int cacheConcurrency)
    {
        MeteredCache<UserContentPath, String> foldersCache = new MeteredCache<>(cacheSize, cacheConcurrency, cacheTtl);

        MonicaStaticRegistry.register(foldersCache, "blocked-folders");

        LentaLimitManagerImpl impl = new LentaLimitManagerImpl(
                lentaManager, dataApiManager, bazingaTaskManager, mpfsClient, foldersCache);

        return retryCount > 0 ? retryProxy(impl, LentaLimitManager.class, Function1B.falseF()) : impl;
    }

    @Bean
    public LentaNotificationManager lentaNotificationManager(
            PgBazingaStorage bazingaStorage,
            UserTimezoneHelper userTimezoneHelper,
            ExperimentsManager experimentsManager,
            @Value("${lenta-loader.notification.create_without_delay}") boolean createWithoutDelay,
            @Value("${lenta-loader.notification.enabled}") boolean notificationEnabled)
    {
        return new LentaNotificationManager(
                bazingaStorage, userTimezoneHelper, experimentsManager,
                createWithoutDelay, notificationEnabled);
    }

    @Bean
    public UserTimezoneHelper userTimezoneHelper(Blackbox2 blackbox2) {
        return new UserTimezoneHelper(blackbox2);
    }

    @Bean
    public BlockItemManager blockItemManager(DataApiManager dataApiManager) {
        return new BlockItemManager(dataApiManager);
    }

    @Bean
    public AlbumBlockManager albumBlockManager(LentaManager lentaManager) {
        return new AlbumBlockManager(lentaManager);
    }

    @Bean
    public ContentBlockManager contentBlockManager(
            LentaManager lentaManager, LentaLimitManager lentaLimitManager,
            LentaNotificationManager lentaNotificationManager, MpfsClient mpfsClient,
            @Value("${lenta-loader.notification.create_on_every_autoupload:-false}") boolean createOnEveryAutoupload)
    {
        if (createOnEveryAutoupload) {
            logger.warn("Enabled create_on_every_autoupload in ContentBlockManager, delay is disabled!");
        }
        return new ContentBlockManager(
                lentaManager, lentaLimitManager, lentaNotificationManager,
                mpfsClient, createOnEveryAutoupload);
    }

    @Bean
    public ResourceBlockManager publicResourceBlockManager(
            LentaManager lentaManager,
            HttpClient client,
            MpfsClient mpfsClient,
            @Value("${ratelimiter.uri}") URI ratelimiterUri,
            @Value("${lenta-public-resource.ratelimiter.group}") String ratelimiterGroup)
    {
        RateLimiterClient rateLimiterClient = new RateLimiterClient(client, ratelimiterUri, ratelimiterGroup, false);

        return new ResourceBlockManager(lentaManager, rateLimiterClient, mpfsClient);
    }

    @Bean
    public FolderCreationBlockManager folderCreationBlockManager(
            LentaManager lentaManager,
            MpfsClient mpfsClient,
            HttpClient client,
            @Value("${ratelimiter.uri}") URI ratelimiterUri,
            @Value("${lenta-folder-creation.ratelimiter.group}") String ratelimiterGroup)
    {
        RateLimiterClient rateLimiterClient = new RateLimiterClient(client, ratelimiterUri, ratelimiterGroup, false);

        return new FolderCreationBlockManager(lentaManager, mpfsClient, rateLimiterClient);
    }

    @Bean
    public SharedFolderBlockManager sharedFolderBlockManager(LentaManager lentaManager) {
        return new SharedFolderBlockManager(lentaManager);
    }

    @Bean
    public PhotoReminderTaskManager photoReminderTaskManager(BazingaStorage bazingaStorage, ExperimentsManager experimentsManager,
            LentaManager lentaManager, UserTimezoneHelper userTimezoneHelper) {
        return new PhotoReminderTaskManager(bazingaStorage, experimentsManager, lentaManager, userTimezoneHelper);
    }

    @Bean
    public CoolLentaConfigurationManager coolLentaConfigurationManager() {
        return new CoolLentaConfigurationManager();
    }
}
