package ru.yandex.chemodan.app.lentaloader;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.utils.BleedingEdgeUtils;
import ru.yandex.chemodan.app.lentaloader.blocks.AlbumBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ModifiedResource;
import ru.yandex.chemodan.app.lentaloader.blocks.OwnedOrPublishedResource;
import ru.yandex.chemodan.app.lentaloader.blocks.ResourceBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderAction;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockManager;
import ru.yandex.chemodan.app.lentaloader.log.ActionEventData;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.ActionSourceHolder;
import ru.yandex.chemodan.app.lentaloader.log.LentaBlockEvent;
import ru.yandex.chemodan.app.lentaloader.log.LentaBlockEventsCache;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.CompoundResourceType;
import ru.yandex.chemodan.eventlog.events.EventType;
import ru.yandex.chemodan.eventlog.events.MediaType;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.eventlog.events.ResourceType;
import ru.yandex.chemodan.eventlog.events.album.CreateAlbumEvent;
import ru.yandex.chemodan.eventlog.events.comment.AbstractCommentEvent;
import ru.yandex.chemodan.eventlog.events.comment.LikableType;
import ru.yandex.chemodan.eventlog.events.comment.LikeDislikeEvent;
import ru.yandex.chemodan.eventlog.events.eventlog.CallbackEventLogListener;
import ru.yandex.chemodan.eventlog.events.fs.DeleteSubdirFsEvent;
import ru.yandex.chemodan.eventlog.events.fs.FsEvent;
import ru.yandex.chemodan.eventlog.events.fs.FsEventType;
import ru.yandex.chemodan.eventlog.events.misc.PublicVisitEvent;
import ru.yandex.chemodan.eventlog.events.sharing.ShareData;
import ru.yandex.chemodan.eventlog.events.sharing.ShareEvent;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author Lev Tolmachev (tolmalev)
 */
public class LentaEventsLogListener extends CallbackEventLogListener {
    private static final Logger logger = LoggerFactory.getLogger(LentaEventsLogListener.class);

    private final static ListF<String> DOWNLOADS_PATH_I18N =
            Cf.list("/disk/Загрузки", "/disk/Downloads", "/disk/Завантаження");

    //this will be removed after fix in mpfs
    private final static SetF<String> ROOT_AREAS = Cf.hashSet("disk", "photounlim");

    private final MpfsClient mpfsClient;
    private final AlbumBlockManager albumBlockManager;
    private final ContentBlockManager contentBlockManager;
    private final ResourceBlockManager resourceBlockManager;
    private final FolderCreationBlockManager folderCreationBlockManager;
    private final SharedFolderBlockManager sharedFolderBlockManager;
    private final BleedingEdge bleedingEdge;

    public LentaEventsLogListener(
            MpfsClient mpfsClient,
            AlbumBlockManager albumBlockManager,
            ContentBlockManager contentBlockManager,
            ResourceBlockManager resourceBlockManager,
            FolderCreationBlockManager folderCreationBlockManager,
            SharedFolderBlockManager sharedFolderBlockManager,
            BleedingEdge bleedingEdge)
    {
        this.mpfsClient = mpfsClient;
        this.albumBlockManager = albumBlockManager;
        this.contentBlockManager = contentBlockManager;
        this.resourceBlockManager = resourceBlockManager;
        this.folderCreationBlockManager = folderCreationBlockManager;
        this.sharedFolderBlockManager = sharedFolderBlockManager;
        this.bleedingEdge = bleedingEdge;

        this.registerProcessor(FsEvent.class, this::processFsEvent);
        this.registerProcessor(DeleteSubdirFsEvent.class, this::processDeleteSubdirEvent);
        this.registerProcessor(AbstractCommentEvent.class, this::processCommentEvent);
        this.registerProcessor(PublicVisitEvent.class, this::processPublicVisitEvent);
        this.registerProcessor(ShareEvent.class, this::processShareEvent);
        this.registerProcessor(CreateAlbumEvent.class, this::processAlbumCreateEvent);
    }

    private void addFakeTimeout() {
        if (DynamicVars.fakeProcessingTime.get() > 0) {
            ThreadUtils.sleep(DynamicVars.fakeProcessingTime.get());
        }
    }

    private boolean isSavedFromPublic(DataApiUserId uid, FsEvent event) {
        // XXX hack until we don't have saved_from_public as a specific EventType

        FsEventType type = event.getFsEventType();
        Option<ResourceLocation> target = event.resourceChange.target;

        if (type == FsEventType.FS_COPY && target.isPresent()) {

            String path = target.get().address.path.getValue();

            Option<MpfsUid> oldOwner = event.resourceChange.source.filterMap(l -> l.folderId).map(id -> id.owner);

            if (!oldOwner.isPresent() || oldOwner.get().isSpecial()) {
                return false;
            }

            boolean isNewOwner = !uid.toPassportUidOrZero().toPassportUidO().isSome(oldOwner.get().getUid());

            if (DOWNLOADS_PATH_I18N.exists(path::startsWith) && isNewOwner) {
                return true;
            }
        }
        return false;
    }

    void processFsEvent(DataApiUserId uid, FsEvent event) {
        if (!BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge).test(uid)) {
            addFakeTimeout();
            return;
        }
        logger.info("About to process {}", event);
        Option<ResourceLocation> source = Option.empty();
        Option<ResourceLocation> target = Option.empty();

        switch (event.getFsEventType()) {
            case FS_STORE:
            case FS_HARDLINK_COPY:
            case FS_COPY:
            case FS_TRASH_RESTORE:
                target = event.resourceChange.target;
                break;
            case FS_RM:
            case FS_TRASH_APPEND:
                source = event.resourceChange.source;
                break;
            case FS_MOVE:
                source = event.resourceChange.parentEquals() ? Option.empty() : event.resourceChange.source;
                target = event.resourceChange.target;
                break;
        }

        boolean isSavedFromPublic = isSavedFromPublic(uid, event);

        Resource resource = event.resourceChange.resource;

        Function2<ResourceLocation, ResourceType, Option<ModifiedResource>> filterResource = (loc, type) -> Option.when(
                loc.folderId.isPresent()
                        && loc.address.path.resourceTypeO.isSome(type)
                        && (ROOT_AREAS.containsF().apply(loc.address.path.service)),
                () -> new ModifiedResource(resource, loc.address, loc.folderId.get(), event.performerUid, resource.owner));

        Function<ResourceLocation, Option<ModifiedResource>> filterFile = filterResource.bind2(ResourceType.FILE);
        Function<ResourceLocation, Option<ModifiedResource>> filterDir = filterResource.bind2(ResourceType.DIRECTORY);

        Tuple2List<String, String> logData = Tuple2List.<String, String>tuple2List()
                .plus(source.map(loc -> Tuple2.tuple("src_path", loc.address.path.getValue())))
                .plus(target.map(loc -> Tuple2.tuple("tgt_path", loc.address.path.getValue())));

        ActionInfo actionInfo = actionInfo(event, logData);

        source.flatMapO(filterFile).forEach(src ->
                contentBlockManager.deleteFile(uid, src.folderId, resource,
                        actionInfo.withReason(ActionReason.FILE_CHANGED_OR_REMOVED)));

        Option<ContentBlockAction> contentBlockAction = isSavedFromPublic
                ? Option.of(ContentBlockAction.SAVE_PUBLIC)
                : ContentBlockAction.byEventType(event.getEventType());

        Option<String> actionArea = event.extra == null && event.extra.typeSubtype == null ? Option.empty() :
                event.extra.typeSubtype.subtype;
        logger.info("contentBlockAction: {}, area: {}", contentBlockAction, actionArea);
        target.flatMapO(filterFile).forEach(tgt ->
                contentBlockManager.createOrUpdateFile(uid, tgt,
                        contentBlockAction, event.resourceChange.source, actionInfo, actionArea));

        if (!target.isPresent()) {
            source.flatMapO(filterDir).forEach(tgt -> {
                ReasonedAction reason = actionInfo.withReason(ActionReason.FOLDER_DISAPPEARED);

                contentBlockManager.deleteFolder(uid, tgt.folderId, reason);
                folderCreationBlockManager.delete(uid, tgt.folderId, reason);
            });
        }

        if (event.getFsEventType() == FsEventType.FS_SET_PUBLIC) {
            event.resourceChange.target.flatMapO(filterFile).forEach(tgt ->
                    contentBlockManager.deleteEmptyOrSingleFileBlock(
                            uid, tgt, actionInfo.withReason(ActionReason.PUBLIC_RESOURCE_DEDUPLICATION)));

            resourceBlockManager.handleChanged(
                    uid, OwnedOrPublishedResource.owned(resource), event.performerUid,
                    event.getEventType(), true, actionInfo);
        }

        if (event.getFsEventType() == FsEventType.FS_SET_PRIVATE || source.isPresent() && !target.isPresent()) {
            resourceBlockManager.handleDeleted(
                    uid, OwnedOrPublishedResource.owned(resource).idOrUrl.getLeft(),
                    actionInfo.withReason(ActionReason.FILE_CHANGED_OR_REMOVED));
        }

        if (event.getFsEventType() == FsEventType.FS_COPY
                && isSavedFromPublic
                && toDataApiUid(event.performerUid).isSome(uid))
        {
            event.resourceChange.target.flatMapO(filterDir).forEach(tgt -> folderCreationBlockManager
                    .handleCreation(uid, tgt, Option.of(FolderCreationBlockAction.SAVE_PUBLIC), actionInfo));
        }

        if (event.getFsEventType() == FsEventType.FS_MKDIR
                && event.metadata.getPlatform().isSome("web")
                && toDataApiUid(event.performerUid).isSome(uid))
        {
            event.resourceChange.target.flatMapO(filterDir).forEach(tgt -> folderCreationBlockManager
                    .handleCreation(uid, tgt, FolderCreationBlockAction.byEventType(event.getEventType()), actionInfo));
        }
    }

    void processDeleteSubdirEvent(DataApiUserId uid, DeleteSubdirFsEvent event) {
        if (!BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge).test(uid)) {
            addFakeTimeout();
            return;
        }
        ReasonedAction reason = actionInfo(event).withReason(ActionReason.FOLDER_DISAPPEARED);

        contentBlockManager.deleteFolder(uid, event.resourceId, reason);
        folderCreationBlockManager.delete(uid, event.resourceId, reason);
    }

    private static final ListF<EventType> USER_RELATED_COMMENT_EVENTS =
            Cf.list(EventType.COMMENT_DELETE, EventType.COMMENT_LIKE_DELETE, EventType.COMMENT_DISLIKE_DELETE);

    private static final ListF<EventType> ACCEPTED_COMMENT_EVENTS = USER_RELATED_COMMENT_EVENTS.plus(
            Cf.list(EventType.COMMENT_ADD, EventType.COMMENT_LIKE_ADD, EventType.COMMENT_DISLIKE_ADD));

    void processCommentEvent(AbstractCommentEvent event) {
        addFakeTimeout();
        if (!ACCEPTED_COMMENT_EVENTS.containsTs(event.getEventType())
            || Option.of(event).filterByType(LikeDislikeEvent.class).exists(e -> e.commentType != LikableType.ENTITY)
            || !event.isResource())
        {
            return;
        }

        boolean isPublicResource = event.isPublicResource();

        MpfsResourceId resourceId = MpfsResourceId.parse(event.entity.id);
        ActionInfo actionInfo = actionInfo(event);

        Option<MpfsFileInfo> resourceInfo = resourceId.owner.getUidO()
                .flatMapO(u -> mpfsClient.getFileInfoOByFileId(MpfsUser.of(u), resourceId.fileId));

        if (!resourceInfo.isPresent()) {
            return;
        }

        CompoundResourceType resourceType = new CompoundResourceType(
                ResourceType.R.fromValue(resourceInfo.get().type.get()),
                resourceInfo.get().getMeta().getMediaType().map(MediaType::new));

        SetF<DataApiUserId> owners = resourceId.owner.getUidO()
                .flatMapO(owner -> mpfsClient.getShareUidsInGroupO(MpfsUser.of(owner), resourceId.fileId))
                .map(users -> users.getUids().<DataApiUserId>map(DataApiPassportUserId::new))
                .getOrElse(toDataApiUid(resourceId.owner)).unique();

        Option<DataApiUserId> publicPerformer = toDataApiUid(event.performerUid)
                .filterNot(owners::containsTs)
                .filter(BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge));

        Option<OwnedOrPublishedResource> published = resourceInfo.get().getMeta().getShortUrl()
                .map(url -> OwnedOrPublishedResource.published(url, resourceType));

        if (USER_RELATED_COMMENT_EVENTS.containsTs(event.getEventType())) {
            if (publicPerformer.isPresent() && published.isPresent()) {
                resourceBlockManager.handleChanged(
                        publicPerformer.get(), published.get(), event.performerUid,
                        event.getEventType(), isPublicResource, actionInfo);
            }
            return;
        }

        if (isPublicResource || owners.size() > 1) {
            owners.filter(BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge))
                    .forEach(u -> resourceBlockManager.handleChanged(
                            u, OwnedOrPublishedResource.owned(resourceId, resourceType),
                            event.performerUid, event.getEventType(), isPublicResource, actionInfo));
        }
    }

    void processPublicVisitEvent(DataApiUserId uid, PublicVisitEvent event) {
        if (!BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge).test(uid)) {
            addFakeTimeout();
            return;
        }
        if (event.isOwner.isSome(true) || event.isInvitee.isSome(true)) {
            return;
        }

        OwnedOrPublishedResource resource = OwnedOrPublishedResource.published(
                event.shortUrl, event.resourceType.getOrElse(CompoundResourceType.file("unknown")));

        if (resource.isAlbum()) {
            albumBlockManager.createPublic(uid, resource.idOrUrl.getRight(), actionInfo(event));
        } else {
            resourceBlockManager.handleChanged(
                    uid, resource, event.getUid(), event.getEventType(), true, actionInfo(event));
        }
    }

    void processAlbumCreateEvent(DataApiUserId uid, CreateAlbumEvent event) {
        if (!BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge).test(uid)) {
            addFakeTimeout();
            return;
        }
        albumBlockManager.createOwned(uid, event.album.id, actionInfo(event));
    }

    void processShareEvent(DataApiUserId uid, ShareEvent event) {
        if (!BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge).test(uid)) {
            addFakeTimeout();
            return;
        }

        ActionInfo actionInfo = actionInfo(event);
        ShareData share = event.data;

        if (event.isOwnerEvent()) {
            switch (event.type) {
                case CREATE_GROUP:
                    sharedFolderBlockManager.handleGroupCreated(uid, share, actionInfo);
                    break;
                case INVITE_USER:
                case REMOVE_INVITE:
                case CHANGE_INVITE_RIGHTS:
                case CHANGE_RIGHTS:
                case KICK_FROM_GROUP:
                case LEAVE_GROUP:
                case ACTIVATE_INVITE:
                case REJECT_INVITE:
                    sharedFolderBlockManager.handleGroupChanged(
                            uid, share, SharedFolderAction.byEventType(event.type), actionInfo);
                    break;
                case UNSHARE_FOLDER:
                    sharedFolderBlockManager.deleteGroup(uid, share, actionInfo);
            }

        } else if (event.isUserEvent()) {
            switch (event.getEventType()) {
                case SHARE_INVITE_USER:
                    sharedFolderBlockManager.handleInvitation(uid, share, actionInfo);
                    break;
                case SHARE_REMOVE_INVITE:
                case SHARE_KICK_FROM_GROUP:
                    sharedFolderBlockManager.kickInvitation(uid, share, actionInfo);
                    break;
                case SHARE_ACTIVATE_INVITE:
                    sharedFolderBlockManager.acceptInvitation(uid, share, actionInfo);
                    break;
                case SHARE_REJECT_INVITE:
                case SHARE_LEAVE_GROUP:
                    sharedFolderBlockManager.declineInvitation(uid, share, actionInfo);
            }
        }
    }

    private ActionInfo actionInfo(AbstractEvent event) {
        return actionInfo(event, Tuple2List.tuple2List());
    }

    private ActionInfo actionInfo(AbstractEvent event, Tuple2List<String, String> data) {
        return ActionInfo.logEvent(
                new ActionEventData(event.getEventType(), event.metadata.commitTime, data),
                ActionSourceHolder.getOrThrow());
    }

    @Override
    protected Tuple2List<String, Object> getNdcValues(AbstractEvent event) {
        if (YandexCloudRequestIdHolder.getO().isPresent()) {
            return Tuple2List.fromPairs(
                    "history_lag", event.metadata.getReceiveDelay().getStandardSeconds(),
                    "history_hostname", event.metadata.host.getOrElse(""));
        } else {
            return super.getNdcValues(event);
        }
    }

    public ListF<LentaBlockEvent> processLogLineAndStoreResult(String line) {
        return LentaBlockEventsCache.executeAndGetBlocks(() -> processLogLine(line));
    }
}
