package ru.yandex.chemodan.app.lentaloader;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.blocks.AlbumBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ResourceBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManagerImpl;
import ru.yandex.chemodan.app.lentaloader.lenta.UserBlockGroupKeyAndType;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.log.ActionSourceHolder;
import ru.yandex.chemodan.app.lentaloader.log.LentaBlockEvent;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsGroupUids;
import ru.yandex.chemodan.ratelimiter.RateLimit;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.pg.storage.JobSaveResult;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyLong;

/**
 * @author yashunsky
 */
@RunWith(MockitoJUnitRunner.class)
public class LentaEventsLogListenerCacheTest {

    @Mock
    private MpfsClient mpfsClient;
    @Mock
    private AlbumBlockManager albumBlockManager;
    @Mock
    private ContentBlockManager contentManager;
    @Mock
    private FolderCreationBlockManager folderCreationManager;
    @Mock
    private SharedFolderBlockManager sharedFolderManager;
    @Mock
    private BleedingEdge bleedingEdge;
    @Mock
    private RateLimiterClient rateLimiterClient;
    @Mock
    private DataApiManager dataApiManager;
    @Mock
    private PgBazingaStorage bazingaStorage;
    @Mock
    private ExperimentsManager experimentsManager;

    @Test
    public void processLogLine() {
        MpfsGroupUids groupUids = new MpfsGroupUids(Cf.list(158426306, 97963799, 127117)
                .map(uid -> new MpfsGroupUids.User(PassportUid.cons(uid), MpfsGroupUids.Status.APPROVED)));

        MeteredCache<UserBlockGroupKeyAndType, LentaBlockRecord> blocksCache =
                new MeteredCache<>(10000, 16, Duration.standardMinutes(1));

        MeteredCache<DataApiUserId, DatabaseHandle> handlesCache =
                new MeteredCache<>(10000, 16, Duration.standardMinutes(16));

        MonicaStaticRegistry.register(blocksCache, "found-blocks");
        MonicaStaticRegistry.register(blocksCache, "blocks-db-handles");

        LentaManagerImpl lentaManager = new LentaManagerImpl(dataApiManager, bazingaStorage, experimentsManager, blocksCache, handlesCache);

        ResourceBlockManager resourceBlockManager = new ResourceBlockManager(lentaManager, rateLimiterClient, mpfsClient);

        LentaEventsLogListener listener = new LentaEventsLogListener(
                mpfsClient, albumBlockManager,
                contentManager, resourceBlockManager, folderCreationManager,
                sharedFolderManager, bleedingEdge);

        ActionSourceHolder.set(ActionSource.lentaApi());

        Mockito.when(rateLimiterClient.queryLimit(any())).thenReturn(RateLimit.proceedWithoutDelay());

        MpfsFileInfo file = MpfsFileInfo.parse("{\"name\":\"name\",\"type\":\"dir\",\"meta\":{}}");
        Mockito.when(mpfsClient.getFileInfoOByFileId(Mockito.any(), Mockito.any())).thenReturn(Option.of(file));

        Mockito.when(mpfsClient.getShareUidsInGroupO(Mockito.any(), Mockito.any())).thenReturn(Option.of(groupUids));

        Mockito.when(bleedingEdge.isOnBleedingEdge(any(PassportUid.class))).thenReturn(true);

        Mockito.when(dataApiManager.getDatabaseO(any())).thenReturn(Option.empty());
        Mockito.when(dataApiManager.getRecords(any(), any())).thenReturn(Cf.list());
        Mockito.when(experimentsManager.getFlags(anyLong())).thenReturn(Cf.list());

        Database database = new Database(
                new DataApiPassportUserId(groupUids.getUids().first()),
                DatabaseHandle.fromAppDbId(Option.of("app"), "db", "handle"), 0,
                new DatabaseMeta(Instant.now(), Instant.now(), DatabaseMeta.EMPTY_DATABASE_SIZE, 0));

        Mockito.when(dataApiManager.getOrCreateDatabase(any())).thenReturn(database);

        Mockito.when(bazingaStorage.addOnetimeJobX(any(), any()))
                .thenReturn(new JobSaveResult.Created(FullJobId.parse("task/NONE")));

        String line = "tskv\tunixtime=0\t"
                + "event_type=comment-add\tuser_uid=1001197903\t"
                + "entity_type=public_resource\tentity_id=" + groupUids.getUids().first() + ":file_id\t"
                + "comment_id=127\tcomment_author_uid=1001197903";

        ListF<LentaBlockEvent> events = listener.processLogLineAndStoreResult(line);
    }
}
