package ru.yandex.chemodan.app.lentaloader;

import java.util.concurrent.atomic.AtomicBoolean;

import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.mockito.stubbing.Stubber;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.blocks.AlbumBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ResourceBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockManager;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.log.ActionSourceHolder;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.eventlog.events.CompoundResourceType;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.eventlog.events.ResourceType;
import ru.yandex.chemodan.eventlog.events.YandexCloudRequestId;
import ru.yandex.chemodan.eventlog.events.album.Album;
import ru.yandex.chemodan.eventlog.events.album.CreateAlbumEvent;
import ru.yandex.chemodan.eventlog.events.comment.CommentEvent;
import ru.yandex.chemodan.eventlog.events.comment.CommentEventType;
import ru.yandex.chemodan.eventlog.events.comment.CommentRef;
import ru.yandex.chemodan.eventlog.events.comment.EntityRef;
import ru.yandex.chemodan.eventlog.events.comment.ParentCommentRef;
import ru.yandex.chemodan.eventlog.events.fs.DeleteSubdirFsEvent;
import ru.yandex.chemodan.eventlog.events.fs.FsEvent;
import ru.yandex.chemodan.eventlog.events.fs.FsEventResourceChange;
import ru.yandex.chemodan.eventlog.events.fs.FsEventType;
import ru.yandex.chemodan.eventlog.events.misc.PublicVisitEvent;
import ru.yandex.chemodan.eventlog.events.sharing.ShareData;
import ru.yandex.chemodan.eventlog.events.sharing.ShareEvent;
import ru.yandex.chemodan.eventlog.events.sharing.ShareEventType;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsGroupUids;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyBoolean;

/**
 * @author dbrylev
 */
@RunWith(MockitoJUnitRunner.class)
public class LentaEventsLogListenerTest {

    @Mock
    private MpfsClient mpfsClient;
    @Mock
    private AlbumBlockManager albumBlockManager;
    @Mock
    private ContentBlockManager contentManager;
    @Mock
    private ResourceBlockManager resourceBlockManager;
    @Mock
    private FolderCreationBlockManager folderCreationManager;
    @Mock
    private SharedFolderBlockManager sharedFolderManager;

    private BleedingEdge bleedingEdge = new BleedingEdge(null) {
        @Override
        public boolean isOnBleedingEdge(PassportUid uid) {
            return true;
        }
    };

    private LentaEventsLogListener listener;

    @Captor
    private ArgumentCaptor<Option<ContentBlockAction>> contentBlockActionCaptor;
    @Captor
    private ArgumentCaptor<Option<FolderCreationBlockAction>> folderBlockActionCaptor;

    @Before
    public void setup() {
        MockitoAnnotations.initMocks(this);

        listener = new LentaEventsLogListener(
                mpfsClient, albumBlockManager,
                contentManager, resourceBlockManager, folderCreationManager,
                sharedFolderManager, bleedingEdge);

        ActionSourceHolder.set(ActionSource.test());
    }

    @After
    public void teardown() {
        ActionSourceHolder.remove();
    }

    private final DataApiUserId uid = new DataApiPassportUserId(1234567);

    private final MpfsUid mpfsUid = new MpfsUid(uid.toPassportUidOrZero().toUid());

    private final DataApiUserId subscriber1 = new DataApiPassportUserId(11L);
    private final DataApiUserId subscriber2 = new DataApiPassportUserId(12L);

    private final DataApiUserId owner1 = new DataApiPassportUserId(1L);
    private final DataApiUserId owner2 = new DataApiPassportUserId(2L);

    private final DataApiUserId foreignUid = new DataApiPassportUserId(2345678);
    private final MpfsUid mpfsForeignUid = new MpfsUid(foreignUid.toPassportUidOrZero().toUid());

    @Test
    public void fsFile() {
        Cf.list(FsEventType.FS_STORE, FsEventType.FS_HARDLINK_COPY, FsEventType.FS_COPY, FsEventType.FS_TRASH_RESTORE)
                .forEach(type -> listener.processFsEvent(uid, consFsEvent(type, ResourceType.FILE)));

        Mockito.verify(contentManager, Mockito.times(4)).createOrUpdateFile(any(), any(), any(), any(), any(), any());
        verifyNoMoreInteractions();

        listener.processFsEvent(uid, consFsEvent(FsEventType.FS_MOVE, ResourceType.FILE));

        Mockito.verify(contentManager, Mockito.times(1)).createOrUpdateFile(any(), any(), any(), any(), any(), any());
        Mockito.verify(contentManager, Mockito.times(1)).deleteFile(any(), any(), any(), any());

        verifyNoMoreInteractions();

        Cf.list(FsEventType.FS_RM, FsEventType.FS_TRASH_APPEND)
                .forEach(type -> listener.processFsEvent(uid, consFsEvent(type, ResourceType.FILE)));

        Mockito.verify(contentManager, Mockito.times(2)).deleteFile(any(), any(), any(), any());
        Mockito.verify(resourceBlockManager, Mockito.times(2)).handleDeleted(any(), any(), any());

        verifyNoMoreInteractions();

        listener.processFsEvent(uid, consFsEvent(FsEventType.FS_SET_PUBLIC, ResourceType.FILE));

        Mockito.verify(contentManager, Mockito.only()).deleteEmptyOrSingleFileBlock(any(), any(), any());
        verifyResourceRecipients(uid);

        listener.processFsEvent(uid, consFsEvent(FsEventType.FS_SET_PRIVATE, ResourceType.FILE));
        Mockito.verify(resourceBlockManager, Mockito.only()).handleDeleted(any(), any(), any());
    }

    @Test
    public void fsDirectory() {
        Cf.list(FsEventType.FS_RM, FsEventType.FS_TRASH_APPEND)
                .forEach(type -> listener.processFsEvent(uid, consFsEvent(type, ResourceType.DIRECTORY)));

        Mockito.verify(contentManager, Mockito.times(2)).deleteFolder(any(), any(), any());
        Mockito.verify(folderCreationManager, Mockito.times(2)).delete(any(), any(), any());
        Mockito.verify(resourceBlockManager, Mockito.times(2)).handleDeleted(any(), any(), any());

        verifyNoMoreInteractions();

        Cf.x(FsEventType.values())
                .filterNot(Cf.list(
                        FsEventType.FS_MKDIR, FsEventType.FS_RM,
                        FsEventType.FS_TRASH_APPEND, FsEventType.FS_SET_PRIVATE)::containsTs)
                .forEach(type -> listener.processFsEvent(uid, consFsEvent(type, ResourceType.DIRECTORY)));

        Mockito.verify(resourceBlockManager, Mockito.only())
                .handleChanged(any(), any(), any(), any(), anyBoolean(), any());

        verifyNoMoreInteractions();

        listener.processFsEvent(uid, consFsEvent(FsEventType.FS_MKDIR, ResourceType.DIRECTORY));

        Mockito.verify(folderCreationManager, Mockito.only()).handleCreation(any(), any(), any(), any());

        verifyNoMoreInteractions();
    }

    @Test
    public void deleteSubdir() {
        MpfsResourceId resourceId = MpfsResourceId.parse(uid + ":fileid");

        ArgumentCaptor<ReasonedAction> actionCaptor = ArgumentCaptor.forClass(ReasonedAction.class);
        ArgumentCaptor<MpfsResourceId> resourceCaptor = ArgumentCaptor.forClass(MpfsResourceId.class);

        Mockito.doNothing().when(contentManager).deleteFolder(any(), resourceCaptor.capture(), actionCaptor.capture());

        listener.processDeleteSubdirEvent(uid, new DeleteSubdirFsEvent(eventMetadata(uid), resourceId));

        Assert.equals(resourceId, resourceCaptor.getValue());
        Assert.equals(ActionReason.FOLDER_DISAPPEARED, actionCaptor.getValue().reason);
    }

    @Test
    public void publicVisit() {
        Function2V<Boolean, Boolean> process = (owner, invitee) -> listener.processPublicVisitEvent(uid,
                new PublicVisitEvent(eventMetadata(uid), "url", Option.empty(), Option.of(owner), Option.of(invitee)));

        process.apply(true, false);
        process.apply(false, true);

        process.apply(false, false);
        Mockito.verify(resourceBlockManager).handleChanged(any(), any(), any(), any(), anyBoolean(), any());
    }

    @Test
    public void albums() {
        listener.processPublicVisitEvent(uid, new PublicVisitEvent(
                eventMetadata(uid), "url", Option.of(CompoundResourceType.album()), Option.empty(), Option.empty()));

        Mockito.verify(albumBlockManager).createPublic(any(), Mockito.eq("url"), any());
        Mockito.verifyNoMoreInteractions(albumBlockManager);

        listener.processAlbumCreateEvent(uid, new CreateAlbumEvent(eventMetadata(uid), new Album("id", "title")));

        Mockito.verify(albumBlockManager).createOwned(any(), Mockito.eq("id"), any());
        Mockito.verifyNoMoreInteractions(albumBlockManager);
    }

    @Test
    public void shareOwned() {
        Function1V<ShareEventType> process = type -> listener.processShareEvent(uid, consShareEvent(type, mpfsUid));

        process.apply(ShareEventType.CREATE_GROUP);

        Mockito.verify(sharedFolderManager, Mockito.only()).handleGroupCreated(any(), any(), any());
        Mockito.reset(sharedFolderManager);

        process.apply(ShareEventType.INVITE_USER);

        Mockito.verify(sharedFolderManager, Mockito.only()).handleGroupChanged(any(), any(), any(), any());
        Mockito.reset(sharedFolderManager);

        process.apply(ShareEventType.ACTIVATE_INVITE);

        Mockito.verify(sharedFolderManager, Mockito.only()).handleGroupChanged(any(), any(), any(), any());
        Mockito.reset(sharedFolderManager);

        process.apply(ShareEventType.UNSHARE_FOLDER);

        Mockito.verify(sharedFolderManager, Mockito.only()).deleteGroup(any(), any(), any());
        Mockito.reset(sharedFolderManager);

        verifyNoMoreInteractions();
    }

    @Test
    public void shareUser() {
        MpfsUid ownerUid = new MpfsUid(7L);

        Function1V<ShareEventType> process = type -> listener.processShareEvent(uid, consShareEvent(type, ownerUid));

        process.apply(ShareEventType.INVITE_USER);

        Mockito.verify(sharedFolderManager, Mockito.only()).handleInvitation(any(), any(), any());
        Mockito.reset(sharedFolderManager);

        Cf.list(ShareEventType.REMOVE_INVITE, ShareEventType.KICK_FROM_GROUP).forEach(process);

        Mockito.verify(sharedFolderManager, Mockito.times(2)).kickInvitation(any(), any(), any());
        verifyNoMoreInteractions();

        Cf.list(ShareEventType.ACTIVATE_INVITE).forEach(process);

        Mockito.verify(sharedFolderManager, Mockito.only()).acceptInvitation(any(), any(), any());
        Mockito.reset(sharedFolderManager);

        Cf.list(ShareEventType.REJECT_INVITE, ShareEventType.LEAVE_GROUP).forEach(process);

        Mockito.verify(sharedFolderManager, Mockito.times(2)).declineInvitation(any(), any(), any());
        verifyNoMoreInteractions();

        process.apply(ShareEventType.CHANGE_INVITE_RIGHTS);
        verifyNoMoreInteractions();
    }

    @Test
    public void saveFileFromPublic() {
        listener.processFsEvent(uid, consFsEvent(FsEventType.FS_COPY, ResourceType.FILE));
        verifyContentAction(ContentBlockAction.ADDITION);

        listener.processFsEvent(uid, consSaveFromPublicFsEvent(ResourceType.FILE));
        verifyContentAction(ContentBlockAction.SAVE_PUBLIC);
    }

    @Test
    public void saveFolderFromPublic() {
        listener.processFsEvent(uid, consFsEvent(FsEventType.FS_MKDIR, ResourceType.DIRECTORY));
        verifyFolderAction(Option.empty());

        listener.processFsEvent(uid, consSaveFromPublicFsEvent(ResourceType.DIRECTORY));
        verifyFolderAction(Option.of(FolderCreationBlockAction.SAVE_PUBLIC));
    }

    private void verifyContentAction(ContentBlockAction action) {
        Mockito.verify(contentManager, Mockito.only())
                .createOrUpdateFile(any(), any(), contentBlockActionCaptor.capture(), any(), any(), any());
        Assert.isTrue(contentBlockActionCaptor.getValue().isSome(action));
        Mockito.reset(contentManager);
    }

    private void verifyFolderAction(Option<FolderCreationBlockAction> action) {
        Mockito.verify(folderCreationManager, Mockito.only())
                .handleCreation(any(), any(), folderBlockActionCaptor.capture(), any());
        Assert.equals(action, folderBlockActionCaptor.getValue());
        Mockito.reset(folderCreationManager);
    }

    private void mockMpfsShareUids(DataApiUserId... uids) {
        if (uids.length > 0) {
            AtomicBoolean first = new AtomicBoolean(true);

            Stubber respond = Mockito.doReturn(Option.of(new MpfsGroupUids(Cf.x(uids)
                    .map(uid -> new MpfsGroupUids.User(uid.toPassportUidOrZero().toUid(),
                            first.getAndSet(false) ? MpfsGroupUids.Status.OWNER : MpfsGroupUids.Status.APPROVED)))));

            respond.when(mpfsClient).getShareUidsInGroupO(any(), any());

        } else {
            Mockito.when(mpfsClient.getShareUidsInGroupO(any(), any())).thenReturn(Option.empty());
        }
    }

    private void mockMpfsFileInfo() {
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder()
                .resourceId(MpfsResourceId.parse("uid:fileid"))
                .source("disk")
                .fileId("fileid")
                .shortUrl("https://yadi.sk/shorturl")
                .shortNamed("https://yadi.sk/shortnamed")
                .preview("https://preview.ru")
                .build();
        Stubber respond = Mockito.doReturn(Option.of(new MpfsFileInfo("Filename", "file", meta)));

        respond.when(mpfsClient).getFileInfoOByFileId(any(), any());
    }

    private FsEvent consFsEvent(FsEventType type, ResourceType resourceType) {
        ResourceLocation target = new ResourceLocation(
                MpfsAddress.parse(uid + ":/disk/stale/name", resourceType),
                Option.of(MpfsResourceId.parse(uid + ":/disk")));

        ResourceLocation source = new ResourceLocation(
                MpfsAddress.parse(uid + ":/disk/fresh/name", resourceType),
                Option.of(MpfsResourceId.parse(uid + ":/disk")));

        Resource resource = resourceType == ResourceType.FILE
                ? Resource.file("document", "fileid", mpfsUid)
                : Resource.directory("fileid", mpfsUid);

        FsEventResourceChange change = new FsEventResourceChange(
                mpfsUid, Option.of(source), Option.of(target), resource);

        return FsEvent.consForTest(type, eventMetadata(uid), change);
    }

    private FsEvent consSaveFromPublicFsEvent(ResourceType resourceType) {
        ResourceLocation source = new ResourceLocation(
                MpfsAddress.parse(foreignUid + ":/disk/name", resourceType),
                Option.of(MpfsResourceId.parse(foreignUid + ":/disk")));

        ResourceLocation target = new ResourceLocation(
                MpfsAddress.parse(uid + ":/disk/Downloads/name", resourceType),
                Option.of(MpfsResourceId.parse(uid + ":/disk")));

        Resource resource = resourceType == ResourceType.FILE
                ? Resource.file("document", "fileid", mpfsForeignUid)
                : Resource.directory("fileid", mpfsForeignUid);

        FsEventResourceChange change = new FsEventResourceChange(
                mpfsUid, Option.of(source), Option.of(target), resource);

        return FsEvent.consForTest(FsEventType.FS_COPY, eventMetadata(uid), change);
    }

    private CommentEvent consPublicCommentEvent(CommentEventType type, DataApiUserId performer) {
        return consCommentEvent(type, performer, true);
    }

    private CommentEvent consPrivateCommentEvent(CommentEventType type, DataApiUserId performer) {
        return consCommentEvent(type, performer, false);
    }

    private CommentEvent consCommentEvent(CommentEventType type, DataApiUserId performer, boolean isPublic) {
        return new CommentEvent(
                type, eventMetadata(uid), MpfsUid.parse(performer.serialize()),
                new EntityRef((isPublic ? "public" : "private") + "_resource", uid + ":fileid"),
                new CommentRef(Option.empty(), Option.empty(), Option.empty()),
                new ParentCommentRef(Option.empty(), Option.empty()), Cf.list());
    }

    private ShareEvent consShareEvent(ShareEventType type, MpfsUid ownerUid) {
        boolean isInvite = !ownerUid.equals(mpfsUid);

        ShareData share = new ShareData(
                eventMetadata(uid), ownerUid, "gid",
                MpfsAddress.parseDir(uid + ":/disk/share"), isInvite, Option.of("hash"));

        return ShareEvent.consForTest(type, mpfsUid, eventMetadata(uid), share);
    }

    private void verifyResourceRecipients(DataApiUserId... uids) {
        ArgumentCaptor<DataApiUserId> uidCaptor = ArgumentCaptor.forClass(DataApiUserId.class);

        Mockito.verify(resourceBlockManager, Mockito.atLeast(0))
                .handleChanged(uidCaptor.capture(), any(), any(), any(), anyBoolean(), any());

        Assert.equals(
                Cf.x(uids).sortedBy(DataApiUserId::serialize),
                Cf.x(uidCaptor.getAllValues()).sortedBy(DataApiUserId::serialize));

        Mockito.reset(resourceBlockManager);
    }

    private void verifyNoMoreInteractions() {
        ListF<?> mocks = Cf.list(
                contentManager, resourceBlockManager,
                sharedFolderManager, folderCreationManager);

        Mockito.verifyNoMoreInteractions(mocks.toArray());

        Mockito.reset(mocks.toArray());
    }

    public static EventMetadata eventMetadata(DataApiUserId userId) {
        return new EventMetadata(
                new MpfsUid(userId.toPassportUidOrZero().toUid()),
                Instant.now(), new YandexCloudRequestId("web", "id", "host"));
    }
}
