package ru.yandex.chemodan.app.lentaloader;

import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.blocks.AlbumBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.ResourceBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateOrDeleteHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.log.ActionSourceHolder;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.DeleteOrUpdateLentaBlockTask;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.eventlog.events.ResourceType;
import ru.yandex.chemodan.eventlog.events.fs.FsEvent;
import ru.yandex.chemodan.eventlog.events.fs.FsEventResourceChange;
import ru.yandex.chemodan.eventlog.events.fs.FsEventType;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.ratelimiter.RateLimit;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;

/**
 * @author yashunsky
 */

@RunWith(MockitoJUnitRunner.class)
public class LentaEventsSaveFolderFromPublicTest {
    @Mock
    private MpfsClient mpfsClient;
    @Mock
    private LentaManager lentaManager;
    @Mock
    private AlbumBlockManager albumBlockManager;
    @Mock
    private ContentBlockManager contentManager;
    @Mock
    private ResourceBlockManager resourceBlockManager;
    @Mock
    private SharedFolderBlockManager sharedFolderManager;
    @Mock
    private BleedingEdge bleedingEdge;
    @Mock
    private RateLimiterClient rateLimiterClient;

    private FolderCreationBlockManager folderCreationBlockManager;

    private LentaEventsLogListener listener;

    private DeleteOrUpdateLentaBlockTaskForTest deleteTask;

    private final DataApiUserId uid = new DataApiPassportUserId(1234567);
    private final MpfsUid mpfsUid = new MpfsUid(uid.toPassportUidOrZero().toUid());

    private final DataApiUserId foreignUid = new DataApiPassportUserId(2345678);
    private final MpfsUid mpfsForeignUid = new MpfsUid(foreignUid.toPassportUidOrZero().toUid());

    private final MpfsResourceId mpfsResourceId = MpfsResourceId.parse(uid.toString() + ":folder_id");

    private final String blockId = "savingInProgressBlock";

    @Before
    public void setup() {
        Mockito.when(bleedingEdge.isOnBleedingEdge(any(PassportUid.class))).thenReturn(true);

        folderCreationBlockManager = new FolderCreationBlockManager(lentaManager, mpfsClient, rateLimiterClient);

        listener = new LentaEventsLogListener(
                mpfsClient, albumBlockManager,
                contentManager, resourceBlockManager, folderCreationBlockManager,
                sharedFolderManager, bleedingEdge);

        deleteTask = new DeleteOrUpdateLentaBlockTaskForTest(
                lentaManager, contentManager, folderCreationBlockManager);

        DeleteOrUpdateLentaBlockTaskForTest taskAsParametersHolder = new DeleteOrUpdateLentaBlockTaskForTest(
                uid, blockId, new ReasonedAction(ActionInfo.internal(ActionSource.test()), ActionReason.UNSPECIFIED));

        deleteTask.setParameters(taskAsParametersHolder.getParameters());

        ActionSourceHolder.set(ActionSource.test());
    }

    @After
    public void teardown() {
        ActionSourceHolder.remove();
    }

    @Test
    public void saveFolderFromPublic() {
        Mockito.when(rateLimiterClient.queryLimit(any())).thenReturn(RateLimit.proceedWithoutDelay());
        folderCreationBlockManager.handleStartSavingFolderFromPublic(
                uid, mpfsResourceId, Option.of("/disk/Downloads/name"), ActionInfo.internal(ActionSource.test()));
        Mockito.verify(lentaManager, Mockito.only()).findOrCreateBlock(any(), any(), any());
        Mockito.reset(lentaManager);

        ArgumentCaptor<UpdateOrDeleteHandler> handlerCaptor = ArgumentCaptor.forClass(UpdateOrDeleteHandler.class);

        deleteTask.executeForTest(uid, blockId, Option.empty());
        Mockito.verify(lentaManager, Mockito.only()).updateOrDeleteBlock(any(), any(), handlerCaptor.capture(), any());

        UpdateOrDeleteHandler handler = handlerCaptor.getValue();

        LentaBlockRecord rec = new LentaBlockRecord(
                blockId, 0, LentaRecordType.FOLDER_CREATION, "folder_id", Option.empty(), Instant.now(), Option.empty(),
                Cf.toMap(Cf.list(FolderCreationBlockFields.ACTION.toData(
                        FolderCreationBlockAction.SAVING_PUBLIC))));

        Assert.equals(UpdateOrDeleteHandler.Ignore.class, handler.getAction(rec).getClass());
        Mockito.reset(lentaManager);

        listener.processFsEvent(uid, consSaveFolderFromPublicFsEvent());
        Mockito.verify(lentaManager, Mockito.only()).findAndUpdateOrCreateBlock(any(), any(), any());
        Mockito.reset(lentaManager);
    }

    private FsEvent consSaveFolderFromPublicFsEvent() {
        ResourceLocation source = new ResourceLocation(
                MpfsAddress.parse(foreignUid + ":/disk/name", ResourceType.DIRECTORY),
                Option.of(MpfsResourceId.parse(foreignUid + ":/disk")));

        ResourceLocation target = new ResourceLocation(
                MpfsAddress.parse(uid + ":/disk/Downloads/name", ResourceType.DIRECTORY),
                Option.of(MpfsResourceId.parse(uid + ":/disk")));

        Resource resource = Resource.directory("fileid", mpfsForeignUid);

        FsEventResourceChange change = new FsEventResourceChange(
                mpfsUid, Option.of(source), Option.of(target), resource);

        return FsEvent.consForTest(FsEventType.FS_COPY, LentaEventsLogListenerTest.eventMetadata(uid), change);
    }

    private static class DeleteOrUpdateLentaBlockTaskForTest extends DeleteOrUpdateLentaBlockTask {

        public DeleteOrUpdateLentaBlockTaskForTest(DataApiUserId uid, String blockId, ReasonedAction actionInfo) {
            super(uid, blockId, actionInfo);
        }

        public DeleteOrUpdateLentaBlockTaskForTest(LentaManager lentaManager,
                ContentBlockManager contentBlockManager,
                FolderCreationBlockManager folderCreationBlockManager)
        {
            super(lentaManager, contentBlockManager, folderCreationBlockManager);
        }

        public void executeForTest(
                DataApiUserId uid, String blockId, Option<ActionReason> reason) {
            execute(uid, blockId, reason, null);
        }
    }
}
