package ru.yandex.chemodan.app.lentaloader;

import java.util.concurrent.ThreadPoolExecutor;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;

import ru.yandex.chemodan.app.lentaloader.web.WebContextConfiguration;
import ru.yandex.chemodan.app.lentaloader.worker.LentaWorkerContextConfiguration;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.LentaLoaderTasksContextConfiguration;
import ru.yandex.chemodan.eventlog.MetricsEventLogListener;
import ru.yandex.chemodan.eventlog.celery.QuellerFallbackEventsLogListener;
import ru.yandex.chemodan.logbroker.DiskLbConsumerContextConfiguration;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.chemodan.queller.worker.CeleryTaskManagerContextConfiguration;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.inside.logbroker.pull.consumer.AppCluster;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerSettingsProvider;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerWorkerService;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumers;
import ru.yandex.inside.logbroker.pull.consumer.LbGroupingAsyncLineListener;
import ru.yandex.inside.logbroker.pull.consumer.LbLineListener;
import ru.yandex.inside.logbroker.util.concurrent.GroupedExecutor;
import ru.yandex.inside.logbroker.util.concurrent.GroupedExecutorUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import({
        LentaCoreContextConfiguration.class,
        WebContextConfiguration.class,
        DiskLbConsumerContextConfiguration.class,
        LentaWorkerContextConfiguration.class,
        LentaLoaderTasksContextConfiguration.class,
        CeleryTaskManagerContextConfiguration.class
})
public class LentaLoaderContextConfiguration {

    @Autowired
    @Qualifier("lbListenerExecutor")
    private ThreadPoolExecutor listenerExecutor;

    @Bean
    public LentaActionSourceLogListener quellerFallbackEventsLogListener(
            LentaEventsLogListener listener, CeleryTaskManager celeryTaskManager,
            @Value("${lenta_all.queller.retry.count}") int retryCount)
    {
        return new LentaActionSourceLogListener(
                new QuellerFallbackEventsLogListener(listener, celeryTaskManager, retryCount,
                        ProcessLentaLogLineCeleryTask::new
                )
        );
    }

    @Bean
    @Primary
    public LbConsumerWorkerService lbConsumerWorkerService(
            AppCluster appCluster, LbConsumerSettingsProvider lbConsumerSettingsProvider,
            @Qualifier("lbConsumerListener") LbLineListener lbLineListener)
    {
        return new LbConsumerWorkerService(
                new LbConsumers(lbConsumerSettingsProvider, lbLineListener)
                        .forEachDcWithFixedNodeCount(appCluster)
        );
    }

    @Bean
    public LbGroupingAsyncLineListener<Long> lbGroupingAsyncLineListener(
            GroupedExecutor<Long, Void> groupedExecutor,
            LentaActionSourceLogListener logListener)
    {
        return new LbGroupingAsyncLineListener<>(logListener, groupedExecutor, TskvUtils::extractUidO);
    }

    @Bean
    public LbMetricsEventLineListener lbConsumerListener(MetricsEventLogListener metricsEventLogListener,
            LbGroupingAsyncLineListener<Long> lbGroupingAsyncLineListener,
            MetricsEventListenerCorePoolSizeConfigurer metricsEventListenerCorePoolSizeConfigurer)
    {
        return new LbMetricsEventLineListener(metricsEventLogListener, lbGroupingAsyncLineListener,
                metricsEventListenerCorePoolSizeConfigurer.getExecutor());
    }

    @Bean
    public GroupedExecutor<Long, Void> groupedExecutor(DynamicPropertyManager dynamicPropertyManager) {
        GroupedExecutor<Long, Void> groupedExecutor = new GroupedExecutor<>(listenerExecutor);
        GroupedExecutorUtils.bindToDynVars(groupedExecutor,
                ru.yandex.inside.logbroker.pull.DynamicVars.groupUidToOneTask,
                ru.yandex.inside.logbroker.pull.DynamicVars.maxUidQueueLength,
                dynamicPropertyManager
        );
        return groupedExecutor;
    }

    @Bean
    public MetricsEventListenerCorePoolSizeConfigurer metricsEventListenerCorePoolSizeConfigurer(
            DynamicPropertyManager dynamicPropertyManager,
            @Value("${logbroker.metrics-event.listener.pool.size}") int poolSize,
            @Value("${logbroker.metrics-event.listener.queue.size}") int queueSize
    )
    {
        MetricsEventListenerCorePoolSizeConfigurer corePoolSizeConfigurer =
                new MetricsEventListenerCorePoolSizeConfigurer(poolSize, queueSize);
        corePoolSizeConfigurer.registerTo(dynamicPropertyManager);
        return corePoolSizeConfigurer;
    }
}
