package ru.yandex.chemodan.app.lentaloader;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;

import org.mockito.Mockito;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.boot.ChemodanCommonContextConfiguration;
import ru.yandex.chemodan.boot.ChemodanMainSupport;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerWorkerService;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.ReaderSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.Service;
import ru.yandex.misc.thread.InterruptedRuntimeException;
import ru.yandex.misc.thread.factory.IncrementThreadFactory;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.version.SimpleAppName;
import ru.yandex.misc.worker.StoppableThread;

/**
 * @author dbrylev
 */
public class LentaLoaderFeedFromFileMain extends ChemodanMainSupport {

    public static void main(String[] args) {
        new LentaLoaderFeedFromFileMain().runMain(args);
    }

    public static class Processor implements Service {
        private static final Logger logger = LoggerFactory.getLogger(Processor.class);

        private final ExecutorService executor;
        private final Semaphore semaphore;

        private final LogListener logListener;
        private final ReaderSource input;
        private final OutputStreamSource rejected;

        public static Processor consAndStart(
                LogListener logListener,
                String inputName, String rejectedName, int threads) throws Exception
        {
            Processor processor = cons(logListener, inputName, rejectedName, threads);

            processor.start();

            return processor;
        }

        public static Processor cons(
                LogListener logListener,
                String inputName, String rejectedName, int threads)
        {
            File2 input = new File2(inputName), rejected = new File2(rejectedName);

            input.asReaderSource().readLine();
            rejected.asAppendOutputStreamTool().write("");

            ExecutorService executor = Executors.newFixedThreadPool(
                    threads, new IncrementThreadFactory("processor-pool-"));

            return new Processor(logListener,
                    input.asReaderSource(), rejected.asAppendOutputStreamTool(),
                    executor, new Semaphore(threads));
        }

        private Processor(
                LogListener logListener,
                ReaderSource input, OutputStreamSource rejected,
                ExecutorService executor, Semaphore semaphore)
        {
            this.logListener = logListener;
            this.input = input;
            this.rejected = rejected;
            this.executor = executor;
            this.semaphore = semaphore;
        }

        private final StoppableThread thread = new StoppableThread("processor") {
            private final AtomicInteger processedCount = new AtomicInteger();
            private final AtomicInteger rejectedCount = new AtomicInteger();

            @Override
            protected void run1() throws Exception {
                logger.info("Started");

                input.forEachLine(line -> {
                    checkTimeToDie();

                    try {
                        semaphore.acquire();

                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        throw new InterruptedRuntimeException(e);
                    }

                    executor.submit(() -> {
                        if (RetryUtils.retryE(logger, 3, () -> logListener.processLogLine(line)).isPresent()) {
                            rejected.writeLine(line);
                            rejectedCount.incrementAndGet();
                        }
                        if (processedCount.incrementAndGet() % 100 == 0) {
                            logger.info("Processed {}, rejected {}", processedCount.get(), rejectedCount.get());
                        }
                        semaphore.release();
                    });
                });
            }
        };

        @Override
        public void start() throws Exception {
            thread.start();
        }

        @Override
        public void stop() throws Exception {
            thread.stopGracefully();
            executor.shutdown();
        }
    }

    @Override
    public AppName applicationName() {
        return new SimpleAppName("disk", "lenta-loader");
    }

    @Override
    public ListF<Class<?>> applicationSpecificContextPath() {
        return Cf.list(
                ChemodanCommonContextConfiguration.class,
                LentaLoaderContextConfiguration.class,
                MockLbConsumerWorkerServiceContextConfiguration.class);
    }

    @Configuration
    public static class MockLbConsumerWorkerServiceContextConfiguration {
        @Bean
        public LbConsumerWorkerService lbConsumerWorkerService() {
            return Mockito.mock(LbConsumerWorkerService.class);
        }
    }
}
