package ru.yandex.chemodan.app.lentaloader;

import java.util.function.Consumer;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoReminderTaskManager;
import ru.yandex.chemodan.app.lentaloader.reminder.RemoveAutouploadBlockNotification;
import ru.yandex.chemodan.eventlog.MetricsEventLogListener;
import ru.yandex.chemodan.eventlog.events.client.MetricsEvent;
import ru.yandex.chemodan.eventlog.events.client.MetricsEventType;
import ru.yandex.chemodan.eventlog.log.TskvClientMetricsEventLogLine;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class LentaMetricsEventsLogListener extends MetricsEventLogListener {

    private static final Logger logger = LoggerFactory.getLogger(LentaMetricsEventsLogListener.class);

    private final DynamicProperty<Boolean> isAsyncEnabled =
            new DynamicProperty<>("cool-lenta-metrics-events-async-processing-enabled", true);

    private final DynamicProperty<Boolean> isProcessingEnabled =
            new DynamicProperty<>("cool-lenta-metrics-events-processing-enabled", false);

    private final DynamicProperty<Integer> maxEventValueLength =
            new DynamicProperty<>("cool-lenta-metrics-events-max-event-value-length", -1);

    private final BazingaTaskManager bazingaTaskManager;

    private final PhotoReminderTaskManager photoReminderTaskManager;

    private final MapF<MetricsEventType, Consumer<MetricsEvent>> EVENT_PROCESSORS =
            Cf.map(MetricsEventType.OPEN_LENTA_TAB, this::processOpenLentaTabEvent);

    public LentaMetricsEventsLogListener(BazingaTaskManager bazingaTaskManager,
            PhotoReminderTaskManager photoReminderTaskManager) {
        this.bazingaTaskManager = bazingaTaskManager;
        this.photoReminderTaskManager = photoReminderTaskManager;
    }

    @Override
    public void processEvent(MetricsEvent event) {
        EVENT_PROCESSORS.getO(event.getType()).ifPresent(clientEventConsumer -> clientEventConsumer.accept(event));
    }

    @Override
    protected Logger loggerForMonitoring() {
        return logger;
    }

    @Override
    protected boolean filterLogLine(TskvClientMetricsEventLogLine logLine) {
       if (!super.filterLogLine(logLine)) {
           return false;
       }
       int maxEventValueLength = this.maxEventValueLength.get();
       if (maxEventValueLength >= 0 && !logLine.getNonEmptyStringO(MetricsEvent.EVENT_VALUE_FIELD)
               .filter(eventValue -> eventValue.length() < maxEventValueLength).isPresent()) {
           return false;
       }
       return true;
    }

    private void processOpenLentaTabEvent(MetricsEvent event) {
        if (!isProcessingEnabled.get()) {
            logger.debug("Processing of Open Lenta Tab Event is disabled by flag");
            return;
        }
        PassportUid uid = new PassportUid(event.getAccountId());
        if (isAsyncEnabled.get()) {
            bazingaTaskManager.schedule(new RemoveAutouploadBlockNotification(uid, event.getTimestamp()),
                    Instant.now().plus(Duration.standardMinutes(5)));
            logger.debug("Scheduled processing of Open Lenta Tab Event in async mode. Event={}", event);
            return;
        }
        logger.debug("Starting processing of Open Lenta Tab Event in sync mode. Event={}", event);
        photoReminderTaskManager.deleteAddNotificationForAutouploadTaskForBlock(uid, event.getTimestamp());
    }
}
