package ru.yandex.chemodan.app.lentaloader.blocks;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.OrderedUUID;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.misc.db.q.SqlLimits;

/**
 * @author dbrylev
 */
public class BlockItemManager {
    private static final AppDatabaseRef DB_REF = new AppDatabaseRef("lenta", "block_items");

    private final DataApiManager dataApiManager;

    public BlockItemManager(DataApiManager dataApiManager) {
        this.dataApiManager = dataApiManager;
    }

    public ListF<BlockItemRecord> getItems(DataApiUserId uid, String blockId, Option<String> lastOrderO,
            Option<Integer> limit)
    {
        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionId(blockId)
                .withDataCondO(lastOrderO.map(BlockItemRecord.Fields.ORDER.column()::lt))
                .withRecordOrder(BlockItemRecord.Fields.ORDER.column().orderByDesc())
                .withLimitsO(limit.map(SqlLimits::first));
        return dataApiManager.getRecords(consDatabaseSpec(uid), filter)
                .map(BlockItemRecord::fromDataRecord);
    }

    public void createOrUpdateItem(DataApiUserId uid, BlockItemId id, MapF<String, DataField> data) {
        Database db = dataApiManager.getOrCreateDatabase(consDatabaseSpec(uid));

        Option<BlockItemRecord> found = findItemRecord(db, id);
        BlockItemRecord record = new BlockItemRecord(id.itemId, id.blockId, OrderedUUID.generateOrderedUUID(), data);

        ListF<RecordChange> changes = found.isPresent()
                ? Cf.list(RecordChange.update(id.blockId, id.itemId, record.diffFrom(found.get().toData())))
                : Cf.list(RecordChange.insert(id.blockId, id.itemId, record.toData()));

        dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    public void deleteItem(DataApiUserId uid, BlockItemId id) {
        deleteItems(uid, Option.of(id.blockId), id.itemId);
    }

    public void deleteItems(DataApiUserId uid, Option<String> blockId, String itemId) {
        dataApiManager.getDatabaseO(consDatabaseSpec(uid)).forEach(db -> {
            ListF<BlockItemRecord> records = findItemRecords(db, blockId, itemId);

            if (records.isNotEmpty()) {
                dataApiManager.applyDelta(
                        db, RevisionCheckMode.PER_RECORD,
                        new Delta(records.map(r -> RecordChange.delete(r.blockId, r.itemId))));
            }
        });
    }

    private Option<BlockItemRecord> findItemRecord(Database db, BlockItemId id) {
        return findItemRecords(db, Option.of(id.blockId), id.itemId).singleO();
    }

    private ListF<BlockItemRecord> findItemRecords(Database db, Option<String> blockId, String itemId) {
        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdO(blockId)
                .withRecordId(itemId)
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_DESC);
        return dataApiManager.getRecords(db.spec(), filter)
                .map(BlockItemRecord::fromDataRecord);
    }

    private UserDatabaseSpec consDatabaseSpec(DataApiUserId uid) {
        return new UserDatabaseSpec(uid, DB_REF);
    }
}
