package ru.yandex.chemodan.app.lentaloader.blocks;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.DynamicVars;
import ru.yandex.chemodan.app.lentaloader.lenta.FieldPredicate;
import ru.yandex.chemodan.app.lentaloader.lenta.FindOrCreateResult;
import ru.yandex.chemodan.app.lentaloader.lenta.GroupKeyPredicate;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaNotificationManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.limit.LentaLimitManager;
import ru.yandex.chemodan.app.lentaloader.lenta.update.CreateHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.DeleteHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockModifyData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockUpdateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.DataOrRefusal;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsLentaBlockFileIds;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.util.TimeUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class ContentBlockManager {

    private static final Logger logger = LoggerFactory.getLogger(ContentBlockManager.class);

    private final LentaManager lentaManager;
    private final LentaLimitManager lentaLimitManager;
    private final LentaNotificationManager lentaNotificationManager;
    private final MpfsClient mpfsClient;
    //used for debug in test environments
    private final boolean createNotificationOnEveryAutoupload;

    public ContentBlockManager(
            LentaManager lentaManager,
            LentaLimitManager lentaLimitManager,
            LentaNotificationManager lentaNotificationManager,
            MpfsClient mpfsClient,
            boolean createNotificationOnEveryAutoupload)
    {
        this.lentaManager = lentaManager;
        this.lentaLimitManager = lentaLimitManager;
        this.lentaNotificationManager = lentaNotificationManager;
        this.mpfsClient = mpfsClient;
        this.createNotificationOnEveryAutoupload = createNotificationOnEveryAutoupload;
    }

    public DataOrRefusal<MpfsLentaBlockFileIds> getFileIds(
            DataApiUserId uid, String folderId, String mediaType,
            MpfsUid modifier, Instant mFrom, Instant mTill, int amount)
    {
        Option<MpfsLentaBlockFileIds> result = mpfsClient.getLentaBlocksFileIds(
                uid.forMpfs(), folderId, mediaType, modifier,
                TimeUtils.unixTime(mFrom), TimeUtils.unixTimeTill(mTill), amount);

        return DataOrRefusal.someOrRefusal(result, ActionReason.MPFS_4XX)
                .matchOrRefusal(c -> c.totalCount > 0, ActionReason.MPFS_NO_FILES);
    }

    public DataOrRefusal<MpfsLentaBlockFileIds> getFilesCountAndFirstId(
            DataApiUserId uid, ModifiedResource resource, Instant mFrom, Instant mTill)
    {
        return getFileIds(uid, resource.folderId.serialize(), resource.mediaType(), resource.modifier, mFrom, mTill, 1);
    }

    public DataOrRefusal<Integer> getFilesCount(
            DataApiUserId uid, ContentBlockMeta meta, Instant mFrom, Instant mTill)
    {
        MpfsUid mpfsUid = MpfsUid.parse(meta.modifierUid);
        return getFileIds(uid, meta.folderId, meta.mediaType, mpfsUid, mFrom, mTill, 0)
                .map(c -> c.totalCount);
    }

    public DataOrRefusal<Integer> getFilesCount(DataApiUserId uid, LentaBlockRecord block) {
        ContentBlockMeta meta = ContentBlockMeta.fromBlock(uid, block);
        return getFilesCount(uid, meta, block.mFrom.get(), block.mTill.get());
    }

    //xxx: to many parameters
    public void createOrUpdateFile(
            DataApiUserId uid,
            ModifiedResource resource,
            Option<ContentBlockAction> action,
            Option<ResourceLocation> sourceLocation,
            ActionInfo actionInfo,
            Option<String> actionArea)
    {
        logger.info("createOrUpdateFile {} {} {} {}", action, sourceLocation, actionInfo, uid);
        MpfsResourceId folderId = resource.folderId;

        String mediaType = resource.mediaType();
        String groupKey = groupKey(resource);

        Option<String> owner = Option.when(
                !resource.owner.getUidO().isSome(uid.toPassportUid()), resource.owner.getRawValue());

        Function<Integer, MapF<String, DataField>> blockFieldsWithCount = filesCount -> Cf.toMap(Cf.list(
                ContentBlockFields.MEDIA_TYPE.toData(mediaType),
                ContentBlockFields.FOLDER_ID.toData(folderId.serialize()),
                ContentBlockFields.MODIFIER_UID.toData(resource.modifier.getRawValue()),
                ContentBlockFields.FILES_COUNT.toData(filesCount))
                .plus(owner.map(ContentBlockFields.OWNER_UID::toData))
                .plus(actionArea.map(ContentBlockFields.AREA::toData)));

        Function<LentaBlockRecord, DataOrRefusal<MapF<String, DataField>>> blockFields = record -> {

            if (sourceLocation.isPresent()) {
                Option<MpfsResourceId> folderIdO = sourceLocation.get().folderId;
                if (folderIdO.isPresent()) {
                    if (DynamicVars.mpfsShareUids.get().containsTs(folderIdO.get().owner.toString())) {
                        return DataOrRefusal.refusal(ActionReason.DEFAULT_FILE);
                    }
                }
            }

            if (resource.address.path.service.equals("photounlim") && !action.isSome(ContentBlockAction.PHOTO_UNLIM)) {
                return DataOrRefusal.refusal(ActionReason.NOT_UPLOAD_INTO_PHOTOUNLIM);
            }

            DataOrRefusal<MpfsLentaBlockFileIds> files = getFilesCountAndFirstId(
                    uid, resource, record.mFrom.get(), record.mTill.get());

            if (files.isRefusal()) {
                return DataOrRefusal.refusal(files.getRefusalReason());
            }

            if (files.getData().totalCount == 1 && files.getData().fileIds.containsTs(resource.fileId())) {

                if (lentaManager.findCachedAndUpdateBlock(uid, new LentaBlockUpdateData(
                            LentaRecordType.PUBLIC_RESOURCE_OWNED,
                            ResourceBlockManager.groupKey(OwnedOrPublishedResource.owned(resource.resource)),

                            rec -> rec.mTime.isAfter(actionInfo.now.minus(Duration.standardDays(1)))
                                    ? UpdateHandler.updateAndUpThrottled(rec.specific)
                                    : UpdateHandler.ignore(ActionReason.PUBLIC_RESOURCE_DEDUPLICATION)),
                        actionInfo))
                {
                    return DataOrRefusal.refusal(ActionReason.PUBLIC_RESOURCE_DEDUPLICATION);
                }
            }
            return DataOrRefusal.data(blockFieldsWithCount.apply(files.getData().totalCount));
        };

        CreateHandler createHandler = record -> {
            DataOrRefusal<MapF<String, DataField>> fields = blockFields.apply(record);

            if (!fields.isRefusal()) {
                return CreateHandler.create(fields.getData()
                        .plus(Cf.toMap(action.map(ContentBlockFields.ACTION::toData))));
            } else {
                return CreateHandler.ignore(fields.getRefusalReason());
            }
        };

        UpdateHandler updateHandler = record -> {
            Option<ContentBlockAction> prevAction = ContentBlockFields.ACTION.getO(record);

            if (prevAction.exists(a -> !action.isSome(a))) {
                DataOrRefusal<MapF<String, DataField>> fields = blockFields.apply(record);

                if (!fields.isRefusal()) {
                    return UpdateHandler.updateAndUp(fields.getData());

                } else {
                    return UpdateHandler.ignore(fields.getRefusalReason());
                }
            } else {
                return UpdateHandler.updateAndUpThrottled(() -> record.specific.plus(Cf.toMap(
                        getFilesCount(uid, record).getDataO().map(ContentBlockFields.FILES_COUNT::toData))));
            }
        };

        if (lentaLimitManager.handleContentPathBlocking(uid, resource, actionInfo)) {
            return;
        }

        FindOrCreateResult result = lentaManager.findCachedAndUpdateOrCreateBlock(uid, new LentaBlockModifyData(
                LentaRecordType.CONTENT_BLOCK, groupKey, createHandler, updateHandler), actionInfo);

        if (result.isCreated()) {
            lentaLimitManager.insertContentBlockCreature(uid, result.getRecordId(), resource, actionInfo);
            deleteFolderCreationBlock(uid, resource, actionInfo.withReason(ActionReason.FOLDER_CREATION_REPLACEMENT));
        }

        scheduleContentBlockNotifications(uid, action, result, resource.modifier);
        lentaManager.deleteOrUpdateBlocksAsync(uid, LentaRecordType.CONTENT_BLOCK,
                GroupKeyPredicate.startsWith(folderId.serialize() + ":" + resource.mediaType() + ":"),
                actionInfo.withReason(ActionReason.FILE_CHANGED_OR_REMOVED));
    }

    private void deleteFolderCreationBlock(DataApiUserId uid, ModifiedResource resource, ReasonedAction actionInfo) {
        Option<LentaBlockRecord> block = lentaManager.findBlocks(uid, LentaRecordType.FOLDER_CREATION,
                GroupKeyPredicate.eq(FolderCreationBlockManager.groupKey(resource)), actionInfo.actionInfo).firstO();

        if (block.isPresent()) {
            try {
                lentaManager.deleteBlock(uid, block.get().id, actionInfo);

            } catch (RuntimeException e) {
                lentaManager.deleteEmptyBlockDelayed(uid, block.get().getIdAndType(), actionInfo);
            }
        }
    }

    private void scheduleContentBlockNotifications(
            DataApiUserId uid, Option<ContentBlockAction> action,
            FindOrCreateResult result, MpfsUid mpfsUid)
    {
        if (action.isSome(ContentBlockAction.AUTOSAVE) || action.isSome(ContentBlockAction.PHOTO_UNLIM)) {
            if (result.isCreated() || createNotificationOnEveryAutoupload) {
                lentaNotificationManager.scheduleAutouploadNotification(uid, result.getRecordId());
            }
        } else if (action.isSome(ContentBlockAction.ADDITION) || action.isSome(ContentBlockAction.UPDATE)) {
            if (uid.isPassportUid() && !mpfsUid.getUidO().isSome(uid.toPassportUid())) {
                lentaNotificationManager.scheduleSharedFolderBlockNotification(uid, result.getRecordId());
            }
        }
    }

    public void deleteFile(DataApiUserId uid, MpfsResourceId folderId, Resource resource, ReasonedAction actionInfo) {
        lentaManager.deleteOrUpdateBlocksAsync(uid, LentaRecordType.CONTENT_BLOCK,
                GroupKeyPredicate.startsWith(folderId.serialize() + ":"
                        + resource.compoundType.mediaTypeValueOrUnknown() + ":"), actionInfo);
    }

    public void deleteFolder(DataApiUserId uid, MpfsResourceId folderId, ReasonedAction actionInfo) {
        lentaManager.deleteOrUpdateBlocksAsync(uid, LentaRecordType.CONTENT_BLOCK,
                GroupKeyPredicate.startsWith(folderId.serialize() + ":"), actionInfo);
    }

    public void deleteEmptyOrSingleFileBlock(DataApiUserId uid, ModifiedResource resource, ReasonedAction actionInfo) {
        String groupKey = groupKey(resource);

        lentaManager.deleteMostRecentBlock(uid, LentaRecordType.CONTENT_BLOCK, groupKey, block -> {
            DataOrRefusal<MpfsLentaBlockFileIds> files = getFilesCountAndFirstId(
                    uid, resource, block.mFrom.get(), block.mTill.get());

            files = files.matchOrRefusal(
                    c -> c.totalCount != 1 || !c.fileIds.containsTs(resource.fileId()),
                    actionInfo.reason);

            return files.isRefusal() ? DeleteHandler.delete(files.getRefusalReason()) : DeleteHandler.ignore();
        }, actionInfo.actionInfo);
    }

    public static String groupKey(ModifiedResource resource) {
        return resource.folderId.serialize()
                + ":" + resource.mediaType()
                + ":" + resource.modifier.getRawValue();
    }

    public static FieldPredicate groupKeyPredicate(MpfsResourceId folderId) {
        return GroupKeyPredicate.startsWith(folderId.serialize() + ":");
    }

}
