package ru.yandex.chemodan.app.lentaloader.blocks;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.lenta.FindOrCreateResult;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaNotificationManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.limit.LentaLimitManager;
import ru.yandex.chemodan.app.lentaloader.lenta.update.DeleteHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockModifyData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockUpdateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.log.DataOrRefusal;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.app.lentaloader.test.FieldUpdateCaptor;
import ru.yandex.chemodan.app.lentaloader.test.TestUtils;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsLentaBlockFileIds;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyInt;

/**
 * @author dbrylev
 */
@RunWith(MockitoJUnitRunner.class)
public class ContentBlockManagerTest {

    @Mock
    private MpfsClient mpfsClient;
    @Mock
    private LentaManager lentaManager;
    @Mock
    private LentaLimitManager lentaLimitManager;
    @Mock
    private LentaNotificationManager lentaNotificationManager;

    private ContentBlockManager blockManager;

    @Before
    public void setup() {
        blockManager = new ContentBlockManager(
                lentaManager, lentaLimitManager, lentaNotificationManager,
                mpfsClient, true);
    }

    private final DataApiUserId uid = new DataApiPassportUserId(1234567);
    private final MpfsUid mpfsUid = new MpfsUid(uid.toPassportUidOrZero().getUid());

    private final Resource document = Resource.file("document", "file_id", mpfsUid);

    private final ModifiedResource modifiedDocument = new ModifiedResource(document,
            MpfsAddress.parseFile(mpfsUid + ":/disk/document"),
            MpfsResourceId.parse(mpfsUid + ":/disk"), mpfsUid, mpfsUid);


    @Test
    public void createOrUpdateIgnore() {
        ArgumentCaptor<LentaBlockModifyData> dataCaptor = captureUpdateOrCreateData();

        Function0<ActionReason> createIgnoreReason = () -> dataCaptor.getValue()
                .createHandler.getAction(consBlock(Cf.map())).specific.getRefusalReason();

        Function<Option<ContentBlockAction>, UpdateHandler.Action> updateAction = action -> dataCaptor.getValue()
                .updateHandler.getAction(consBlock(Cf.toMap(action.map(ContentBlockFields.ACTION::toData))));

        Function0<ActionReason> updateIgnoreReason = () ->
                ((UpdateHandler.Ignore) updateAction.apply(Option.of(ContentBlockAction.UPDATE))).reason;

        blockManager.createOrUpdateFile(uid, modifiedDocument,
                Option.of(ContentBlockAction.AUTOSAVE), Option.empty(), actionInfo(), Option.of("unlim"));

        mockMpfsLentaBlocksFileIds(Option.empty());

        Assert.equals(ActionReason.MPFS_4XX, createIgnoreReason.apply());
        Assert.equals(ActionReason.MPFS_4XX, updateIgnoreReason.apply());

        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(0, Cf.list())));

        Assert.equals(ActionReason.MPFS_NO_FILES, createIgnoreReason.apply());
        Assert.equals(ActionReason.MPFS_NO_FILES, updateIgnoreReason.apply());

        Assert.isInstance(updateAction.apply(Option.empty()), UpdateHandler.Update.class);

        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(1, Cf.list(document.fileId))));

        Mockito.when(lentaManager.findCachedAndUpdateBlock(Mockito.eq(uid), any(), any())).thenReturn(true);

        Assert.equals(ActionReason.PUBLIC_RESOURCE_DEDUPLICATION, createIgnoreReason.apply());

        Mockito.reset(lentaManager);

        Mockito.when(lentaLimitManager.handleContentPathBlocking(any(), any(), any())).thenReturn(true);
        blockManager.createOrUpdateFile(uid, modifiedDocument,
                Option.of(ContentBlockAction.AUTOSAVE), Option.empty(), actionInfo(), Option.of("unlim"));

        Mockito.verifyNoMoreInteractions(lentaManager);
    }

    @Test
    public void publicResourceDeduplication() {
        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(1, Cf.list(document.fileId))));

        ArgumentCaptor<LentaBlockModifyData> modifyCaptor = captureUpdateOrCreateData();

        ArgumentCaptor<LentaBlockUpdateData> updateCaptor = captureUpdateData();

        blockManager.createOrUpdateFile(uid, modifiedDocument, Option.empty(), Option.empty(), actionInfo(),
                Option.of("unlim"));

        modifyCaptor.getValue().createHandler.getAction(consBlock(Cf.map()));

        UpdateHandler handler = updateCaptor.getValue().handler;

        LentaBlockRecord rec = TestUtils.consBlock(LentaRecordType.PUBLIC_RESOURCE_OWNED,
                Instant.now().minus(Duration.standardDays(2)), Cf.map());

        Assert.isInstance(handler.getAction(rec), UpdateHandler.Ignore.class);

        rec = TestUtils.consBlock(LentaRecordType.PUBLIC_RESOURCE_OWNED, Instant.now(), Cf.map());

        Assert.isInstance(handler.getAction(rec), UpdateHandler.Update.class);
    }

    @Test
    public void contentBlockAction() {
        ContentBlockManager blockManager = Mockito.spy(this.blockManager);

        FieldUpdateCaptor<ContentBlockAction, ContentBlockAction> captor = new FieldUpdateCaptor<>(
                ContentBlockFields.ACTION, LentaRecordType.CONTENT_BLOCK,
                Cf.toMap(Cf.list(ContentBlockFields.FILES_COUNT.toData(2))), lentaManager,
                action -> blockManager.createOrUpdateFile(uid, modifiedDocument, action, Option.empty(), actionInfo(),
                        Option.of("unlim")));

        Mockito.doReturn(DataOrRefusal.data(2)).when(blockManager).getFilesCount(any(), any());
        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(2, Cf.list())));

        captor.create(Option.empty()).assertChangedToNone();
        captor.create(ContentBlockAction.UPDATE).assertChangedTo(ContentBlockAction.UPDATE);

        captor.update(Option.empty(), Option.empty()).assertThrottledToNone();
        captor.update(Option.empty(), ContentBlockAction.UPDATE).assertThrottledToNone();
        captor.update(ContentBlockAction.UPDATE, ContentBlockAction.UPDATE).assertThrottledTo(ContentBlockAction.UPDATE);

        captor.update(ContentBlockAction.UPDATE, ContentBlockAction.AUTOSAVE).assertChangedToNone();
        captor.update(ContentBlockAction.UPDATE, Option.empty()).assertChangedToNone();
    }

    @Test
    public void noContentBlockForDefaultFile() {
        Option<ResourceLocation> sourceLocation = Option.of(new ResourceLocation(null,
                Option.of(new MpfsResourceId(MpfsUid.parse("share_production"), "some file")))
        );

        FieldUpdateCaptor<ContentBlockAction, ContentBlockAction> captor = new FieldUpdateCaptor<>(
                ContentBlockFields.ACTION, LentaRecordType.CONTENT_BLOCK, lentaManager,
                action -> blockManager.createOrUpdateFile(uid, modifiedDocument, action, sourceLocation, actionInfo(),
                        Option.of("unlim")));

        captor.create(Option.empty()).assertIgnored();
        captor.create(ContentBlockAction.ADDITION).assertIgnored();
        captor.create(ContentBlockAction.AUTOSAVE).assertIgnored();
        captor.create(ContentBlockAction.UPDATE).assertIgnored();
    }

    @Test
    public void deleteEmptyOrSingleFileBlock() {
        ReasonedAction actionInfo = reasonedAction();

        ArgumentCaptor<DeleteHandler> handlerCaptor = ArgumentCaptor.forClass(DeleteHandler.class);
        Mockito.doNothing().when(lentaManager)
                .deleteMostRecentBlock(any(), any(), any(), handlerCaptor.capture(), any());

        blockManager.deleteEmptyOrSingleFileBlock(uid, modifiedDocument, actionInfo);

        Function0<Option<ActionReason>> deleteReason = () ->
                Option.of(handlerCaptor.getValue().getAction(consBlock(Cf.map())))
                        .filterByType(DeleteHandler.Delete.class).map(a -> a.reason);

        mockMpfsLentaBlocksFileIds(Option.empty());
        Assert.some(ActionReason.MPFS_4XX, deleteReason.apply());

        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(0, Cf.list())));
        Assert.some(ActionReason.MPFS_NO_FILES, deleteReason.apply());

        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(1, Cf.list(document.fileId))));
        Assert.some(actionInfo.reason, deleteReason.apply());

        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(1, Cf.list("zzz"))));
        Assert.none(deleteReason.apply());

        mockMpfsLentaBlocksFileIds(Option.of(new MpfsLentaBlockFileIds(2, Cf.list(document.fileId, "zzz"))));
        Assert.none(deleteReason.apply());
    }

    private void mockMpfsLentaBlocksFileIds(Option<MpfsLentaBlockFileIds> response) {
        Mockito.when(mpfsClient.getLentaBlocksFileIds(any(), any(), any(), any(), anyInt(), anyInt(), anyInt()))
                .thenReturn(response);
    }

    private ArgumentCaptor<LentaBlockModifyData> captureUpdateOrCreateData() {
        ArgumentCaptor<LentaBlockModifyData> dataCaptor = ArgumentCaptor.forClass(LentaBlockModifyData.class);

        Mockito.when(lentaManager.findCachedAndUpdateOrCreateBlock(any(), dataCaptor.capture(), any()))
                .thenReturn(Mockito.mock(FindOrCreateResult.class));

        return dataCaptor;
    }

    private ArgumentCaptor<LentaBlockUpdateData> captureUpdateData() {
        ArgumentCaptor<LentaBlockUpdateData> dataCaptor = ArgumentCaptor.forClass(LentaBlockUpdateData.class);

        Mockito.when(lentaManager.findCachedAndUpdateBlock(any(), dataCaptor.capture(), any())).thenReturn(true);

        return dataCaptor;
    }

    private static LentaBlockRecord consBlock(MapF<String, DataField> specific) {
        return TestUtils.consBlock(LentaRecordType.CONTENT_BLOCK, specific);
    }

    private static ActionInfo actionInfo() {
        return ActionInfo.internal(ActionSource.test());
    }

    private static ReasonedAction reasonedAction() {
        return new ReasonedAction(actionInfo(), ActionReason.UNSPECIFIED);
    }
}
