package ru.yandex.chemodan.app.lentaloader.blocks;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.lenta.FieldPredicate;
import ru.yandex.chemodan.app.lentaloader.lenta.GroupKeyPredicate;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.CreateHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockModifyData;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.LentaBlockEvent;
import ru.yandex.chemodan.app.lentaloader.log.LentaBlockEventsCache;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class FolderCreationBlockManager {
    private static final Logger logger = LoggerFactory.getLogger(FolderCreationBlockManager.class);

    private final LentaManager lentaManager;
    private final MpfsClient mpfsClient;
    private final RateLimiterClient rateLimiterClient;

    public FolderCreationBlockManager(LentaManager lentaManager, MpfsClient mpfsClient,
            RateLimiterClient rateLimiterClient)
    {
        this.lentaManager = lentaManager;
        this.mpfsClient = mpfsClient;
        this.rateLimiterClient = rateLimiterClient;
    }

    public void handleCreation(
            DataApiUserId uid, ModifiedResource folder, Option<FolderCreationBlockAction> action, ActionInfo actionInfo)
    {
        handleCreation(uid, folder.folderId, Option.empty(), action, actionInfo);
    }

    public ListF<LentaBlockEvent> handleStartSavingFolderFromPublic(
            DataApiUserId uid, MpfsResourceId folderId, Option<String> folderPath, ActionInfo actionInfo)
    {
        return LentaBlockEventsCache.executeAndGetBlocks(() -> handleCreation(
                uid, folderId, folderPath, Option.of(FolderCreationBlockAction.SAVING_PUBLIC), actionInfo)
        );
    }

    private void handleCreation(
            DataApiUserId uid, MpfsResourceId folderId, Option<String> folderPath,
            Option<FolderCreationBlockAction> action, ActionInfo actionInfo)
    {
        if (!rateLimiterClient.queryLimit(uid.toString()).proceed) {
            logger.warn("Too many folders creation for uid={}; Ignore processing", uid);
            return;
        }

        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                FolderCreationBlockFields.FOLDER_ID.toData(folderId.serialize())))
                .plus(Cf.toMap(action.map(FolderCreationBlockFields.ACTION::toData)))
                .plus(Cf.toMap(folderPath.map(FolderCreationBlockFields.FOLDER_PATH::toData)));

        CreateHandler createHandler = rec -> {
            if (!lentaManager.findMostRecentBlock(
                    uid, LentaRecordType.CONTENT_BLOCK,
                    ContentBlockManager.groupKeyPredicate(folderId), actionInfo).isPresent())
            {
                return CreateHandler.create(specific);
            } else {
                return CreateHandler.ignore(ActionReason.FOLDER_CREATION_REPLACEMENT);
            }
        };
        if (action.isSome(FolderCreationBlockAction.SAVE_PUBLIC)) {
            lentaManager.findAndUpdateOrCreateBlock(uid, LentaBlockModifyData.createOrUpdate(
                    LentaRecordType.FOLDER_CREATION, groupKey(folderId), specific), actionInfo);
        } else {
            lentaManager.findOrCreateBlock(uid, new LentaBlockCreateData(
                    LentaRecordType.FOLDER_CREATION, groupKey(folderId), createHandler), actionInfo);
        }
    }

    public void createDirectly(DataApiUserId uid, MpfsResourceId folderId, ActionInfo actionInfo) {
        lentaManager.findOrCreateBlock(uid, new LentaBlockCreateData(
                LentaRecordType.FOLDER_CREATION, groupKey(folderId),
                Cf.toMap(Cf.list(FolderCreationBlockFields.FOLDER_ID.toData(folderId.serialize())))), actionInfo);
    }

    public void delete(DataApiUserId uid, MpfsResourceId folderId, ReasonedAction actionInfo) {
        lentaManager.deleteBlocks(uid, LentaRecordType.FOLDER_CREATION, groupKey(folderId), actionInfo);
    }

    public boolean folderExists(DataApiUserId uid, MpfsResourceId folderId) {
        return mpfsClient.getFileInfoOByFileId(
                uid.forMpfs(), folderId.owner.getUid().toString(), folderId.fileId).isPresent();
    }

    public static String groupKey(ModifiedResource resource) {
        return groupKey(resource.folderId);
    }

    public static String groupKey(MpfsResourceId folderId) {
        return folderId.owner.getRawValue() + ":" + folderId.fileId;
    }

    public static FieldPredicate groupKeyPredicate(MpfsUid owner) {
        return GroupKeyPredicate.startsWith(owner.getRawValue() + ":");
    }
}
