package ru.yandex.chemodan.app.lentaloader.blocks;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.CreateHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.DeleteHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockBaseData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockModifyData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.eventlog.events.EventType;
import ru.yandex.chemodan.eventlog.events.ResourceType;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class ResourceBlockManager {
    private static final Logger logger = LoggerFactory.getLogger(ResourceBlockManager.class);

    private final LentaManager lentaManager;
    private final RateLimiterClient rateLimiterClient;
    private final MpfsClient mpfsClient;

    public ResourceBlockManager(LentaManager lentaManager,
            RateLimiterClient rateLimiterClient,
            MpfsClient mpfsClient)
    {
        this.lentaManager = lentaManager;
        this.rateLimiterClient = rateLimiterClient;
        this.mpfsClient = mpfsClient;
    }

    public void handleChanged(
            DataApiUserId uid, OwnedOrPublishedResource resource,
            MpfsUid performer, EventType action, boolean isPublicResource, ActionInfo actionInfo)
    {
        if (!rateLimiterClient.queryLimit(uid.toString()).proceed) {
            logger.warn("Too many public links for uid={}; Ignore processing", uid);
            return;
        }

        String mediaType = resource.type.type == ResourceType.FILE
                ? resource.type.mediaTypeValueOrUnknown()
                : resource.type.type == ResourceType.DIRECTORY
                ? "folder"
                : resource.type.type.value();

        String groupKey = groupKey(resource);

        if (!isPublicResource) {
            ListF<Tuple2<String, DataField>> specific = Cf.list(
                    SharedResourceBlockFields.ENTITY_ID.toData(resource.idOrUrl.getLeft().serialize()),
                    SharedResourceBlockFields.MEDIA_TYPE.toData(mediaType));

            lentaManager.findCachedAndUpdateOrCreateBlock(uid,
                    LentaBlockModifyData.createOrUpdateAndUpThrottled(
                            LentaRecordType.SHARED_RESOURCE, groupKey, Cf.toMap(specific)), actionInfo);

        } else if (resource.idOrUrl.isLeft()) {
            MpfsResourceId resourceId = resource.idOrUrl.getLeft();

            LentaBlockBaseData base = new LentaBlockBaseData(LentaRecordType.PUBLIC_RESOURCE_OWNED, groupKey);

            Option<String> publisher = Option.when(action == EventType.FS_SET_PUBLIC, performer.getRawValue());

            MapF<String, DataField> specific = Cf.toMap(Cf.list(
                    PublicResourceOwnedBlockFields.OWNER_UID.toData(resourceId.owner.getRawValue()),
                    PublicResourceOwnedBlockFields.FILE_ID.toData(resourceId.fileId),
                    PublicResourceOwnedBlockFields.MEDIA_TYPE.toData(mediaType))
                    .plus(publisher.map(PublicResourceOwnedBlockFields.PUBLISHER_UID::toData)));

            lentaManager.findCachedAndUpdateOrCreateBlock(uid, new LentaBlockModifyData(base,
                    rec -> CreateHandler.create(specific),
                    rec -> {
                        Option<String> curPublisher = PublicResourceOwnedBlockFields.PUBLISHER_UID.getO(rec);

                        return publisher.exists(m -> !curPublisher.isSome(m))
                                ? UpdateHandler.updateAndUp(specific)
                                : UpdateHandler.updateAndUpThrottled(rec.specific);
                    }), actionInfo);
        } else {
            String resourceUrl = resource.idOrUrl.getRight();

            Option<EventType> userAction = Option.when(uid.serialize().equals(performer.getRawValue()), action);

            LentaBlockBaseData base = new LentaBlockBaseData(LentaRecordType.PUBLIC_RESOURCE, groupKey);

            ListF<Tuple2<String, DataField>> basicFields = Cf.list(
                    PublicResourceBlockFields.SHORT_URL.toData(resourceUrl),
                    PublicResourceBlockFields.MEDIA_TYPE.toData(mediaType));

            Option<Tuple2<String, DataField>> actionField = userAction
                    .flatMapO(PublicResourceAction::byEventType)
                    .map(PublicResourceBlockFields.MY_ACTION::toData);

            if (userAction.isSome(EventType.PUBLIC_VISIT)) {
                lentaManager.findOrCreateBlock(uid, new LentaBlockCreateData(
                        base.type, base.groupKey, Cf.toMap(basicFields.plus(actionField))), actionInfo);

            } else if (userAction.isPresent()) {
                lentaManager.findAndUpdateOrCreateBlock(uid, new LentaBlockModifyData(base,
                        rec -> CreateHandler.create(Cf.toMap(basicFields.plus(actionField))),
                        rec -> UpdateHandler.updateAndUp(Cf.toMap(basicFields.plus(actionField
                                .filter(f -> PublicResourceBlockFields.MY_ACTION.getO(rec)
                                        .exists(a -> !a.isCancelledBy(userAction.get()))))))),
                        actionInfo);
            } else {
                lentaManager.findCachedAndUpdateOrCreateBlock(uid,
                        LentaBlockModifyData.createOrUpdateAndUpThrottled(
                                base.type, base.groupKey, Cf.toMap(basicFields)), actionInfo);
            }
        }
    }

    public void handleDeleted(DataApiUserId uid, MpfsResourceId resourceId, ReasonedAction actionInfo) {
        lentaManager.deleteMostRecentBlock(
                uid, LentaRecordType.PUBLIC_RESOURCE_OWNED, resourceId.serialize(),
                rec -> getPublicHash(uid, rec).isPresent()
                        ? DeleteHandler.ignore()
                        : DeleteHandler.delete(actionInfo.reason), actionInfo.actionInfo);
    }

    public static String groupKey(OwnedOrPublishedResource resource) {
        return resource.idOrUrl.fold(MpfsResourceId::serialize, url -> url);
    }

    public Option<String> getPublicHash(DataApiUserId uid, LentaBlockRecord record) {
        Option<String> fileIdO = PublicResourceOwnedBlockFields.FILE_ID.getO(record);
        Option<String> ownerUidO = PublicResourceOwnedBlockFields.OWNER_UID.getO(record);
        String ownerUid = ownerUidO.getOrElse(uid.serialize());
        Option<MpfsFileInfo> mpfsFileInfos = fileIdO.filterMap(fid ->
                mpfsClient.getFileInfoOByFileId(uid.forMpfs(), ownerUid, fid));

        return mpfsFileInfos.filterMap(mpfsFileInfo -> mpfsFileInfo.getMeta().getPublicHash());
    }
}
