package ru.yandex.chemodan.app.lentaloader.blocks;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.test.FieldUpdateCaptor;
import ru.yandex.chemodan.app.lentaloader.test.FieldUpdateCaptor2;
import ru.yandex.chemodan.app.lentaloader.test.FieldUpdateCaptorClosure;
import ru.yandex.chemodan.eventlog.events.CompoundResourceType;
import ru.yandex.chemodan.eventlog.events.EventType;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.ratelimiter.RateLimit;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;

/**
 * @author dbrylev
 */
@RunWith(MockitoJUnitRunner.class)
public class ResourceBlockManagerTest {

    @Mock
    private LentaManager lentaManager;

    @Mock
    private MpfsClient mpfsClient;

    private ResourceBlockManager blockManager;

    @Before
    public void setup() {
        RateLimiterClient client = Mockito.mock(RateLimiterClient.class);
        Mockito.when(client.queryLimit(Mockito.anyString())).thenReturn(RateLimit.proceedWithoutDelay());

        blockManager = new ResourceBlockManager(lentaManager, client, mpfsClient);
    }

    private final DataApiUserId uid = new DataApiPassportUserId(1234567);

    private final MpfsUid mpfsUid = new MpfsUid(uid.toPassportUidOrZero().toUid());
    private final MpfsUid guestUid = new MpfsUid(7L);


    @Test
    public void publisherUid() {
        OwnedOrPublishedResource resource = OwnedOrPublishedResource.owned(
                MpfsResourceId.parse(uid.serialize() + ":file_id"), CompoundResourceType.file("document"));

        FieldUpdateCaptor2<MpfsUid, EventType, String> captor = new FieldUpdateCaptor2<>(
                PublicResourceOwnedBlockFields.PUBLISHER_UID, LentaRecordType.PUBLIC_RESOURCE_OWNED, lentaManager,
                (performer, event) -> blockManager.handleChanged(uid, resource, performer, event, true, actionInfo()));

        String uidStr = mpfsUid.getRawValue();
        String guestStr = guestUid.getRawValue();

        captor.create(mpfsUid, EventType.COMMENT_ADD).assertChangedToNone();
        captor.create(mpfsUid, EventType.FS_SET_PRIVATE).assertChangedToNone();
        captor.create(mpfsUid, EventType.FS_SET_PUBLIC).assertChangedTo(uidStr);

        captor.update(Option.empty(), mpfsUid, EventType.COMMENT_ADD).assertThrottledToNone();
        captor.update(Option.empty(), mpfsUid, EventType.FS_SET_PRIVATE).assertThrottledToNone();
        captor.update(Option.empty(), mpfsUid, EventType.FS_SET_PUBLIC).assertChangedTo(uidStr);

        captor.update(uidStr, mpfsUid, EventType.COMMENT_ADD).assertThrottledTo(uidStr);
        captor.update(uidStr, mpfsUid, EventType.FS_SET_PRIVATE).assertThrottledTo(uidStr);
        captor.update(uidStr, mpfsUid, EventType.FS_SET_PUBLIC).assertThrottledTo(uidStr);

        captor.update(uidStr, guestUid, EventType.COMMENT_ADD).assertThrottledTo(uidStr);
        captor.update(uidStr, guestUid, EventType.FS_SET_PRIVATE).assertThrottledTo(uidStr);
        captor.update(uidStr, guestUid, EventType.FS_SET_PUBLIC).assertChangedTo(guestStr);
    }

    @Test
    public void myAction() {
        OwnedOrPublishedResource resource = OwnedOrPublishedResource.published(
                "http://yadi.sk/x/", CompoundResourceType.file("document"));

        Function1V<Option<EventType>> invoker = eventO -> blockManager.handleChanged(uid, resource,
                eventO.map(e -> mpfsUid).getOrElse(guestUid),
                eventO.getOrElse(EventType.COMMENT_ADD), true, actionInfo());

        FieldUpdateCaptor<EventType, PublicResourceAction> captor = new FieldUpdateCaptor<>(
                PublicResourceBlockFields.MY_ACTION, LentaRecordType.PUBLIC_RESOURCE, lentaManager, invoker);

        captor.create(Option.empty()).assertChangedToNone();
        captor.create(EventType.ALBUM_CREATE).assertChangedToNone();
        captor.create(EventType.PUBLIC_VISIT).assertChangedTo(PublicResourceAction.VISIT);

        FieldUpdateCaptorClosure<EventType, PublicResourceAction> update;

        update = captor.update(Option.empty());

        update.with(Option.empty()).assertThrottledToNone();
        update.with(EventType.COMMENT_LIKE_ADD).assertChangedToNone();
        update.with(EventType.ALBUM_CREATE).assertChangedToNone();
        update.with(EventType.PUBLIC_VISIT).assertIgnored();

        update = captor.update(PublicResourceAction.VISIT);

        update.with(Option.empty()).assertThrottledToNone();
        update.with(EventType.COMMENT_LIKE_ADD).assertChangedTo(PublicResourceAction.LIKE);
        update.with(EventType.ALBUM_CREATE).assertChangedToNone();
        update.with(EventType.PUBLIC_VISIT).assertIgnored();

        update = captor.update(PublicResourceAction.COMMENT);

        update.with(Option.empty()).assertThrottledToNone();
        update.with(EventType.COMMENT_DELETE).assertChangedToNone();
        update.with(EventType.COMMENT_LIKE_ADD).assertChangedTo(PublicResourceAction.LIKE);
        update.with(EventType.ALBUM_CREATE).assertChangedToNone();
        update.with(EventType.PUBLIC_VISIT).assertIgnored();
    }

    private static ActionInfo actionInfo() {
        return ActionInfo.internal(ActionSource.test());
    }
}
