package ru.yandex.chemodan.app.lentaloader.blocks;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.lenta.FieldPredicate;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.CreateHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockModifyData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.eventlog.events.sharing.ShareData;

/**
 * @author dbrylev
 */
public class SharedFolderBlockManager {

    private final LentaManager lentaManager;

    public SharedFolderBlockManager(LentaManager lentaManager) {
        this.lentaManager = lentaManager;
    }

    public void handleGroupCreated(DataApiUserId uid, ShareData share, ActionInfo actionInfo) {
        lentaManager.findAndUpdateOrCreateBlock(uid, LentaBlockModifyData.createOrUpdateAndUp(
                LentaRecordType.SHARED_FOLDER, groupKey(share), Cf.toMap(Cf.list(
                        SharedFolderBlockFields.OWNER_UID.toData(share.owner.getRawValue()),
                        SharedFolderBlockFields.GROUP_ID.toData(share.groupId),
                        SharedFolderBlockFields.GROUP_CTIME.toData(actionInfo.now)))),
                actionInfo);
    }

    public void handleGroupChanged(
            DataApiUserId uid, ShareData share, Option<SharedFolderAction> action, ActionInfo actionInfo)
    {
        CreateHandler createHandler = rec -> CreateHandler.create(Cf.toMap(Cf.list(
                SharedFolderBlockFields.OWNER_UID.toData(share.owner.getRawValue()),
                SharedFolderBlockFields.GROUP_ID.toData(share.groupId))
                .plus(action.map(SharedFolderBlockFields.ACTION::toData))));

        UpdateHandler updateHandler = rec -> {
            Option<Instant> cTime = SharedFolderBlockFields.GROUP_CTIME.getO(rec);
            Option<SharedFolderAction> prevAction = SharedFolderBlockFields.ACTION.getO(rec);

            if (!cTime.exists(actionInfo.now.minus(Duration.standardDays(1))::isBefore)
                    && action.exists(a -> !prevAction.isSome(a)))
            {
                return UpdateHandler.updateAndUp(rec.specific.plus(Cf.toMap(Cf.list(
                        SharedFolderBlockFields.ACTION.toData(action.get())))));
            } else {
                return UpdateHandler.updateAndUpThrottled(rec.specific);
            }
        };

        lentaManager.findAndUpdateOrCreateBlock(uid, new LentaBlockModifyData(
                LentaRecordType.SHARED_FOLDER, groupKey(share), createHandler, updateHandler), actionInfo);
    }

    public void deleteGroup(DataApiUserId uid, ShareData share, ActionInfo actionInfo) {
        lentaManager.deleteBlocks(uid, LentaRecordType.SHARED_FOLDER, groupKey(share),
                actionInfo.withReason(ActionReason.UNSHARED_FOLDER));
    }

    public void handleInvitation(DataApiUserId uid, ShareData share, ActionInfo actionInfo) {
        lentaManager.findAndPinOrCreatePinnedBlock(uid, inviteCreateData(share), actionInfo);
    }

    public void kickInvitation(DataApiUserId uid, ShareData share, ActionInfo actionInfo) {
        Option<?> pinned = lentaManager.unpinBlockIfPinned(
                uid, LentaRecordType.SHARED_FOLDER_INVITE, groupKey(share), actionInfo);

        if (!pinned.isPresent()) {
            lentaManager.findOrCreateBlock(uid, inviteCreateData(share), actionInfo);
        }

        cleanupContentBlocks(uid, share, actionInfo);
    }

    public void acceptInvitation(DataApiUserId uid, ShareData share, ActionInfo actionInfo) {
        lentaManager.unpinBlockIfPinned(uid, LentaRecordType.SHARED_FOLDER_INVITE, groupKey(share), actionInfo);
    }

    public void declineInvitation(DataApiUserId uid, ShareData share, ActionInfo actionInfo) {
        lentaManager.deleteBlocks(uid, LentaRecordType.SHARED_FOLDER_INVITE, groupKey(share),
                actionInfo.withReason(ActionReason.SHARE_INVITATION_REJECT));

        cleanupContentBlocks(uid, share, actionInfo);
    }

    private void cleanupContentBlocks(DataApiUserId uid, ShareData share, ActionInfo actionInfo) {
        lentaManager.deleteOrUpdateBlocksAsync(
                uid, LentaRecordType.CONTENT_BLOCK,
                FieldPredicate.field(ContentBlockFields.OWNER_UID).eq(share.owner.getRawValue()),
                actionInfo.withReason(ActionReason.UNSHARED_FOLDER));

        lentaManager.deleteOrUpdateBlocksAsync(
                uid, LentaRecordType.FOLDER_CREATION,
                FolderCreationBlockManager.groupKeyPredicate(share.owner),
                actionInfo.withReason(ActionReason.UNSHARED_FOLDER));
    }

    private static String groupKey(ShareData share) {
        return share.owner.getRawValue() + ":" + share.groupId;
    }

    private static LentaBlockCreateData inviteCreateData(ShareData share) {
        return new LentaBlockCreateData(LentaRecordType.SHARED_FOLDER_INVITE, groupKey(share), Cf.toMap(Cf.list(
                SharedFolderInvitationBlockFields.OWNER_UID.toData(share.owner.getRawValue()),
                SharedFolderInvitationBlockFields.GROUP_ID.toData(share.groupId))
                .plus(share.inviteHash.map(SharedFolderInvitationBlockFields.INVITE_HASH::toData))));
    }
}
