package ru.yandex.chemodan.app.lentaloader.blocks;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.test.FieldUpdateCaptor;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.sharing.ShareData;
import ru.yandex.chemodan.mpfs.MpfsUid;

import static ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderAction.ACCEPT;
import static ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderAction.CHANGE;
import static ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderAction.DECLINE;

/**
 * @author dbrylev
 */
@RunWith(MockitoJUnitRunner.class)
public class SharedFolderBlockManagerTest {

    @Mock
    private LentaManager lentaManager;

    private SharedFolderBlockManager blockManager;

    @Before
    public void setup() {
        blockManager = new SharedFolderBlockManager(lentaManager);
    }

    private final DataApiUserId uid = new DataApiPassportUserId(1234567);
    private final MpfsUid mpfsUid = new MpfsUid(uid.toPassportUid());

    @Test
    public void ownedAction() {
        EventMetadata meta = new EventMetadata(mpfsUid, Instant.now(), Option.empty());

        ShareData data = new ShareData(meta, mpfsUid, "group", MpfsAddress.parseDir("1:/disk"), false, Option.empty());

        Function<Option<Instant>, FieldUpdateCaptor<SharedFolderAction, SharedFolderAction>> captorF =
                cTimeO -> new FieldUpdateCaptor<>(
                        SharedFolderBlockFields.ACTION, LentaRecordType.SHARED_FOLDER,
                        Cf.toMap(cTimeO.map(SharedFolderBlockFields.GROUP_CTIME::toData)), lentaManager,
                        action -> blockManager.handleGroupChanged(uid, data, action, actionInfo()));

        FieldUpdateCaptor<SharedFolderAction, SharedFolderAction> todayCaptor = captorF.apply(Option.of(Instant.now()));

        todayCaptor.create(Option.empty()).assertChangedToNone();
        todayCaptor.create(Option.of(SharedFolderAction.CHANGE)).assertChangedTo(SharedFolderAction.CHANGE);

        todayCaptor.update(Option.empty(), Option.empty()).assertThrottledToNone();
        todayCaptor.update(Option.empty(), ACCEPT).assertThrottledToNone();

        todayCaptor.update(ACCEPT, Option.empty()).assertThrottledTo(ACCEPT);
        todayCaptor.update(ACCEPT, DECLINE).assertThrottledTo(ACCEPT);

        FieldUpdateCaptor<SharedFolderAction, SharedFolderAction> dayAgoCaptor = captorF.apply(
                Option.of(Instant.now().minus(Duration.standardDays(1))));

        dayAgoCaptor.update(Option.empty(), Option.empty()).assertThrottledToNone();
        dayAgoCaptor.update(Option.empty(), ACCEPT).assertChangedTo(ACCEPT);

        dayAgoCaptor.update(ACCEPT, Option.empty()).assertThrottledTo(ACCEPT);
        dayAgoCaptor.update(ACCEPT, DECLINE).assertChangedTo(DECLINE);

        FieldUpdateCaptor<SharedFolderAction, SharedFolderAction> noCtimeCaptor = captorF.apply(Option.empty());

        noCtimeCaptor.update(Option.empty(), CHANGE).assertChangedTo(CHANGE);
        noCtimeCaptor.update(CHANGE, ACCEPT).assertChangedTo(ACCEPT);
    }

    private static ActionInfo actionInfo() {
        return ActionInfo.internal(ActionSource.test());
    }
}
