package ru.yandex.chemodan.app.lentaloader.cool;

import org.joda.time.Duration;
import org.junit.Before;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BestOfYearBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.DefaultBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThematicBlocksGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.cool.generator.WordMatch;
import ru.yandex.chemodan.app.lentaloader.cool.imageparser.ImageparserClient;
import ru.yandex.chemodan.app.lentaloader.cool.imageparser.ImageparserClientImpl;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TermDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TermLanguageDefinition;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchResponse;
import ru.yandex.chemodan.app.lentaloader.test.TestUtils;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsResourceTimes;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.db.embedded.OS;
import ru.yandex.misc.db.embedded.sandbox.SandBoxResourceRule;
import ru.yandex.misc.db.embedded.sandbox.SandBoxTestResourceResolver;
import ru.yandex.misc.time.InstantInterval;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public abstract class AbstractCoolLentaManagerTest {

    protected static final String STOP_WORD = "собор";

    protected static final long STOP_WORD_SIMILARITY_THRESHOLD = 7000L;

    protected static SandBoxResourceRule createSandboxResourceRule(String resourceId) {
        return SandBoxResourceRule
                .create(new SandBoxTestResourceResolver(Cf.map(
                        OS.ALL, resourceId
                )))
                .withSingleFileName("data.json")
                .withEnabledCache(true)
                .build();
    }

    protected static DiskSearchResponse parseTestData(SandBoxResourceRule rule) {
        return Bender.jsonParser(DiskSearchResponse.class)
                .parseJson(rule.getExtractedResourceDir().child("data.json"));
    }

    private PassportUid uid = PassportUid.cons(50273844);

    private DiskSearchClient searchClient = Mockito.mock(DiskSearchClient.class);
    private MpfsClient mpfsClient = Mockito.mock(MpfsClient.class);
    private Blackbox2 blackbox = Mockito.mock(Blackbox2.class);
    private UserTimezoneHelper userTimezoneHelper = Mockito.mock(UserTimezoneHelper.class);
    private ImageparserClient imageparserClient = Mockito.mock(ImageparserClientImpl.class);
    private CoolLentaManagerTest.DataApiManagerMock dataApiManager =
            Mockito.mock(CoolLentaManagerTest.DataApiManagerMock.class,  Mockito.CALLS_REAL_METHODS);
    private BazingaTaskManager bazingaTaskManager = Mockito.mock(BazingaTaskManager.class);
    private Cvi2tProcessor cvi2tProcessor = new Cvi2tProcessor(searchClient, Cf.list());
    private ThemeDefinitionRegistry themeDefinitionRegistry = mock(ThemeDefinitionRegistry.class);
    private final ExperimentMembersByOverrideManager experimentMembersByOverrideManager =
            mock(ExperimentMembersByOverrideManager.class);
    private ExperimentsManager experimentsManager = mock(ExperimentsManager.class);
    private CoolLentaConfigurationManager coolLentaConfigurationManager = new CoolLentaConfigurationManager();

    private CoolLentaManager manager = createManager(false, false);

    @Before
    public void initMocks() {
        initMocks(getTestData());
    }

    protected void initMocks(DiskSearchResponse testData) {
        when(searchClient.findPhotosWithBeauty(
                any(PassportUid.class),
                any(InstantInterval.class),
                any(Option.class),
                any(Integer.class),
                any(Integer.class))
        ).thenAnswer(inv -> {
            InstantInterval interval = inv.getArgument(1);
            int offset = inv.getArgument(3);
            int limit = inv.getArgument(4);

            ListF<DiskSearchFileInfo> allHits = testData.hitsArray
                    .filter(i -> i.etime.isPresent())
                    .filter(i -> i.etime.get() >= interval.getStartMillis() / 1000)
                    .filter(i -> i.etime.get() <= interval.getEndMillis() / 1000);

            return new DiskSearchResponse(
                    allHits.drop(offset).take(limit),
                    allHits.size()
            );
        });

        Function<DiskSearchFileInfo, MpfsFileInfo> mpfsFileBySearchItemF = i ->
                new MpfsFileInfo(Option.of(i.name), Option.of("file"), i.key,
                        MpfsFileMetaDto
                                .builder()
                                .resourceId(new MpfsResourceId(MpfsUid.parse(uid.toString()), i.getId()))
                                .build(),
                        new MpfsResourceTimes(0, i.etime, 0, 0)
                );

        when(mpfsClient.getFileInfoOByFileId(any(MpfsUser.class), anyString()))
                .thenAnswer(inv -> {
                    String fileId = (String) inv.getArguments()[1];
                    return testData.hitsArray.find(i -> i.id.equals(fileId)).map(mpfsFileBySearchItemF);
                });
        when(mpfsClient.bulkInfoByResourceIds(any(MpfsUser.class), any(), any(), any())).thenReturn(getMpfsTestData());
        when(searchClient.getTextCvVectors(Cf.list(STOP_WORD))).thenReturn(Cf.map(STOP_WORD, new byte[] {
                -9, -7, 17, -24, 2, 5, 0, -10, 0, 15, 11, -10, -13, 8, 1, -7, -2, 11, -13, 5, 4, -8, 1, -11, 20, 8, 8, 1,
                -1, -9, 2, 0, -7, 3, 2, 7, 5, 1, -6, -8, 12, 8, 8, -1, -2, -10, -11, 2, -10, 2, 5, 4, 0, -5, 0, -10, 10,
                -6, 19, 1, 10, -11, 14, -9, -2, -9, -20, 4, -3, 7, 10, 9, -17, -3, 2, 5, 22, -1, 3, -10, 1, 4, 1, -4, -9,
                -9, -3, 3, 11, 10, 10, -10, -13, -2, -10, -5, 21, 3, 2, -2, -1, 10, 12, 10, -6, 5, -5, 5, 1, -8, 0, -22,
                -6, -1, -5, -13, 5, 12, 0, 14, 3, 9, 8, 5, -6, 4, 9, 0, 14, -3, 4, -2, -10, 12, 24, -6, 5, -5, 0, -12, 0,
                -9, 12, 6, 4, -5, 4, 0, -1, -2, -8, 6, -17, 8, -3, 5, 0, 10, -1, 0, -8, -17, 3, 4, -2, 7, 13, 7, -16, -2,
                -1, -7, -14, -18, 7, -9, 5, -4, 1, -1, 3, 16, -3, -4, -2, -8, 22, 0, -9, 7, 5, 4, -1, 5, 0, -9, 9, 9, -3,
                -2
        }));
        when(searchClient.getTextCvVectors(Cf.list("природа"))).thenReturn(Cf.map("природа", new byte[]{
                0, -10, -2, -3, -16, 6, -1, -4, 3, 5, -5, 19, 13, 2, -7, -1, 10, 4, -7, 1, -9, 0, -4, -7, 26, 0, 18, -1,
                4, -12, -5, -5, -6, 8, 7, -12, -2, 18, 5, 6, -11, 6, 2, 9, 5, -17, -3, 4, -3, -2, 20, 5, -1, 5, 11, -8,
                10, -2, 3, 8, 4, -10, 1, -13, -4, -18, -12, 8, -9, 4, 7, 14, -11, 0, 2, -2, -12, -8, -9, 0, -3, 0, 6, 7,
                -5, -1, -12, -5, -5, -13, -2, 10, -15, -8, 3, -3, 1, -10, -5, -4, -2, 2, 1, -1, 2, 15, -6, -8, -5, 1, 5,
                0, 0, 6, 7, 1, 1, 13, 3, -5, -10, 0, 2, 7, -3, -11, 11, -5, -12, -7, -8, 0, -8, -4, 21, -6, -1, -2, 14,
                -8, 1, -6, -5, 6, -3, -1, 12, -7, -5, -8, -14, -8, -10, 1, -20, 20, 0, 4, -14, 1, -14, 3, 12, 7, 3, 3, 7,
                10, 4, -2, 10, -21, -13, -2, -14, 1, 0, 3, 14, 4, 12, 18, -4, 10, -9, -7, 0, -2, -8, -14, -10, -16, -4,
                5, 0, 2, 0, -4, 17, 14
        }));
        when(searchClient.getTextCvVectors(Cf.list("nature"))).thenReturn(Cf.map("nature", new byte[]{
                0, -15, -11, -2, -20, 11, -2, -5, 1, 0, -8, 14, 12, -4, -16, -1, 3, 3, -5, -7, -7, -2, -6, -1, 23, 3, 6,
                1, -4, 0, -7, -1, -11, -10, 7, -12, 0, 7, 8, 2, -9, 12, 2, 11, 0, -9, -4, 10, 9, -4, 15, -2, -4, 3, 10,
                -1, 6, -9, 1, 5, 8, 0, 1, -8, -3, -16, -12, -1, -8, 0, 11, 25, -6, 0, 4, -3, -5, 0, -7, 0, -8, 0, 0, 4,
                -11, -6, -13, -1, -7, -16, -2, -4, -17, -15, -5, 0, -2, -6, -3, -2, 2, 7, -3, 3, 3, 10, -11, -11, -3, 0,
                14, 2, 16, 0, 0, 8, 1, 9, -1, -1, -14, 2, 0, 7, -9, -21, 15, -3, -14, -13, -12, 1, 1, 4, 18, -6, -4, 2,
                8, -11, 1, -5, -13, 11, 0, -2, 10, -4, -8, 2, -13, -6, -6, 0, -15, 13, 1, -2, -13, 5, -3, 0, 7, 6, -5, 9,
                4, 4, 10, -5, 5, -25, 0, 3, -18, 1, 4, 1, 6, 9, 7, 18, -11, 7, -10, -14, -7, 0, -11, -12, -4, -19, 1, 3,
                5, 10, 0, 0, 10, 22
        }));
        when(themeDefinitionRegistry.getAll()).thenReturn(Cf.list(TestUtils.createNatureThemeDefinition(),
                createDisabledThemeDefinition()));
        dataApiManager.init();
    }

    protected CoolLentaManager getManager() {
        return manager;
    }

    protected PassportUid getUid() {
        return uid;
    }

    protected CoolLentaManager createManager(boolean enableSimilarityCheck, boolean enableStopWordsCheck)
    {
        return new CoolLentaManager(
                searchClient, imageparserClient, Option.empty(), mpfsClient, userTimezoneHelper,
                Cf.list(new BestOfYearBlockGenerator(), new DefaultBlockGenerator(),
                        new ThematicBlocksGenerator(themeDefinitionRegistry, cvi2tProcessor, experimentMembersByOverrideManager,
                                3, 20, false)),
                dataApiManager,
                bazingaTaskManager, coolLentaConfigurationManager, Duration.standardDays(10),
                Duration.standardDays(3), new MordaPushManager(blackbox, null, mpfsClient, dataApiManager, null, coolLentaConfigurationManager),
                cvi2tProcessor,
                enableSimilarityCheck,
                enableStopWordsCheck,
                Cf.list(STOP_WORD + ":" + STOP_WORD_SIMILARITY_THRESHOLD), experimentsManager);
    }

    protected CoolLentaManager createManagerWithMinThematicBlocksCount(int minItemsCount) {
        return new CoolLentaManager(
                searchClient, imageparserClient, Option.empty(), mpfsClient, userTimezoneHelper,
                Cf.list(new BestOfYearBlockGenerator(), new DefaultBlockGenerator(),
                        new ThematicBlocksGenerator(themeDefinitionRegistry, cvi2tProcessor, experimentMembersByOverrideManager,
                                minItemsCount, 20, false)),
                dataApiManager,
                bazingaTaskManager, coolLentaConfigurationManager, Duration.standardDays(10),
                Duration.standardDays(3), new MordaPushManager(blackbox, null, mpfsClient, dataApiManager, null, coolLentaConfigurationManager),
                cvi2tProcessor,
                false,
                false,
                Cf.list(STOP_WORD + ":" + STOP_WORD_SIMILARITY_THRESHOLD), experimentsManager);
    }

    protected Cvi2tProcessor getCvi2tProcessor() {
        return cvi2tProcessor;
    }

    protected ThemeDefinition createDisabledThemeDefinition() {
        ListF<WordMatch> words = Cf.list(new WordMatch("предмет", 100), new WordMatch("subject", 100));
        TermLanguageDefinition termLanguageDefinition = new TermLanguageDefinition("предмет", "предмета", "о", "предмете",
                "предметом", "предмета", "к предмету", Option.empty());
        return new ThemeDefinition("subject", words,
                new TermDefinition(Cf.map(Language.RUSSIAN, termLanguageDefinition)), false, Option.empty(),
                Option.empty(), Option.empty(), Option.empty());
    }

    abstract protected DiskSearchResponse getTestData();

    private ListF<MpfsFileInfo> getMpfsTestData() {
        return getTestData().hitsArray.map(this::createMpfsFileInfo);
    }

    private MpfsFileInfo createMpfsFileInfo(DiskSearchFileInfo fileInfo) {
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder().fileId(fileInfo.getId())
                .resourceId(new MpfsResourceId(new MpfsUid(getUid()), fileInfo.getId()))
                .build();
        return new MpfsFileInfo(Option.of(fileInfo.getName()), Option.of("file"),
                Option.of("/photounlim/" + fileInfo.getName()), meta, MpfsResourceTimes.ZERO);
    }
}
