package ru.yandex.chemodan.app.lentaloader.cool;

import lombok.Data;
import org.apache.commons.io.FilenameUtils;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.OrderedUUID;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BlockGeneratingResult;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BlockGeneratorUtils;
import ru.yandex.chemodan.app.lentaloader.cool.generator.CoolLentaBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.RandomBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.WordMatch;
import ru.yandex.chemodan.app.lentaloader.cool.imageparser.ImageparserClient;
import ru.yandex.chemodan.app.lentaloader.cool.log.CoolLentaBlockEvent;
import ru.yandex.chemodan.app.lentaloader.cool.log.CoolLentaEventLogger;
import ru.yandex.chemodan.app.lentaloader.cool.log.CoolLentaEventType;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaBlockFields;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaModelUtils;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaMordaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.app.lentaloader.cool.worker.CheckAndDeleteInvalidBlock;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchResponse;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.geobase.Geobase6;
import ru.yandex.inside.geobase.LinguisticsItem;
import ru.yandex.inside.geobase.RegionNode;
import ru.yandex.inside.geobase.RegionType;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author tolmalev
 */
public class CoolLentaManager {
    private static final Logger logger = LoggerFactory.getLogger(CoolLentaManager.class);

    public static final AppDatabaseRef ALL_BLOCKS_DB_REF = new AppDatabaseRef("cool_lenta", "all_blocks");
    public static final String ALL_BLOCKS_COLLECTION_ID = "all_blocks";

    public static final AppDatabaseRef MORDA_BLOCKS_DB_REF = new AppDatabaseRef("cool_lenta", "morda_blocks");
    public static final String MORDA_BLOCKS_COLLECTION_ID = "morda_blocks";

    // curl "http://mpfs.disk.yandex.net/json/list?uid=50273844&path=/share/img&meta=" | python -mjson.tool | fgrep md5
    public static final SetF<String> DEFAULT_FILES_MD5 = Cf.set(
            "e583702fdd79a70d75f2738f16129104",
            "569a1c98696050439b5b2a1ecfa52d19",
            "d27d72a3059ad5ebed7a5470459d2670",
            "1392851f0668017168ee4b5a59d66e7b",
            "ab903d9cab031eca2a8f12f37bbc9d37",
            "f1abe3b27b410128623fd1ca00a45c29",
            "a64146fee5e15b3b94c204e544426d43",
            "52faf880e17c83442646a50393436efd",
            "8d101cc936699b030090003f2f13778e",
            "098f6bcd4621d373cade4e832627b4f6",
            "52faf880e17c83442646a50393436efd",
            "20fdf8f43f20b02869a38c5323fe5ded",
            "914fe1f19c06f6aa6c25abf02fa06f80",
            "798a4e6b30ef77cb72055f63e02515e5",
            "914fe1f19c06f6aa6c25abf02fa06f80",
            "b7a353cfce207c94db8fbf2a6b1642a6"
    );
    public static final SetF<String> VALID_EXTENSIONS = Cf.set("jpg", "jpeg", "heic");

    // TODO: Get from generator? From generated block? From global config?
    private static final ListF<String> ALLOWED_SPACES = Cf.list("disk", "photounlim");
    private static final SetF<String> REQUIRED_META =
            Cf.set("mediatype", "short_url", "public_hash", "resource_id", "file_id", "pmid", "file_mid");

    private static final MapF<Integer, Duration> FETCH_FILES_FOR_RANDOM_BLOCK_RETRY_INTERVALS =
            Cf.map(1, Duration.standardDays(30),
                    2, Duration.standardDays(90),
                    3, Duration.standardDays(366));

    private final DataApiManager dataApiManager;

    private final DiskSearchClient searchClient;
    private final ImageparserClient imageparserClient;
    private final Option<Geobase6> geobase;
    private final MpfsClient mpfsClient;
    private final UserTimezoneHelper userTimezoneHelper;
    private final ListF<CoolLentaBlockGenerator> blockGenerators;
    private final BazingaTaskManager bazingaTaskManager;
    private final MordaPushManager mordaPushManager;
    private final Cvi2tProcessor cvi2tProcessor;
    private final CoolLentaConfigurationManager coolLentaConfigurationManager;
    private final ExperimentsManager experimentsManager;

    private final Duration minRepeatDuration;
    private final Duration maxLastBlockAge;

    public final DynamicProperty<Long> similarityThreshold =
            new DynamicProperty<>("cool-lenta-similarity-threshold", 11_200L);
    private final DynamicProperty<Boolean> checkOnlyBestPhotoInBlock =
            new DynamicProperty<>("cool-lenta-check-only-best", false);
    private final DynamicProperty<Boolean> enableSimilarityCheck;
    private final DynamicProperty<Boolean> enableStopWordsCheck;
    private final DynamicProperty<ListF<String>> stopWords;
    private final DynamicProperty<Integer> generationRetryCount =
            new DynamicProperty<>("cool-lenta-generation-retry-count", 10);
    private final DynamicProperty<Integer> randomGenerationRetryCount =
            new DynamicProperty<>("cool-lenta-generation-random-retry-count", 10);
    private final DynamicProperty<Long> similarityFilteringIntervalMs =
            new DynamicProperty<>("cool-lenta-manager-similarity-interval-ms", 86_400_000L);
    private final DynamicProperty<Integer> similarityFilteringMaxCount =
            new DynamicProperty<>("cool-lenta-manager-similarity-max-count", 1000);

    public CoolLentaManager(DiskSearchClient searchClient, ImageparserClient imageparserClient,
            Option<Geobase6> geobase,
            MpfsClient mpfsClient, UserTimezoneHelper userTimezoneHelper,
            ListF<CoolLentaBlockGenerator> blockGenerators,
            DataApiManager dataApiManager, BazingaTaskManager bazingaTaskManager, Duration minRepeatDuration,
            Duration maxLastBlockAge, MordaPushManager mordaPushManager, Cvi2tProcessor cvi2tProcessor,
            CoolLentaConfigurationManager coolLentaConfigurationManager, ExperimentsManager experimentsManager)
    {
        this(searchClient, imageparserClient, geobase, mpfsClient, userTimezoneHelper, blockGenerators, dataApiManager,
                bazingaTaskManager, coolLentaConfigurationManager, minRepeatDuration, maxLastBlockAge, mordaPushManager,
                cvi2tProcessor, false, false,
                Cf.list("дтп:4275", "реанимация:6204", "больница:5321", "похороны:8479", "смайлик с оружием:5082",
                        "cemetery:6147", "кладбище:6622", "могила:7777"), experimentsManager);
    }

    public CoolLentaManager(DiskSearchClient searchClient, ImageparserClient imageparserClient,
            Option<Geobase6> geobase,
            MpfsClient mpfsClient, UserTimezoneHelper userTimezoneHelper,
            ListF<CoolLentaBlockGenerator> blockGenerators,
            DataApiManager dataApiManager, BazingaTaskManager bazingaTaskManager,
            CoolLentaConfigurationManager coolLentaConfigurationManager, Duration minRepeatDuration,
            Duration maxLastBlockAge, MordaPushManager mordaPushManager, Cvi2tProcessor cvi2tProcessor,
            boolean enableSimilarityCheck, boolean enableStopWordsCheck, ListF<String> stopWords, ExperimentsManager experimentsManager)
    {
        this.searchClient = searchClient;
        this.imageparserClient = imageparserClient;
        this.geobase = geobase;
        this.mpfsClient = mpfsClient;
        this.userTimezoneHelper = userTimezoneHelper;
        this.blockGenerators = blockGenerators;
        this.dataApiManager = dataApiManager;
        this.bazingaTaskManager = bazingaTaskManager;
        this.minRepeatDuration = minRepeatDuration;
        this.maxLastBlockAge = maxLastBlockAge;
        this.mordaPushManager = mordaPushManager;
        this.cvi2tProcessor = cvi2tProcessor;
        this.coolLentaConfigurationManager = coolLentaConfigurationManager;
        this.enableSimilarityCheck = new DynamicProperty<>("cool-lenta-manager-similarity-check-enable",
                enableSimilarityCheck);
        this.enableStopWordsCheck = new DynamicProperty<>("cool-lenta-manager-stop-words-check-enable",
                enableStopWordsCheck);
        this.stopWords = new DynamicProperty<>("cool-lenta-stop-words-list", stopWords);
        this.experimentsManager = experimentsManager;
    }

    public CoolLentaBlock getExtendedBlock(PassportUid uid, MinimalCoolLentaBlock block,
            Option<String> experimentName)
    {
        DateTimeZone timezone = userTimezoneHelper.getUserTimezone(uid);

        MapF<String, DiskSearchFileInfo> searchInfoById = searchClient
                .getFilesInfoBatch(uid, block.resourceIds.map(r -> r.fileId), experimentName)
                .toMapMappingToKey(f -> f.id);

        MapF<MpfsResourceId, MpfsFileInfo> infoByResourceId = mpfsClient
                .bulkInfoByResourceIds(MpfsUser.of(uid), REQUIRED_META,
                        block.resourceIds.map(MpfsResourceId::serialize),
                        Cf.list("/disk", "/photounlim")
                ).toMapMappingToKey(f -> f.getMeta().getResourceId().get());

        ListF<MpfsResourceId> notFoundInMpfs = block.resourceIds.filter(r -> !infoByResourceId.containsKeyTs(r));
        ListF<MpfsResourceId> notFoundInSearch = block.resourceIds.filter(r -> !searchInfoById.containsKeyTs(r.fileId));

        if (notFoundInMpfs.isNotEmpty() || notFoundInSearch.isNotEmpty()) {
            throw new InconsistentCoolLentaBlockException(notFoundInMpfs, notFoundInSearch);
        }
        if (searchInfoById.values().exists(searchInfo -> !searchInfo.getEtime().isPresent())) {
            throw new InconsistentCoolLentaBlockException(
                    String.format("Block contains items with empty etime. blockId=%s", block.getId()));
        }

        ListF<CoolLentaFileItem> coolLentaFileItems = block.resourceIds.map(id -> new CoolLentaFileItem(
                uid, timezone, searchInfoById.getTs(id.fileId), infoByResourceId.getTs(id)
        ));
        Function<Instant, DateTime> instantToDateTimeFunction = instant -> new DateTime(instant, timezone);
        return new CoolLentaBlock(
                block.id,
                block.generationType,
                coolLentaFileItems,
                block.photosliceDate.map(instantToDateTimeFunction),
                block.lastShowDate.map(instantToDateTimeFunction),
                block.legacyLentaLastShowDate.map(instantToDateTimeFunction),
                block.getLegacyLentaLastPushDate().map(instantToDateTimeFunction),
                block.pushSent
        );
    }

    /**
     * @param coolLentaFileItems must be mutable
     * @param blockId
     * @throws InconsistentCoolLentaBlockException if list contains any invalid (similar or contains stop content) items
     */
    private void tryInspectBlockItemsForValidity(ListF<CoolLentaFileItem> coolLentaFileItems, String blockId)
            throws InconsistentCoolLentaBlockException
    {
        if (coolLentaFileItems.exists(item -> !hasI2tVector(item))) {
            throw new InconsistentCoolLentaBlockException(
                    String.format("Block contains files with empty i2tVector. blockId=%s", blockId)
            );
        }
        if (coolLentaFileItems.exists(item -> !item.getNewBeauty().isPresent())) {
            throw new InconsistentCoolLentaBlockException(
                    String.format("Block contains items with empty beauty value. blockId=%s", blockId));
        }
        if (!coolLentaFileItems.isSortedBy(CoolLentaFileItem::getUserEtime)) {
            throw new InconsistentCoolLentaBlockException(
                    String.format("Block is not sorted by user etime. blockId=%s", blockId));
        }
        if (enableSimilarityCheck.get() &&
                !BlockGeneratorUtils.removeSimilarItems(coolLentaFileItems, similarityThreshold.get()).isEmpty())
        {
            throw new InconsistentCoolLentaBlockException(
                    String.format("Block contains similar items. blockId=%s", blockId));
        }
        if (enableStopWordsCheck.get() &&
                !BlockGeneratorUtils.removeItemsByStopWords(coolLentaFileItems, getStopWords(), cvi2tProcessor)
                        .isEmpty())
        {
            throw new InconsistentCoolLentaBlockException(
                    String.format("Block contains stop words content. blockId=%s", blockId));
        }
    }

    public void process(PassportUid uid, Instant time, IntervalType type) {
        DateTimeZone userTimezone = getUserTimezone(uid);
        process(uid, time, userTimezone, type);
    }

    public void process(PassportUid uid, Instant time, DateTimeZone userTimezone, IntervalType type) {
        DateTime userTime = new DateTime(time, userTimezone);

        DateTime intervalStart = type.getIntervalStart(userTime);
        DateTime intervalEnd = type.getIntervalEnd(userTime);

        if (!TimeIntervalUtils.isIntervalFinished(type, intervalStart)) {
            logger.debug("Interval not finished. Skip processing");
            return;
        }

        ListF<CoolLentaBlockGenerator> generators = getActualBlockGenerators(type, uid.toUidOrZero().getUid());
        ListF<CoolLentaBlock> blocks = generateAllBlocks(uid, intervalStart, intervalEnd, generators);
        logger.debug("Generated {} blocks for uid={}, time={}, type={}", blocks.size(), uid, time, type);
        if (blocks.isNotEmpty()) {
            createOrUpdateBlocks(uid, blocks);
        }
        ListF<String> justGeneratedIds = blocks.map(b -> b.id);

        scheduleBlockCheck(uid, userTime, type, justGeneratedIds, generators);
    }

    public ListF<WordMatch> getStopWords() {
        return stopWords.get().map(this::parseStopWord);
    }

    private WordMatch parseStopWord(String source) {
        String[] blocks = source.split(":");
        if (blocks.length != 2) {
            throw new IllegalArgumentException(String.format("Wrong format of input stop word `%s`", source));
        }
        try {
            return new WordMatch(blocks[0], Long.parseLong(blocks[1]));
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException(String.format("Cannot parse similarity threshold value for word `%s`",
                    source));
        }
    }

    private void scheduleBlockCheck(PassportUid uid, DateTime userTime, IntervalType type, ListF<String> skipBlockIds,
            ListF<CoolLentaBlockGenerator> generators)
    {
        if (!IntervalType.getTypesForDateTime(userTime).containsTs(type)) {
            // disabled checking or invalid interval type for given date
            return;
        }

        Option<Database> databaseO = getAllBlocksDatabaseO(uid);
        if (!databaseO.isPresent()) {
            return;
        }

        DateTime intervalStart = type.getIntervalStart(userTime);
        logger.debug("Going to schedule blocks check for uid={}, type={}, intervalStart={}", uid, type, intervalStart);

        ListF<String> blockIds = generators.flatMap(g -> g.generateAllPossibleIds(type, intervalStart)).unique()
                .filterNot(skipBlockIds::containsTs).toList();

        dataApiManager.getRecords(databaseO.get().spec(), RecordsFilter.DEFAULT
                .withCollectionId(ALL_BLOCKS_COLLECTION_ID)
                .withRecordIdCond(RecordIdCondition.inSet(blockIds)))
                .forEach(record -> {
                    bazingaTaskManager.schedule(new CheckAndDeleteInvalidBlock(uid, record.id().recordId()));
                });
    }

    public void generateAllBlocksForDate(PassportUid uid, Instant time) {
        DateTime userTime = new DateTime(time, getUserTimezone(uid));
        ListF<CoolLentaBlock> blocks = Cf.arrayList();
        IntervalType.getTypesForDateTime(userTime)
                .forEach(type -> blocks.addAll(
                        generateAllBlocks(uid, type.getIntervalStart(userTime), type.getIntervalEnd(userTime),
                                getActualBlockGenerators(type, uid.toUidOrZero().getUid()))
                ));
        createOrUpdateBlocks(uid, blocks);
    }

    ListF<CoolLentaBlock> generateAllBlocks(PassportUid uid, DateTime intervalStart,
            DateTime intervalEnd, ListF<CoolLentaBlockGenerator> generators)
    {
        if (intervalStart.getMillis() < 0 || intervalEnd.getMillis() < 0) {
            // skip dates before year 1970
            return Cf.list();
        }
        ListF<CoolLentaBlock> result = Cf.arrayList();
        double beautyLimit = blockGenerators.map(CoolLentaBlockGenerator::getBeautyLimit).min();
        ListF<CoolLentaFileItem> files =
                getFiles(uid, new InstantInterval(intervalStart, intervalEnd), Option.of("cool-lenta"), beautyLimit);

        generators.forEach(generator -> result.addAll(generateBlockWithRetries(generator, files, uid)));
        return result;
    }

    public ListF<CoolLentaBlockGenerator> getActualBlockGenerators(IntervalType type, long uid) {
        return blockGenerators.filter(g -> g.isEnableForUid(uid) && g.isAcceptableIntervalType(type));
    }

    private ListF<CoolLentaBlock> generateBlockWithRetries(CoolLentaBlockGenerator generator,
            ListF<CoolLentaFileItem> sourceItems, PassportUid uid)
    {
        return generateBlockContentWithRetries(generator, sourceItems, uid).map(blockGeneratingResult -> {
            ListF<CoolLentaFileItem> items = blockGeneratingResult.getItems();
            Tuple2<IntervalType, DateTime> intervalWithDate = BlockGeneratorUtils.getInterval(items);
            IntervalType intervalType = intervalWithDate._1;
            DateTime intervalStart = intervalWithDate._2;

            String blockId = generator.generateBlockId(intervalType, intervalStart,
                    blockGeneratingResult.getBlockIdPostfix());

            CoolLentaBlock coolLentaBlock = new CoolLentaBlock(
                    blockId, generator.getGenerationType(intervalType), items,
                    Option.when(intervalType != IntervalType.YEAR, items.first().getUserEtime()),
                    Option.empty(), Option.empty(), Option.empty(), false
            );

            logBlockGeodata(coolLentaBlock);

            return coolLentaBlock;
        });
    }

    private void logBlockGeodata(CoolLentaBlock block) {
        geobase.forEach(geobase -> {
            try {
                Tuple2<ListF<CoolLentaFileItem>, ListF<CoolLentaFileItem>> t2 =
                        block.items.partition(item ->
                                item.searchFileInfo.latitude.isPresent() && item.searchFileInfo.longitude.isPresent());

                ListF<CoolLentaFileItem> itemsWithGeo = t2._1;
                ListF<CoolLentaFileItem> itemsWithoutGeo = t2._2;

                ListF<RegionNode> regionNodes = itemsWithGeo
                        .map(i -> new Coordinates(i.searchFileInfo.latitude.get(), i.searchFileInfo.longitude.get()))
                        .flatMap(geobase::getRegionIdByCoordinates)
                        .flatMap(geobase::getRegionById);

                String geoDescription;
                String finalGeoString = "";

                if (!regionNodes.isEmpty()) {
                    geoDescription = "geo_names: " + regionNodes.map(r -> r.getId()).unique()
                            .flatMap(id -> geobase.getLinguisticsItemByRegionId(id, "ru"))
                            .map(LinguisticsItem::getNominativeCase)
                            .mkString(", ");

                    SetF<Integer> commonParents = regionNodes
                            .map(RegionNode::getId)
                            .map(id -> geobase.getParentIdsById(id).plus(id))
                            .foldLeft(null, (s, l) -> {
                                if (s == null) {
                                    return l.unique();
                                } else {
                                    return s.intersect(l.unique());
                                }
                            });

                    ListF<RegionNode> sortedCommonNodes = commonParents
                            .flatMap(geobase::getRegionById)
                            .sortedBy(r -> r.getType().ordinal());

                    geoDescription += " Common parents by type: " + sortedCommonNodes
                            .map(r -> r.getId())
                            .flatMap(id -> geobase.getLinguisticsItemByRegionId(id, "ru"))
                            .map(LinguisticsItem::getNominativeCase)
                            .mkString(" -> ");

                    Option<RegionNode> goodNodeO = sortedCommonNodes.reverse().find(rn -> {
                        return Cf.list(
                                RegionType.COUNTRY,
                                RegionType.FEDERAL_DISTRICT,
                                RegionType.FEDERAL_SUBJECT,
                                RegionType.CITY,
                                RegionType.VILLAGE
                        ).containsTs(rn.getType());
                    });

                    if (goodNodeO.isPresent()) {
                        finalGeoString = geobase.getLinguisticsItemByRegionId(goodNodeO.get().getId(), "ru").get()
                                .getNominativeCase();
                    }

                } else {
                    geoDescription = "no geodata";
                }

                logger.debug("{} Block geodata. " +
                                " block_id: {}" +
                                " total_items: {}, " +
                                " with_geo: {}, " +
                                " found_regions: {}" +
                                " without_geo: {}" +
                                " description: {}",
                        finalGeoString,
                        block.id,
                        block.items.size(),
                        itemsWithGeo.size(),
                        regionNodes.size(),
                        itemsWithoutGeo.size(),
                        geoDescription
                );
            } catch (Throwable e) {
                ExceptionUtils.throwIfUnrecoverable(e);
                logger.error("Failed to log geodata: {}", e);
            }
        });
    }

    public ListF<BlockGeneratingResult> generateBlockContentWithRetries(CoolLentaBlockGenerator generator,
            ListF<CoolLentaFileItem> items, PassportUid uid)
    {
        ListF<CoolLentaFileItem> sourceFiles = Cf.toArrayList(items);

        // try to pick some photos. Check if all the photos are present in mpfs. If not - remove them from source list
        // and try again.
        for (int i = 0; i < generationRetryCount.get(); i++) {
            logger.debug("Try to generate blocks. Try number is {}", i + 1);
            ListF<BlockGeneratingResult> blocks = generator.generateBlocks(sourceFiles, Option.of(uid));
            if (blocks.isEmpty()) {
                return Cf.list();
            }
            boolean hasInvalidBlocks = false;
            for (BlockGeneratingResult block : blocks) {
                ListF<CoolLentaFileItem> blockItems = Cf.toArrayList(block.getItems());
                bulkLoadMpfsInfo(blockItems.filter(item -> !item.mpfsFileInfo.isPresent()), ALLOWED_SPACES);
                for (CoolLentaFileItem item : blockItems) {
                    if (!item.mpfsFileInfo.isPresent()) {
                        sourceFiles.removeTs(item);
                        hasInvalidBlocks = true;
                    }
                }
                if (hasInvalidBlocks) {
                    break;
                }
                if (invalidItemsHasBeenFiltered(blockItems, sourceFiles,
                        coolLentaFileItems ->
                                BlockGeneratorUtils.removeSimilarItems(coolLentaFileItems, similarityThreshold.get()),
                        enableSimilarityCheck))
                {
                    hasInvalidBlocks = true;
                    break;
                }
                if (invalidItemsHasBeenFiltered(blockItems, sourceFiles,
                        coolLentaFileItems -> BlockGeneratorUtils
                                .removeItemsByStopWords(coolLentaFileItems, getStopWords(), cvi2tProcessor),
                        enableStopWordsCheck))
                {
                    hasInvalidBlocks = true;
                    break;
                }
            }
            if (!hasInvalidBlocks) {
                return blocks;
            }
            // after 3 retries check all the photos against MPFS, similarity and stop words conditions to prevent more retries
            if (i == 3) {
                logger.debug("Starting filter invalid items");
                filterInvalidItems(sourceFiles, ALLOWED_SPACES);
                continue;
            }
        }
        logger.debug("The retry limit {} has been exceeded", generationRetryCount.get());
        return Cf.list();
    }

    public ListF<CoolLentaFileItem> getFiles(PassportUid uid, InstantInterval interval, Option<String> expName,
            double beautyLimit)
    {
        DateTimeZone timezone = getUserTimezone(uid);

        int maxCount = 50000;
        int offset = 0;

        ListF<CoolLentaFileItem> result = Cf.arrayList();
        while (true) {
            DiskSearchResponse part = searchClient.findPhotosWithBeauty(uid, interval, expName, offset, maxCount);
            ListF<CoolLentaFileItem> filtered = part.hitsArray
                    // filter trash without mpfs request - much faster
                    .filter(DiskSearchFileInfo::isInDiskOrPhotounlim)
                    .filter(item -> item.etime.isPresent())
                    .filter(item -> hasOkExtension(item.name))
                    .filter(item -> !item.md5.isPresent() || !DEFAULT_FILES_MD5.containsTs(item.md5.get()))
                    .map(info -> new CoolLentaFileItem(uid, timezone, info))
                    .filter(item -> isBeautyOk(item, beautyLimit))
                    .filter(CoolLentaManager::hasI2tVector)
                    .filter(item -> isPornoOkOrNotPresent(item.searchFileInfo));

            result.addAll(filtered);
            offset += part.hitsArray.size();
            if (part.hitsArray.size() < maxCount) {
                break;
            }
        }
        logger.debug("Filtered by beautiful and etime: start_count={}, final_count={}", offset, result.length());
        return result.sortedBy(CoolLentaFileItem::getUserEtime);
    }

    public CoolLentaBlock createRandomBlockForUser(PassportUid uid, int count, DateTime userDateTime) {
        ListF<CoolLentaFileItem> userFiles = getFilesForRandomBlock(uid, userDateTime, count);

        MapF<Integer, ListF<CoolLentaFileItem>> itemsByYear =
                Cf.toHashMap(userFiles.groupBy(item -> item.getUserEtime().getYear())
                        .mapValues(Cf::toArrayList));
        if (itemsByYear.isEmpty()) {
            throw new IllegalStateException(String.format("No items for user uid=%s", uid));
        }
        ListF<CoolLentaFileItem> validBlockItems = null;
        CoolLentaBlockGenerator generator = new RandomBlockGenerator(count);

        for (int i = 0; i < randomGenerationRetryCount.get(); i++) {
            Integer year = Random2.R
                    .weightedRandomElements(itemsByYear.keys(), yearValue -> itemsByYear.getTs(yearValue).size(), 1)
                    .first();
            userFiles = itemsByYear.getTs(year);
            ListF<CoolLentaFileItem> sortedByEtimeUserFiles = userFiles.sortedBy(CoolLentaFileItem::getUserEtime);
            logger.debug("Starting generation random block allYears={} selectedYear={} selectedItemsCount={}",
                    itemsByYear.keys(), year, userFiles.size());
            BlockGeneratingResult blockGeneratingResult =
                    generator.generateBlocks(sortedByEtimeUserFiles, Option.of(uid)).firstO()
                            .getOrThrow(() -> new IllegalStateException(
                                    String.format("Empty block generation result for random generation uid=%s", uid)));
            ListF<CoolLentaFileItem> blockItems = blockGeneratingResult.getItems();
            bulkLoadMpfsInfo(blockItems, ALLOWED_SPACES);
            boolean valid = true;
            for (CoolLentaFileItem item : blockItems) {
                if (item.getMpfsFileInfo().isPresent()) {
                    continue;
                }
                valid = false;
                userFiles.removeTs(item);
            }
            if (!valid) {
                if (userFiles.isEmpty()) {
                    itemsByYear.removeTs(year);
                }
                if (itemsByYear.isEmpty()) {
                    throw new IllegalStateException(String.format("Cannot generate random block for uid=%s", uid));
                }
                continue;
            }
            validBlockItems = blockItems;
            break;
        }
        if (validBlockItems == null) {
            throw new IllegalStateException(
                    String.format("Cannot generate random block (tries exceeding) for uid=%s tries=%s",
                            uid, randomGenerationRetryCount.get()));
        }
        Tuple2<IntervalType, DateTime> intervalWithDate = BlockGeneratorUtils.getInterval(validBlockItems);
        IntervalType intervalType = intervalWithDate._1;
        DateTime intervalStart = intervalWithDate._2;

        String blockId = generator.generateBlockId(intervalType, intervalStart,
                Option.empty());
        return new CoolLentaBlock(
                blockId, generator.getGenerationType(intervalType), validBlockItems,
                Option.when(intervalType != IntervalType.YEAR, validBlockItems.first().getUserEtime()),
                Option.empty(), Option.empty(), Option.empty(), false
        );
    }

    private ListF<CoolLentaFileItem> getFilesForRandomBlock(PassportUid uid, DateTime userNow, int filesCount) {
        int maxCount = 10000;
        int offset = 0;

        int retryNumber = 1;
        while (true) {
            FetchFilesRandomBlockRetryData retryData = getRetryData(retryNumber, userNow.toInstant());
            ListF<CoolLentaFileItem> userFiles = Cf.arrayList();
            while (true) {
                DiskSearchResponse part = searchClient.findPhotosWithBeauty(uid,
                        new InstantInterval(retryData.getStartTime(), userNow), Option.empty(), offset, maxCount);
                ListF<CoolLentaFileItem> filtered = part.hitsArray
                        .filter(DiskSearchFileInfo::isInDiskOrPhotounlim)
                        .filter(item -> item.etime.isPresent())
                        .filter(info -> info.getBeautiful2().isPresent())
                        .filter(item -> hasOkExtension(item.name))
                        .filter(item -> !item.md5.isPresent() || !DEFAULT_FILES_MD5.containsTs(item.md5.get()))
                        .map(info -> new CoolLentaFileItem(uid, userNow.getZone(), info))
                        .filter(item -> isPornoOkOrNotPresent(item.searchFileInfo));

                userFiles.addAll(filtered);
                offset += part.hitsArray.size();
                if (part.hitsArray.size() < maxCount) {
                    break;
                }
            }
            if (userFiles.size() >= filesCount || retryData.isLastRetry()) {
                logger.debug("Filtered by beautiful2 and etime for random block: start_count={}, final_count={}",
                        offset, userFiles.length());
                return userFiles;
            }
            retryNumber++;
        }
    }

    private FetchFilesRandomBlockRetryData getRetryData(int retryNumber, Instant now) {
        return FETCH_FILES_FOR_RANDOM_BLOCK_RETRY_INTERVALS.getO(retryNumber)
                .map(duration -> new FetchFilesRandomBlockRetryData(now.minus(duration), false))
                .getOrElse(() -> new FetchFilesRandomBlockRetryData(new Instant(0L), true));
    }

    public static boolean hasI2tVector(CoolLentaFileItem item) {
        byte[] i2tVector = item.getSearchFileInfo().geti2tVector();
        return i2tVector != null && i2tVector.length > 0;
    }

    private boolean hasOkExtension(String name) {
        String extension = FilenameUtils.getExtension(name).toLowerCase();
        return VALID_EXTENSIONS.containsTs(extension);
    }

    public boolean isBeautyOk(CoolLentaFileItem item, double beautyLimit) {
        return item.getNewBeauty().filter(beauty -> beauty >= beautyLimit).isPresent();
    }

    private void filterInvalidItems(ListF<CoolLentaFileItem> itemsToFilter, ListF<String> allowedSpaces) {
        logger.debug("Filtering all items. Initial size {}", itemsToFilter.size());
        itemsToFilter.filter(sf -> !sf.mpfsFileInfo.isPresent()).paginate(100)
                .forEach(page -> bulkLoadMpfsInfo(page, allowedSpaces));
        itemsToFilter.removeIf(photo -> !photo.mpfsFileInfo.isPresent());
        logger.debug("Items size after invalid MPFS files is {}", itemsToFilter.size());
        removeInvalidItemsByRemover(itemsToFilter, enableSimilarityCheck,
                items -> BlockGeneratorUtils.removeSimilarItemsInWindow(items, similarityThreshold.get(),
                        similarityFilteringIntervalMs.get(), similarityFilteringMaxCount.get()), "similarity");
        removeInvalidItemsByRemover(itemsToFilter, enableStopWordsCheck,
                items -> BlockGeneratorUtils.removeItemsByStopWords(items, getStopWords(), cvi2tProcessor),
                "stop words");
    }

    private void removeInvalidItemsByRemover(ListF<CoolLentaFileItem> items, DynamicProperty<Boolean> enableRemover,
            Function<ListF<CoolLentaFileItem>, ListF<CoolLentaFileItem>> remover, String removerName)
    {
        if (enableRemover.get()) {
            ListF<CoolLentaFileItem> removedItems = remover.apply(items);
            logger.debug("{} items has been removed after {} filtering. New size {}",
                    removedItems.size(), removerName, items.size());
            return;
        }
        logger.debug("{} is disabled", removerName);
    }

    private boolean invalidItemsHasBeenFiltered(ListF<CoolLentaFileItem> itemsToFilter,
            ListF<CoolLentaFileItem> itemsWhereRemove,
            Function<ListF<CoolLentaFileItem>, ListF<CoolLentaFileItem>> filteringFunction,
            DynamicProperty<Boolean> enableFlag)
    {
        if (!enableFlag.get()) {
            return false;
        }
        ListF<CoolLentaFileItem> itemsToRemove = filteringFunction.apply(itemsToFilter);
        if (itemsToRemove.isEmpty()) {
            return false;
        }
        itemsWhereRemove.removeAllTs(itemsToRemove);
        return true;
    }

    private void bulkLoadMpfsInfo(ListF<CoolLentaFileItem> items, ListF<String> allowedSpaces) {
        if (items.isEmpty()) {
            return;
        }
        PassportUid uid = items.first().uid;

        ListF<String> resourceIds = items.map(item -> uid.toString() + ":" + item.searchFileInfo.getId());

        MapF<String, MpfsFileInfo> mpfsInfoByFileId = mpfsClient.bulkInfoByResourceIds(
                MpfsUser.of(uid), REQUIRED_META, resourceIds, allowedSpaces.map(space -> "/" + space))
                .toMapMappingToKey(info -> info.getMeta().getFileId().get());

        for (CoolLentaFileItem item : items) {
            String fileId = item.searchFileInfo.getId();

            Option<MpfsFileInfo> infoO = mpfsInfoByFileId.getO(fileId);

            if (infoO.isPresent()) {
                Option<String> path = infoO.get().path;
                if (!path.isPresent()) {
                    continue;
                }

                Option<PassportUid> ownerUid =
                        infoO.get().getMeta().getResourceId().map(r -> r.owner).map(MpfsUid::getUid);
                if (!ownerUid.isSome(uid)) {
                    continue;
                }

                if (!allowedSpaces.exists(space -> path.get().startsWith("/" + space))) {
                    continue;
                }
                Option<MpfsResourceId> resourceId = infoO.get().getMeta().getResourceId();
                if (resourceId.isPresent()) {
                    item.mpfsFileInfo = infoO;
                } else {
                    logger.warn("Failed to clear file in mpfs: {}:{} ({})", uid, fileId);
                }
            }
        }
    }

    protected DateTimeZone getUserTimezone(PassportUid uid) {
        return userTimezoneHelper.getUserTimezone(uid);
    }

    private void createOrUpdateBlocks(PassportUid puid, ListF<CoolLentaBlock> blocks) {
        if (blocks.isEmpty()) {
            return;
        }

        Database database = getOrCreateAllBlocksDatabase(puid);
        ListF<MinimalCoolLentaBlock> minimalBlocks = blocks.map(CoolLentaBlock::toMinimalBlock);

        ListF<MinimalCoolLentaBlock> existingBlocks = findBlocksInDb(database, blocks.map(CoolLentaBlock::getId));
        MapF<String, MinimalCoolLentaBlock> existingById =
                existingBlocks.toMapMappingToKey(MinimalCoolLentaBlock::getId);

        ListF<MinimalCoolLentaBlock> blocksToUpdate = minimalBlocks.flatMap(block -> {
            Option<MinimalCoolLentaBlock> existingBlockO = existingById.getO(block.getId());

            if (!existingBlockO.isPresent()) {
                return Option.of(block);
            }

            if (block.hasSameData(existingBlockO.get())) {
                return Option.empty();
            }
            if (existingBlockO.get().lastShowDate.isPresent() ||
                    existingBlockO.get().legacyLentaLastShowDate.isPresent())
            {
                block = block.withLastShowDates(
                        existingBlockO.get().lastShowDate,
                        existingBlockO.get().legacyLentaLastShowDate
                );
            }
            return Option.of(block);
        });

        ListF<RecordChange> changes = blocksToUpdate.map(CoolLentaModelUtils::blockToRecordChangeForAllBlocks);
        ListF<CoolLentaBlockEvent> events = blocksToUpdate.map(block -> new CoolLentaBlockEvent(puid,
                existingById.containsKeyTs(block.id)
                        ? CoolLentaEventType.ALL_BLOCKS_UPDATE
                        : CoolLentaEventType.ALL_BLOCKS_CREATE,
                block)
        );

        if (changes.size() > 0) {
            logger.debug("Updating {} blocks in DB", blocksToUpdate.size());
            long rev = dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes)).rev;
            CoolLentaEventLogger.log(events.map(e -> e.withRevision(rev)));
        }
    }

    /**
     * - Блок в all_blocks помечается датой последнего показа равной Instant.now()
     * - Блок добавляется в morda_blocks с максимальным id
     * - Eсли в morda_blocks уже был этот блок (с таким же block.id) - он удаляется из morda_blocks
     * - Если после изменений в morda_blocks будет больше чем MAX_MORDA_BLOCKS_COUNT - удаляем все лишнее начиная с самых старых
     *
     * @param uid
     * @param block
     * @return true if block was added
     */
    public Option<String> saveBlockForMorda(PassportUid uid, MinimalCoolLentaBlock block) {
        logger.debug("Saving block for morda. uid={}, id={}", uid, block.getId());
        int maxMordaBlocksCount = coolLentaConfigurationManager.maxMordaBlocksCount.get();

        MinimalCoolLentaMordaBlock mordaBlock = block.toMordaBlock(OrderedUUID.generateOrderedUUID(), Instant.now());
        updateLastShowDate(uid, block.id, Instant.now());

        Database mordaDatabase = getOrCreateMordaBlocksDatabase(uid);

        // we need it to remove same block if present
        ListF<DataRecord> existingMordaBlocks = dataApiManager.getRecords(mordaDatabase.spec(), RecordsFilter.DEFAULT
                .withCollectionId(MORDA_BLOCKS_COLLECTION_ID)
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_DESC)
                .withLimits(SqlLimits.first(maxMordaBlocksCount))
        );

        // не добавляем блок если в последних 5 уже есть блок с такой же заглавной картинкой
        if (existingMordaBlocks.take(5).exists(r -> CoolLentaBlockFields.BEST_RESOURCE_ID.getO(r)
                .isSome(block.bestResourceId.serialize())))
        {
            logger.debug("Block with same best_resource_id exists");
            return Option.empty();
        }

        MapF<String, String> idByAllBlocksId
                = existingMordaBlocks
                .toMap(r -> Tuple2.tuple(CoolLentaBlockFields.ALL_BLOCKS_ID.get(r), r.id.recordId()));
        MapF<String, DataRecord> blockByAllBlocksId =
                existingMordaBlocks.toMapMappingToKey(CoolLentaBlockFields.ALL_BLOCKS_ID::get);
        MapF<String, DataRecord> blockByBlocksId = existingMordaBlocks.toMapMappingToKey(r -> r.id.recordId());

        ListF<RecordChange> changes = Cf.arrayList(CoolLentaModelUtils.blockToRecordChangeForMordaBlocks(mordaBlock));
        ListF<CoolLentaBlockEvent> events = Cf.arrayList(new CoolLentaBlockEvent(
                uid, CoolLentaEventType.MORDA_BLOCKS_ADD, mordaBlock
        ));

        int targetSize = existingMordaBlocks.size() + 1;

        if (blockByAllBlocksId.containsKeyTs(block.getId())) {
            // Found block with save id in list of existing. Remove it.
            String mordaBlockId = idByAllBlocksId.getTs(block.getId());
            changes.add(CoolLentaModelUtils.removeMordaBlockChange(mordaBlockId));
            events.add(new CoolLentaBlockEvent(
                    uid, CoolLentaEventType.MORDA_BLOCKS_DELETE,
                    CoolLentaModelUtils.recordToMordaBlock(blockByAllBlocksId.getTs(block.id))
            ));
            targetSize--;
        }

        if (targetSize > maxMordaBlocksCount) {
            // max limit reached - remove blocks
            ListF<String> recordsIdToRemove = existingMordaBlocks.map(r -> r.id.recordId())
                    .sorted()
                    .take(targetSize - maxMordaBlocksCount);

            changes.addAll(recordsIdToRemove.map(CoolLentaModelUtils::removeMordaBlockChange));
            events.addAll(recordsIdToRemove
                    .map(recordId -> new CoolLentaBlockEvent(uid, CoolLentaEventType.MORDA_BLOCKS_DELETE,
                            CoolLentaModelUtils.recordToMordaBlock(blockByBlocksId.getTs(recordId)))));
        }

        long rev = dataApiManager.applyDelta(mordaDatabase, RevisionCheckMode.PER_RECORD, new Delta(changes)).rev;
        CoolLentaEventLogger.log(events.map(event -> event.withRevision(rev)));
        return Option.of(mordaBlock.id);
    }

    public void updateLastShowDate(PassportUid uid, String blockId, Instant time) {
        logger.debug("Update last block show time. uid={}, id={}, time={}", uid, blockId, time);

        Database database = getOrCreateAllBlocksDatabase(uid);
        RecordChange changes = CoolLentaModelUtils.updateLastShowDate(blockId, time);
        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    public void updatePushSent(PassportUid uid, String blockId, boolean pushSent) {
        logger.debug("Update block pushSent. uid={}, id={}, pushSent={}", uid, blockId, pushSent);

        Database database = getOrCreateAllBlocksDatabase(uid);
        RecordChange changes = CoolLentaModelUtils.updatePushSent(blockId, pushSent);
        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    public void updateLegacyLentaDates(PassportUid uid, String blockId, Option<Instant> sendDate,
            Option<Instant> pushDate)
    {
        ListF<RecordChange> changes = Cf.list(
                sendDate.map(date -> CoolLentaModelUtils.updateLegacyLentaLastShowDate(blockId, date)),
                pushDate.map(date -> CoolLentaModelUtils.updateLegacyLentaLastPushDate(blockId, date))
        ).filter(Option::isPresent).map(Option::get);
        if (changes.isEmpty()) {
            return;
        }
        Database database = getOrCreateAllBlocksDatabase(uid);
        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    public Database getOrCreateMordaBlocksDatabase(PassportUid uid) {
        return dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(new DataApiPassportUserId(uid),
                MORDA_BLOCKS_DB_REF));
    }


    private Database getOrCreateAllBlocksDatabase(PassportUid uid) {
        return dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(new DataApiPassportUserId(uid),
                ALL_BLOCKS_DB_REF));
    }

    private Option<Database> getAllBlocksDatabaseO(PassportUid uid) {
        return dataApiManager.getDatabaseO(new UserDatabaseSpec(new DataApiPassportUserId(uid), ALL_BLOCKS_DB_REF));
    }

    private ListF<MinimalCoolLentaBlock> findBlocksInDb(Database database, ListF<String> blockIds) {
        return dataApiManager.getRecords(database.spec(), RecordsFilter.DEFAULT
                .withCollectionId(ALL_BLOCKS_COLLECTION_ID)
                .withRecordIdCond(RecordIdCondition.inSet(blockIds))
        ).map(CoolLentaModelUtils::recordToBlock);
    }

    public ListF<MinimalCoolLentaBlock> getRandomBlocks(PassportUid uid, int count) {
        //TODO: optimize random selection - select random from DB

        return Random2.R.randomElements(getAllBlocks(uid), count);
    }

    public ListF<MinimalCoolLentaBlock> getAllBlocks(PassportUid uid) {
        UserDatabaseSpec spec = new UserDatabaseSpec(new DataApiPassportUserId(uid), ALL_BLOCKS_DB_REF);

        return dataApiManager
                .getRecords(spec, RecordsFilter.DEFAULT.withCollectionId(ALL_BLOCKS_COLLECTION_ID))
                .map(CoolLentaModelUtils::recordToBlock);
    }

    public ListF<MinimalCoolLentaBlock> getAllMordaBlocks(PassportUid uid) {
        UserDatabaseSpec spec = new UserDatabaseSpec(new DataApiPassportUserId(uid), MORDA_BLOCKS_DB_REF);

        return dataApiManager
                .getRecords(spec, RecordsFilter.DEFAULT.withCollectionId(MORDA_BLOCKS_COLLECTION_ID))
                .map(CoolLentaModelUtils::recordToBlock);
    }

    private boolean isPornoOkOrNotPresent(DiskSearchFileInfo info) {
        if (info.gruesome.isPresent() && info.binaryPorn.isPresent() && info.mobilePorn.isPresent()) {
            return !isBadByCv(
                    info.mobilePorn.get(),
                    info.binaryPorn.get(),
                    info.gruesome.get()
            );
        }
        return true;
    }

    private boolean isBadByCv(double mobilePorn, double binaryPorn, double gruesome) {
        // https://st.yandex-team.ru/CV-1115#5c4861a82f38ef001f5b7524
        // https://st.yandex-team.ru/CV-852#5afd5702dbed4f001b117d86
        return mobilePorn > 0.37 || binaryPorn > 0.86 || gruesome > 0.85;
    }

    public void removeAllBlockForDebug(PassportUid uid) {
        dataApiManager.deleteDatabaseIfExists(new UserDatabaseSpec(new DataApiPassportUserId(uid), ALL_BLOCKS_DB_REF),
                DatabaseDeletionMode.REMOVE_COMPLETELY);

        dataApiManager.deleteDatabaseIfExists(new UserDatabaseSpec(new DataApiPassportUserId(uid), MORDA_BLOCKS_DB_REF),
                DatabaseDeletionMode.REMOVE_COMPLETELY);
    }

    public void checkInvalidBlock(PassportUid uid, String blockId) {
        Database database = getOrCreateAllBlocksDatabase(uid);

        ListF<DataRecord> blockRecords = dataApiManager.getRecords(database.spec(), RecordsFilter.DEFAULT
                .withCollectionId(ALL_BLOCKS_COLLECTION_ID)
                .withRecordIdCond(RecordIdCondition.eq(blockId)));

        ListF<MinimalCoolLentaBlock> blocks = blockRecords.map(CoolLentaModelUtils::recordToBlock);
        DateTimeZone timeZone = getUserTimezone(uid);

        if (blocks.isNotEmpty()) {
            MinimalCoolLentaBlock block = blocks.first();
            try {
                CoolLentaBlock extendedBlock = getExtendedBlock(uid, block, Option.empty());
                tryInspectBlockItemsForValidity(Cf.toArrayList(extendedBlock.getItems()), extendedBlock.getId());
                // block is ok, do nothing
            } catch (InconsistentCoolLentaBlockException e) {
                logger.debug("Block is invalid because of '{}'", e.getMessage());
                long rev = dataApiManager.applyDelta(database.spec(), blockRecords.first().rev,
                        RevisionCheckMode.PER_RECORD, new Delta(RecordChange.delete(ALL_BLOCKS_COLLECTION_ID, blockId)))
                        .rev;

                CoolLentaEventLogger.log(new CoolLentaBlockEvent(uid, CoolLentaEventType.ALL_BLOCKS_DELETE, block,
                        Option.of(rev)));

                Function<Instant, DateTime> instantToDateTimeFunction = instant -> new DateTime(instant, timeZone);
                tryToRegenerateBlockForUser(uid, blockId, block.getLastShowDate().map(instantToDateTimeFunction),
                        block.getLegacyLentaLastShowDate().map(instantToDateTimeFunction),
                        block.getLegacyLentaLastPushDate().map(instantToDateTimeFunction));
            }
        }
    }

    private void tryToRegenerateBlockForUser(PassportUid uid, String blockId, Option<DateTime> lastShowDate,
            Option<DateTime> legacyLastShowDate, Option<DateTime> legacyLastPushDate)
    {
        logger.debug("Try to regenerate block for user uid={} blockId={}", uid, blockId);
        Option<Tuple3<CoolLentaBlockGenerator, IntervalType, Instant>> blockGenerationDataO = parseBlockIdSafe(blockId);
        if (!blockGenerationDataO.isPresent()) {
            return;
        }
        Tuple3<CoolLentaBlockGenerator, IntervalType, Instant> blockGenerationData = blockGenerationDataO.get();
        CoolLentaBlockGenerator generator = blockGenerationData._1;
        if (!generator.isEnableForUid(uid.toUidOrZero().getUid())) {
            logger.debug("The generator is disabled for user uid={} blockId={}");
            return;
        }
        Instant instant = blockGenerationData._3;
        DateTime userTime = new DateTime(instant, getUserTimezone(uid));
        IntervalType intervalType = blockGenerationData._2;
        if (intervalType == IntervalType.WEEKEND) {
            //user could change the timezone and Saturday could become Friday. Trying to prevent it here
            userTime = getUserTimeForClosestWeekend(userTime);
        }
        DateTime intervalStart = intervalType.getIntervalStart(userTime);
        DateTime intervalEnd = intervalType.getIntervalEnd(userTime);
        ListF<CoolLentaBlock> blocks = generateAllBlocks(uid, intervalStart, intervalEnd, Cf.list(generator));
        logger.debug("Generated {} blocks for uid={}, time={}, type={}", blocks.size(), uid, instant, intervalType);
        Option<CoolLentaBlock> regeneratedBlockO = blocks.find(block -> blockId.equals(block.getId()));
        if (!regeneratedBlockO.isPresent()) {
            logger.debug("The block has not been regenerated. uid={} blockId={}", uid, blockId);
            createOrUpdateBlocks(uid, blocks);
            return;
        }
        CoolLentaBlock regeneratedBlock =
                regeneratedBlockO.get().withLastShowAndPushDates(lastShowDate, legacyLastShowDate,
                        legacyLastPushDate);
        blocks = Cf.toArrayList(blocks);
        blocks.removeIf(block -> blockId.equals(block.getId()));
        blocks.add(regeneratedBlock);
        createOrUpdateBlocks(uid, blocks.unmodifiable());
        logger.debug("The block has been regenerated successfully. uid={} blockId={}", uid, blockId);
    }

    private DateTime getUserTimeForClosestWeekend(DateTime currentTime) {
        if (currentTime.getDayOfWeek() == 6) {
            return currentTime;
        }
        return currentTime.plusDays(1);
    }

    private Option<Tuple3<CoolLentaBlockGenerator, IntervalType, Instant>> parseBlockIdSafe(String blockId) {
        Option<CoolLentaBlockGenerator> generatorO = blockGenerators
                .find(g -> isAcceptableForBlockIdPart(g, CoolLentaBlockGenerator::generatorIdPrefix, blockId));
        if (!generatorO.isPresent()) {
            logger.debug("Cannot find generator for id {}", blockId);
            return Option.empty();
        }
        CoolLentaBlockGenerator generator = generatorO.get();
        String blockIdWithoutGeneratorPrefix = blockId.substring(generator.generatorIdPrefix().length() + 1);
        Option<IntervalType> intervalTypeO = Cf.x(IntervalType.values())
                .find(intervalType -> isAcceptableForBlockIdPart(intervalType, IntervalType::toBlockIdPart,
                        blockIdWithoutGeneratorPrefix));
        if (!intervalTypeO.isPresent()) {
            logger.debug("Cannot parse interval for block id {}. Value to find interval {}",
                    blockId, blockIdWithoutGeneratorPrefix);
            return Option.empty();
        }
        IntervalType intervalType = intervalTypeO.get();
        String blockIdWithoutGeneratorPrefixAndIntervalType = blockIdWithoutGeneratorPrefix
                .substring(intervalType.toBlockIdPart().length() + 1);
        String millis = blockIdWithoutGeneratorPrefixAndIntervalType;
        if (millis.contains(CoolLentaBlockGenerator.BLOCK_ID_SEPARATOR)) {
            millis = millis.substring(0, millis.indexOf(CoolLentaBlockGenerator.BLOCK_ID_SEPARATOR));
        }
        Option<Long> instantMillisO = Cf.Long.parseSafe(millis);
        if (!instantMillisO.isPresent()) {
            logger.debug("Cannot parse instant millis for block id {}. Instant value {}", blockId, millis);
            return Option.empty();
        }
        Instant instant = instantMillisO.map(Instant::new).get();

        return Option.of(new Tuple3<>(generator, intervalType, instant));
    }

    private <T> boolean isAcceptableForBlockIdPart(T item, Function<T, String> blockIdPartPrefixGetter,
            String blockIdPart)
    {
        String blockIdPartPrefix = blockIdPartPrefixGetter.apply(item);
        int blockIdPartPrefixLength = blockIdPartPrefix.length();
        return blockIdPart.length() > blockIdPartPrefixLength &&
                blockIdPart.startsWith(blockIdPartPrefix) &&
                blockIdPart.substring(blockIdPartPrefixLength).startsWith(CoolLentaBlockGenerator.BLOCK_ID_SEPARATOR);
    }

    @Data
    private static class FetchFilesRandomBlockRetryData {
        private final Instant startTime;
        private final boolean lastRetry;
    }
}
