package ru.yandex.chemodan.app.lentaloader.cool;

import org.joda.time.Instant;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.PatchableSnapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchResponse;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.embedded.sandbox.SandBoxResourceRule;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.random.Random2;


/**
 * @author tolmalev
 */
public class CoolLentaManagerTest extends AbstractCoolLentaManagerTest {
    @ClassRule
    public static final SandBoxResourceRule resourceRule =
            createSandboxResourceRule("743403010");

    private static DiskSearchResponse TEST_DATA;

    @BeforeClass
    public static void init() {
        TEST_DATA = parseTestData(resourceRule);
    }

    @Test
    public void simpleGenerateBlock() {
        getManager().generateAllBlocksForDate(getUid(), Instant.parse("2018-09-12T12:00:00"));
    }

    @Override
    protected DiskSearchResponse getTestData() {
        return TEST_DATA;
    }

    // Mock object only to check limits
    static abstract class DataApiManagerMock implements DataApiManager {
        private MapF<DatabaseRef, Database> databases;
        private MapF<DatabaseRef, ListF<DataRecord>> records;

        public void init() {
            records = Cf.hashMap();
            databases = Cf.hashMap();
        }

        @Override
        public int getRecordsCount(UserDatabaseSpec databaseSpec, RecordsFilter filter) {
            return getRecords(databaseSpec, filter).size();
        }

        @Override
        public Database getOrCreateDatabase(UserDatabaseSpec spec) {
            return databases.getOrElseUpdate(spec.databaseRef(), () -> new Database(
                    spec.uid(),
                    new DatabaseHandle(spec.databaseRef(), Random2.R.nextAlnum(30)),
                    0,
                    new DatabaseMeta(Instant.now(), Instant.now(), DataSize.ZERO, 0),
                    DatabaseAccessType.READ_WRITE
            ));
        }

        @Override
        public Database applyDelta(Database database, RevisionCheckMode revCheckMode, Delta delta) {
            PatchableSnapshot patchableSnapshot =
                    new Snapshot(database, getRecords(database.spec(), RecordsFilter.DEFAULT)).toPatchable();

            Snapshot finalSnapshot = patchableSnapshot.patch(delta).unmodifiable();

            records.put(database.dbRef(), finalSnapshot.records.records().toList());

            Database finalDatabase = finalSnapshot.database;
            databases.put(database.dbRef(), finalDatabase);
            return finalDatabase;
        }

        @Override
        public ListF<DataRecord> getRecords(UserDatabaseSpec databaseSpec, RecordsFilter filter) {
            ListF<DataRecord> filtered = records.getOrElse(databaseSpec.databaseRef(), Cf.list());
            filtered = filtered.filter(filter::matches);
            filtered = filtered.sorted(filter.getRecordOrder().comparator());

            SqlLimits limits = filter.limits();
            if (limits.isAll()) {
                return filtered;
            } else {
                int toIndex = limits.getCount() == Integer.MAX_VALUE ?
                        filtered.size() :
                        Math.min(limits.getFirst() + limits.getCount(), filtered.size());
                int fromIndex = Math.min(limits.getFirst(), filtered.size());
                return filtered.subList(fromIndex, toIndex);
            }
        }
    }
}
