package ru.yandex.chemodan.app.lentaloader.cool;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.joda.time.DateTime;
import org.joda.time.DateTimeFieldType;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.json.JSONObject;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaModelUtils;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaBlock;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.misc.net.LocalhostUtils;

import static ru.yandex.chemodan.app.lentaloader.cool.CoolLentaManager.ALL_BLOCKS_DB_REF;

@RequiredArgsConstructor
public class MordaPushManager {

    private final DynamicProperty<Integer> startHour = new DynamicProperty<>("morda-push-hour-start", 10);
    private final DynamicProperty<Integer> endHour = new DynamicProperty<>("morda-push-hour-end", 18);
    private final DynamicProperty<Integer> pushInterval = new DynamicProperty<>("morda-push-interval", 0);
    private final DynamicProperty<Integer> pushTtl = new DynamicProperty<>("morda-push-ttl", 3600);
    private final DynamicProperty<Integer> pushImageTtl = new DynamicProperty<>("morda-push-image-ttl", 4*3600);
    private final DynamicProperty<String> pushImageOverride = new DynamicProperty<>("morda-push-image-override", "");
    private final DynamicProperty<String> pushFilter = new DynamicProperty<>("morda-push-filter", "AND app_id LIKE 'ru.yandex.searchplugin%' AND NOT (apps IN ('ru.yandex.disk')) AND platform == 'android'");
    private final DynamicProperty<String> pushProject = new DynamicProperty<>("morda-push-project", "disk");
    private final DynamicProperty<Boolean> pushCheckLocale = new DynamicProperty<>("morda-push-check-locale", true);
    private final DynamicProperty<ListF<Long>> pushForceUids = new DynamicProperty<>("morda-push-force-uids", Cf.list());
    private final BleedingEdge pushEnabled = new BleedingEdge("morda-push");

    private final Blackbox2 blackbox;
    private final SupClient supClient;
    private final MpfsClient mpfsClient;
    private final DataApiManager dataApiManager;
    private final UserTimezoneHelper tzHelper;
    private final CoolLentaConfigurationManager coolLentaConfigurationManager;

    public ListF<MinimalCoolLentaBlock> getAllBlocks(PassportUid uid) {
        UserDatabaseSpec spec = new UserDatabaseSpec(new DataApiPassportUserId(uid), ALL_BLOCKS_DB_REF);

        return dataApiManager
                .getRecords(spec, RecordsFilter.DEFAULT.withCollectionId(CoolLentaManager.ALL_BLOCKS_COLLECTION_ID))
                .map(CoolLentaModelUtils::recordToBlock);
    }

    public boolean isRussianLocale(PassportUid uid) {
        return !pushCheckLocale.get() || blackbox.query()
                .userInfo(LocalhostUtils.localAddress(), uid, Cf.list(BlackboxDbFields.LANG))
                .getDbFields()
                .getO(BlackboxDbFields.LANG)
                .isSome("ru");
    }

    public boolean shouldSendPush(PassportUid uid, Option<ListF<MinimalCoolLentaBlock>> allBlocksO) {
        if (pushEnabled.isOnBleedingEdge(uid) && pushInterval.get() > 0 && isRussianLocale(uid)) {
            Option<Instant> lastPush = allBlocksO.getOrElse(() -> getAllBlocks(uid))
                    .filter(MinimalCoolLentaBlock::isPushSent)
                    .flatMap(MinimalCoolLentaBlock::getLastShowDate)
                    .maxO();
            if (lastPush.isPresent()) {
                return lastPush.isMatch(t -> t.isBefore(Instant.now().minus(Duration.standardDays(pushInterval.get()))));
            } else {
                return uid.getUid() % pushInterval.get() == Instant.now().get(DateTimeFieldType.dayOfYear()) % pushInterval.get();
            }
        } else {
            return false;
        }
    }

    public boolean isForced(PassportUid uid) {
        return pushForceUids.get().containsTs(uid.getUid());
    }

    @SneakyThrows
    public void sendPush(PassportUid uid, MpfsResourceId bestResourceId, String mordaBlockId) {
        String icon = Option.of(pushImageOverride.get()).filterNot(String::isEmpty).getOrElse(() -> mpfsClient.generateZaberunUrl(
                mpfsClient.getFileInfoByFileId(
                        MpfsUser.of(bestResourceId.owner),
                        bestResourceId.fileId
                ).getMeta().getPmid().get(),
                "image.jpg",
                "preview",
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.of("M"),
                Option.empty(),
                Option.empty(),
                Option.of(pushImageTtl.get()),
                Option.empty(),
                Option.empty(),
                Option.empty()));
        MapF<String, Object> data = Cf.<String, Object>map()
                .plus1("receiver", Cf.list(String.format("tag:uid==%s %s", uid.getUid(), isForced(uid) ? "" : pushFilter.get())))
                .plus1("ttl", pushTtl.get())
                .plus1("notification", Cf.map(
                        "title", "Диск: воспоминания",
                        "body", "Смотрите ваши снимки из прошлого",
                        "icon", icon,
                        "iconId", "1"
                ))
                .plus1("data", Cf.map(
                        "push_id", "disk_new_photo_selection",
                        "push_uri", coolLentaConfigurationManager.getBaseMordaBlockLink(mordaBlockId) + "?from=search_push",
                        "push_action", "uri"
                ))
                .plus1("android_features", Cf.map(
                        "style", "system"
                ))
                .plus1("is_data_only", isForced(uid))
                .plus1("project", pushProject.get());
        getPushDate(uid).ifPresent(date -> {
            data.put("schedule", date.toString());
            data.put("adjust_time_zone", "false");
        });
        supClient.sendPush(JSONObject.wrap(data).toString());
    }

    public Option<DateTime> getPushDate(PassportUid uid) {
        DateTimeZone tz = tzHelper.getUserTimezone(uid);
        DateTime now = Instant.now().toDateTime(tz);
        int hour = startHour.get() + (int)(Math.random() * (endHour.get() - startHour.get()));
        if (now.getHourOfDay() < startHour.get()) {
            return Option.of(now.withHourOfDay(hour));
        } else if (now.getHourOfDay() > endHour.get()) {
            return Option.of(now.withHourOfDay(hour).plusDays(1));
        } else {
            return Option.empty();
        }
    }

    public boolean mordaBlockAdded(PassportUid uid, MinimalCoolLentaBlock block, String mordaBlockId) {
        if (isForced(uid) || shouldSendPush(uid, Option.empty())) {
            sendPush(uid, block.bestResourceId, mordaBlockId);
            return true;
        } else {
            return false;
        }
    }

}
