package ru.yandex.chemodan.app.lentaloader.cool;

import org.joda.time.DateTime;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BlockGeneratorUtils;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchResponse;
import ru.yandex.misc.db.embedded.sandbox.SandBoxResourceRule;
import ru.yandex.misc.time.InstantInterval;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

public class SimilarityFilteringTest extends AbstractCoolLentaManagerTest {

    @ClassRule
    public static final SandBoxResourceRule resourceRule =
            createSandboxResourceRule("1055604097");

    private static final DateTime START_TIME_INTERVAL = new DateTime(2016, 5, 20, 11, 13, 20);
    private static final DateTime END_TIME_INTERVAL = new DateTime(2019, 7, 21, 21, 0, 0);

    private static DiskSearchResponse TEST_DATA;

    @BeforeClass
    public static void init() {
        TEST_DATA = parseTestData(resourceRule);
    }

    @Test
    public void testFiltering() {
        CoolLentaManager manager = createManager(true, false);
        ListF<CoolLentaBlock> blocks =  manager
                .generateAllBlocks(getUid(), START_TIME_INTERVAL, END_TIME_INTERVAL,
                        manager.getActualBlockGenerators(IntervalType.WEEK, getUid().getUid()));
        assertFalse(blocks.isEmpty());
        blocks.map(CoolLentaBlock::getItems).forEach(this::checkNonSimilarity);
        blocks.map(CoolLentaBlock::getItems).forEach(this::isOrderedByEtime);
    }

    @Test
    public void testDisabledFiltering() {
        CoolLentaManager manager = getManager();
        ListF<CoolLentaBlock> blocks = manager
                .generateAllBlocks(getUid(), START_TIME_INTERVAL, END_TIME_INTERVAL,
                        manager.getActualBlockGenerators(IntervalType.WEEK, getUid().getUid()));
        assertFalse(blocks.isEmpty());
        assertTrue(blocks.map(CoolLentaBlock::getItems).exists(this::hasSimilarities));
        blocks.map(CoolLentaBlock::getItems).forEach(this::isOrderedByEtime);
    }

    @Test
    public void testFullFiltering() {
        CoolLentaManager manager = getManager();
        ListF<CoolLentaFileItem> items = Cf.toArrayList(manager.getFiles(getUid(),
                new InstantInterval(START_TIME_INTERVAL, END_TIME_INTERVAL),
                Option.of("cool-lenta"), -2.0));
        int initialSize = items.size();
        long periodLimit = 86_400_000L;
        ListF<CoolLentaFileItem> removedItems = BlockGeneratorUtils.removeSimilarItemsInWindow(items, 11200L,
                periodLimit, 1000);
        checkNonSimilarityInInterval(items, periodLimit);
        assertEquals(Cf.toHashSet(items.map(CoolLentaFileItem::getSearchFileInfo)
                .map(DiskSearchFileInfo::getId)).size(), items.size());
        isOrderedByEtime(items);
        assertEquals(initialSize, items.size() + removedItems.size());
    }

    @Override
    protected DiskSearchResponse getTestData() {
        return TEST_DATA;
    }

    private void checkNonSimilarity(ListF<CoolLentaFileItem> items) {
        assertTrue(items.size() > 1);
        for (int i = 0; i < items.size() - 1; i++) {
            CoolLentaFileItem currentItem = items.get(i);
            for (int j = i + 1; j < items.size(); j++) {
                CoolLentaFileItem itemToCompare = items.get(j);
                long dotProduct = Cvi2tProcessor
                        .dotProduct(currentItem.getSearchFileInfo().geti2tVector(),
                                itemToCompare.getSearchFileInfo().geti2tVector());
                assertTrue(String.format("items `%s` and `%s` are too similar. Dot product is %s",
                        currentItem.getSearchFileInfo().getId(), itemToCompare.getSearchFileInfo().getId(), dotProduct),
                        dotProduct <= getManager().similarityThreshold.get());
            }
        }
    }

    private void checkNonSimilarityInInterval(ListF<CoolLentaFileItem> items, long intervalInMs) {
        assertTrue(items.size() > 1);
        for (int i = 0; i < items.size() - 1; i++) {
            CoolLentaFileItem currentItem = items.get(i);
            for (int j = i + 1; j < items.size(); j++) {
                CoolLentaFileItem itemToCompare = items.get(j);
                if (itemToCompare.getUserEtime().getMillis() - currentItem.getUserEtime().getMillis() > intervalInMs) {
                    break;
                }
                long dotProduct = Cvi2tProcessor
                        .dotProduct(currentItem.getSearchFileInfo().geti2tVector(),
                                itemToCompare.getSearchFileInfo().geti2tVector());
                assertTrue(String.format("items `%s` and `%s` are too similar. Dot product is %s",
                        currentItem.getSearchFileInfo().getId(), itemToCompare.getSearchFileInfo().getId(), dotProduct),
                        dotProduct <= getManager().similarityThreshold.get());
            }
        }
    }

    private void isOrderedByEtime(ListF<CoolLentaFileItem> items) {
       if (items.size() < 2) {
           return;
       }
       for (int i = 1; i < items.size(); i++) {
           assertTrue(items.get(i).getUserEtime().compareTo(items.get(i - 1).getUserEtime()) >= 0);
       }
    }

    private boolean hasSimilarities(ListF<CoolLentaFileItem> items) {
        for (int i = 0; i < items.size() - 1; i++) {
            CoolLentaFileItem currentItem = items.get(i);
            for (int j = i + 1; j < items.size(); j++) {
                CoolLentaFileItem itemToCompare = items.get(j);
                long dotProduct = Cvi2tProcessor
                        .dotProduct(currentItem.getSearchFileInfo().geti2tVector(),
                                itemToCompare.getSearchFileInfo().geti2tVector());
                if (dotProduct > getManager().similarityThreshold.get()) {
                    return true;
                }
            }
        }
        return false;
    }
}
