package ru.yandex.chemodan.app.lentaloader.cool;

import java.util.Objects;

import org.joda.time.DateTime;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.lentaloader.cool.generator.CoolLentaBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.DefaultBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThematicBlocksGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.generator.WordMatch;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TermLanguageDefinition;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchResponse;
import ru.yandex.chemodan.app.lentaloader.test.TestUtils;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.db.embedded.sandbox.SandBoxResourceRule;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

public class ThematicBlocksTest extends AbstractCoolLentaManagerTest {

    @ClassRule
    public static final SandBoxResourceRule resourceRule =
            createSandboxResourceRule("1055609325");

    private static final DateTime START_TIME_INTERVAL = new DateTime(2016, 5, 20, 11, 13, 20);
    private static final DateTime END_TIME_INTERVAL = new DateTime(2019, 7, 30, 13, 36, 40);

    private static DiskSearchResponse TEST_DATA;

    @BeforeClass
    public static void init() {
        TEST_DATA = parseTestData(resourceRule);
    }

    @Test
    public void testLongThematicBlocks() {
        CoolLentaManager manager = createManagerWithMinThematicBlocksCount(6);
        ListF<CoolLentaBlockGenerator> generators = manager
                .getActualBlockGenerators(IntervalType.WHOLE_TIME, getUid().getUid());
        assertFalse(generators.filter(ThematicBlocksGenerator.class::isInstance).isEmpty());

        ListF<CoolLentaBlock> blocks = manager.generateAllBlocks(getUid(), START_TIME_INTERVAL, END_TIME_INTERVAL,
                generators);
        assertFalse(blocks.isEmpty());
        assertFalse(blocks.find(coolLentaBlock -> coolLentaBlock.getGenerationType().startsWith("thematic")).isPresent());
    }

    @Test
    public void testThemeDefinitionJson() {
        ThemeDefinition theme = TestUtils.createNatureThemeDefinition();
        String json = new String(Bender.jsonSerializer(ThemeDefinition.class).serializeJson(theme));
        ThemeDefinition deserializedTheme = Bender.jsonParser(ThemeDefinition.class).parseJson(json);
        assertEquals(theme.getName(), deserializedTheme.getName());

        ListF<WordMatch> sourceWords = theme.getWords();
        ListF<WordMatch> jsonWords = deserializedTheme.getWords();
        assertEquals(sourceWords.size(), jsonWords.size());
        assertTrue(sourceWords.filterNot(word -> jsonWords
                .find(jsonWord -> word.getWord().equals(jsonWord.getWord()) &&
                        word.getSimilarityThreshold() == jsonWord.getSimilarityThreshold()).isPresent()).isEmpty());

        MapF<Language, TermLanguageDefinition> sourceTermsDefinition = theme.getForms().getLanguageTerms();
        MapF<Language, TermLanguageDefinition> jsonTermsDefinition = deserializedTheme.getForms().getLanguageTerms();
        assertEquals(sourceTermsDefinition.size(), jsonTermsDefinition.size());
        assertTrue(sourceTermsDefinition
                .filter((language, termLanguageDefinition) ->
                        termDefinitionDoesNotContainLanguage(language, termLanguageDefinition, jsonTermsDefinition)).isEmpty());

        assertTrue(theme.isEnabled() == deserializedTheme.isEnabled());
    }

    @Test
    public void testEnabledThematicBlocks() {
        CoolLentaManager manager = createManager(false, false);
        ListF<CoolLentaBlockGenerator> generators = manager
                .getActualBlockGenerators(IntervalType.WHOLE_TIME, getUid().getUid());
        assertFalse(generators.filter(ThematicBlocksGenerator.class::isInstance).isEmpty());
        assertFalse(generators.filter(DefaultBlockGenerator.class::isInstance).isEmpty());

        ListF<CoolLentaBlock> blocks = manager.generateAllBlocks(getUid(), START_TIME_INTERVAL, END_TIME_INTERVAL,
                generators);
        assertFalse(blocks.isEmpty());
        assertTrue(blocks
                .find(coolLentaBlock ->
                        "thematic_WHOLE_TIME".equals(coolLentaBlock.getGenerationType()) &&
                        coolLentaBlock.getId().endsWith("_nature"))
                .isPresent());
        assertFalse(blocks.find(coolLentaBlock -> coolLentaBlock.getId().endsWith("_subject")).isPresent());
    }

    private boolean termDefinitionDoesNotContainLanguage(Language language, TermLanguageDefinition termLanguageDefinition,
            MapF<Language, TermLanguageDefinition> termDefinitions)
    {
        return !termDefinitions.getO(language)
                .filter(term -> Objects.equals(termLanguageDefinition.getGenitive(), term.getGenitive()) &&
                        Objects.equals(termLanguageDefinition.getNominative(), term.getNominative()) &&
                        Objects.equals(termLanguageDefinition.getPreposition(), term.getPreposition()) &&
                        Objects.equals(termLanguageDefinition.getPrepositionalCase(), term.getPrepositionalCase())).isPresent();
    }

    @Override
    protected DiskSearchResponse getTestData() {
        return TEST_DATA;
    }
}
