package ru.yandex.chemodan.app.lentaloader.cool.generator;

import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaFileItem;
import ru.yandex.chemodan.app.lentaloader.cool.HasEtimeAndBeauty;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;


/**
 * @author tolmalev
 */
public class BlockGeneratorUtils {
    static final int MINIMAL_JOIN_INTERVAL = 5;

    public static MapF<DateTime, ListF<CoolLentaFileItem>> groupByDay(ListF<CoolLentaFileItem> items) {
        return items.groupBy(item -> TimeIntervalUtils.getDayStart(item.userEtime));
    }

    public static MapF<DateTime, ListF<CoolLentaFileItem>> groupByWeek(ListF<CoolLentaFileItem> items) {
        return items.groupBy(item -> TimeIntervalUtils.getWeekStart(item.userEtime));
    }

    public static CoolLentaFileItem selectOneBest(ListF<CoolLentaFileItem> items) {
        return selectBest(items, 1, BeautySelectorType.BOTH).first();
    }

    public static <T extends HasEtimeAndBeauty> ListF<T> selectBest(ListF<T> items, int count,
            BeautySelectorType selectorType)
    {
        if (items.size() <= count) {
            return items;
        }

        switch (selectorType) {
            case NEW: return sortByNewBeautyDesc(items).take(count);
            case OLD: return sortByOldBeautyDesc(items).take(count);
            case BOTH: {
                int cntNewBeauty = items.count(i -> i.getNewBeauty().isPresent());
                int cntOldBeauty = items.count(i -> !i.getNewBeauty().isPresent() && i.getOldBeauty().isPresent());

                int selectCntByOld = count * cntOldBeauty / (cntNewBeauty + cntOldBeauty);
                int selectCntByNew = count - selectCntByOld;

                ListF<T> part1 = sortByNewBeautyDesc(items).take(selectCntByNew);
                if (part1.length() >= count) {
                    return part1;
                }

                ListF<T> part2 = sortByOldBeautyDesc(items.unique().minus(part1).toList()).take(selectCntByOld);

                return part1.plus(part2);
            }
            default:
                throw new IllegalStateException("Unknown beauty select type " + selectorType);
        }
    }

    public static <T extends HasEtimeAndBeauty> ListF<T> sortByNewBeautyDesc(ListF<T> items) {
        return items
                .filter(i -> i.getNewBeauty().isPresent())
                .sortedByDesc(i -> i.getNewBeauty().get());
    }

    public static <T extends HasEtimeAndBeauty> ListF<T> sortByOldBeautyDesc(ListF<T> items) {
        return items
                .filter(i -> i.getOldBeauty().isPresent())
                .sortedByDesc(i -> i.getOldBeauty().get());
    }

    public static ListF<CoolLentaFileItem> smartJoinGroups(ListF<CoolLentaFileItem> source, int preferredCount) {
        source = filterByNearGroups(source, MINIMAL_JOIN_INTERVAL);
        if (source.size() < preferredCount) {
            return source;
        }

        SetF<CoolLentaFileItem> result = Cf.hashSet();

        for (int interval : Cf.list(7200, 3600, 1800, 900, 450, 220, 120, 60)) {
            ListF<CoolLentaFileItem> groupped = filterByNearGroups(source, interval);
            result.addAll(groupped);

            if (result.size() > preferredCount) {
                return sortByEtime(result.toList());
            }
        }
        return sortByEtime(result.toList());
    }

    public static <T extends HasEtimeAndBeauty> ListF<T> filterByNearGroups(ListF<T> source, long joinDistanceSeconds) {
        long deltaMillis = joinDistanceSeconds * 1000;
        ListF<T> sorted = sortByEtime(source);

        ListF<ListF<T>> groups = Cf.arrayList();

        ListF<T> currentGroup = Cf.arrayList();
        for (int i = 0; i < sorted.size(); i++) {
            if (i == 0 || (sorted.get(i).getUserEtime().getMillis() - sorted.get(i - 1).getUserEtime().getMillis() > deltaMillis)) {
                if (!currentGroup.isEmpty()) {
                    groups.add(currentGroup);
                }
                currentGroup = Cf.arrayList();
            }
            currentGroup.add(sorted.get(i));
        }
        if (!currentGroup.isEmpty()) {
            groups.add(currentGroup);
        }

        return groups.flatMap(group -> selectBest(group, 1, BeautySelectorType.BOTH));
    }

    public static <T extends HasEtimeAndBeauty> ListF<T> sortByEtime(ListF<T> items) {
        return items.sortedBy(HasEtimeAndBeauty::getUserEtime);
    }

    public static Tuple2<IntervalType, DateTime> getInterval(ListF<CoolLentaFileItem> items) {
        DateTime startDate = items.first().userEtime;
        DateTime endDate = items.last().userEtime;

        IntervalType intervalType = TimeIntervalUtils.getMinimalInterval(startDate, endDate);

        DateTime intervalStart = intervalType.getIntervalStart(startDate);

        return Tuple2.tuple(intervalType, intervalStart);
    }

    /**
     *
     * @param items must be mutable and sorted by etime (ASC)
     * @param similarityThreshold
     * @return removed items
     */
    public static ListF<CoolLentaFileItem> removeSimilarItems(ListF<CoolLentaFileItem> items,
            long similarityThreshold)
    {
        if (!items.isSortedBy(CoolLentaFileItem::getUserEtime)) {
            throw new IllegalArgumentException("Items must be sorted by user Etime (ASC)");
        }
        ListF<CoolLentaFileItem> removedItems = Cf.arrayList();
        int i = 0;
        while (i < items.size() - 1) {
            CoolLentaFileItem currentItem = items.get(i);
            ListF<CoolLentaFileItem> itemsToCompareBeauty = items.drop(i + 1).filter(item ->
                    Cvi2tProcessor.dotProduct(currentItem.getSearchFileInfo().geti2tVector(),
                            item.getSearchFileInfo().geti2tVector()) > similarityThreshold);
            if (itemsToCompareBeauty.isEmpty()) {
                i++;
                continue;
            }
            itemsToCompareBeauty = Cf.toArrayList(itemsToCompareBeauty);
            itemsToCompareBeauty.add(currentItem);
            itemsToCompareBeauty = sortByNewBeautyDesc(itemsToCompareBeauty);
            CoolLentaFileItem newItem = itemsToCompareBeauty.first();
            removedItems.addAll(removeItemsStaringFromIndex(items, itemsToCompareBeauty, i + 1));
            if (newItem == currentItem) {
                i++;
                continue;
            }
            items.set(i, newItem);
            restoreEtimeOrder(items, i);
        }
        return removedItems.unmodifiable();
    }

    private static ListF<CoolLentaFileItem> removeItemsStaringFromIndex(ListF<CoolLentaFileItem> sourceItems,
            ListF<CoolLentaFileItem> itemsToRemove, int startIndex) {
        if (itemsToRemove.isEmpty()) {
            return Cf.list();
        }
        ListF<CoolLentaFileItem> removedItems = Cf.arrayList();
        int i = startIndex;
        while (i < sourceItems.size()) {
            CoolLentaFileItem item = sourceItems.get(i);
            if (itemsToRemove.containsTs(item)) {
                removedItems.add(item);
                sourceItems.remove(i);
                continue;
            }
            i++;
        }
        return removedItems.unmodifiable();
    }

    /**
     *
     * @param allItems must be mutable and sorted by user Etime (ASC)
     * @param similarityThreshold
     * @param limitPeriodInMs limits the items to check by period after in ms
     * @param maxCountToCompare limits the items to check by count (is the top priority limit)
     * @return removed items
     */
    public static ListF<CoolLentaFileItem> removeSimilarItemsInWindow(ListF<CoolLentaFileItem> allItems,
            long similarityThreshold, long limitPeriodInMs, int maxCountToCompare)
    {
        if (!allItems.isSortedBy(CoolLentaFileItem::getUserEtime)) {
            throw new IllegalArgumentException("Items must be sorted by user Etime (ASC)");
        }
        ListF<CoolLentaFileItem> itemsToRemove = Cf.arrayList();
        int i = 0;
        while (i < allItems.size() - 1) {
            CoolLentaFileItem currentItem = allItems.get(i);
            long currentItemEtime = currentItem.getUserEtime().getMillis();
            ListF<CoolLentaFileItem> itemsToCompareBeauty = Cf.arrayList();
            for (int j = i + 1; j < allItems.size(); j++) {
                CoolLentaFileItem itemToCompare = allItems.get(j);
                if ((j - (i + 1) > maxCountToCompare) ||
                        itemToCompare.getUserEtime().getMillis() - currentItemEtime > limitPeriodInMs) {
                    break;
                }
                if (Cvi2tProcessor.dotProduct(currentItem.getSearchFileInfo().geti2tVector(),
                        itemToCompare.getSearchFileInfo().geti2tVector()) > similarityThreshold) {
                    itemsToCompareBeauty.add(itemToCompare);
                }
            }
            if (itemsToCompareBeauty.isEmpty()) {
                i++;
                continue;
            }
            itemsToCompareBeauty.add(currentItem);
            CoolLentaFileItem newItem = sortByNewBeautyDesc(itemsToCompareBeauty).first();
            itemsToRemove.addAll(removeItemsFromListInSublist(allItems, itemsToCompareBeauty, i + 1,
                    maxCountToCompare, currentItemEtime, limitPeriodInMs));
            if (newItem == currentItem) {
                i++;
                continue;
            }
            allItems.set(i, newItem);
            restoreEtimeOrder(allItems, i);
        }
        return itemsToRemove.unmodifiable();
    }

    private static void restoreEtimeOrder(ListF<CoolLentaFileItem> items, int currentItemIndex) {
        if (currentItemIndex >= items.size() - 1) {
            return;
        }
        CoolLentaFileItem currentItem = items.get(currentItemIndex);
        CoolLentaFileItem nextItem = items.get(currentItemIndex + 1);
        while (currentItemIndex < items.size() - 1 &&
                currentItem.getUserEtime().getMillis() > nextItem.getUserEtime().getMillis()) {
            CoolLentaFileItem swapItem = currentItem;
            items.set(currentItemIndex, nextItem);
            items.set(currentItemIndex + 1, swapItem);
            currentItemIndex++;
            if (currentItemIndex < items.size() - 1) {
                nextItem = items.get(currentItemIndex + 1);
            }
        }
    }

    private static ListF<CoolLentaFileItem> removeItemsFromListInSublist(ListF<CoolLentaFileItem> allItems,
            ListF<CoolLentaFileItem> itemsToRemove, int startIndex, int maxCount, long currentItemPosition,
            long limitPeriodInMs)
    {
        if (itemsToRemove.isEmpty()) {
            return Cf.list();
        }
        ListF<CoolLentaFileItem> removedItems = Cf.arrayList();
        int i = startIndex;
        int j = 0;
        while (i < allItems.size()) {
            CoolLentaFileItem item = allItems.get(i);
            j++;
            if (j > maxCount || item.getUserEtime().getMillis() - currentItemPosition > limitPeriodInMs) {
                break;
            }
            if (itemsToRemove.containsTs(item)) {
                removedItems.add(item);
                allItems.remove(i);
                continue;
            }
            i++;
        }
        return removedItems.unmodifiable();
    }

    /**
     *
     * @param items must be mutable
     * @return the items that has been removed
     */
    public static ListF<CoolLentaFileItem> removeItemsByStopWords(ListF<CoolLentaFileItem> items, ListF<WordMatch> stopWords,
            Cvi2tProcessor cvi2tProcessor)
    {
        ListF<CoolLentaFileItem> itemsToRemove = Cf.arrayList();
        MapF<String, byte[]> vectorsForWords = stopWords.toMap(WordMatch::getWord,
                wordMatch -> cvi2tProcessor.getTextVector(wordMatch.getWord()));
        int i = 0;
        while (i < items.size()) {
            CoolLentaFileItem item = items.get(i);
            if (containsAnyOfWordsContent(item, stopWords, vectorsForWords)) {
                items.remove(i);
                itemsToRemove.add(item);
                continue;
            }
            i++;
        }
        return itemsToRemove.unmodifiable();
    }

    public static boolean containsAnyOfWordsContent(CoolLentaFileItem item, ListF<WordMatch> words,
            MapF<String, byte[]> vectorsForWords)
    {
        return words.exists(word -> Cvi2tProcessor.dotProduct(item.getSearchFileInfo().geti2tVector(),
                vectorsForWords.getOrThrow(word.getWord())) > word.getSimilarityThreshold());
    }

}
