package ru.yandex.chemodan.app.lentaloader.cool.generator;

import lombok.Data;
import org.apache.commons.lang3.mutable.MutableLong;
import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaFileItem;
import ru.yandex.chemodan.app.lentaloader.cool.ExperimentMembersByOverrideManager;
import ru.yandex.chemodan.app.lentaloader.cool.HasEtimeAndBeauty;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.commune.bazinga.impl.worker.ThreadCpuUsageCalculator;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class ThematicBlocksGenerator extends AbstractCoolLentaBlockGenerator {

    public static final String BLOCK_ID_PREFIX = "thematic";

    private static final String EXPERIMENTAL_THEMES_EXPERIMENT_NAME = "disk_cool_lenta_experimental_themes_generating";
    private static final Logger logger = LoggerFactory.getLogger(ThematicBlocksGenerator.class);

    private final ThemeDefinitionRegistry themeDefinitionRegistry;
    private final Cvi2tProcessor cvi2tProcessor;
    private final ExperimentMembersByOverrideManager experimentMembersByOverrideManager;
    private final DynamicProperty<Integer> minCount;
    private final DynamicProperty<Integer> maxCount;
    private final DynamicProperty<Boolean> enableWeekGeneration;
    private final DynamicProperty<Double> beautyLimit =
            new DynamicProperty<>("cool-lenta-hard-new-beauty-limit", -2.0);
    private final DynamicProperty<Integer> relevanceLimit =
            new DynamicProperty<>("cool-lenta-thematic-relevance-limit", 40);

    public ThematicBlocksGenerator(ThemeDefinitionRegistry themeDefinitionRegistry, Cvi2tProcessor cvi2tProcessor,
            ExperimentMembersByOverrideManager experimentMembersByOverrideManager, int minCount, int maxCount,
            boolean enableWeekGeneration)
    {
        this.themeDefinitionRegistry = themeDefinitionRegistry;
        this.cvi2tProcessor = cvi2tProcessor;
        this.experimentMembersByOverrideManager = experimentMembersByOverrideManager;
        this.minCount = new DynamicProperty<>("cool-lenta-thematic-block-items-count-min", minCount);
        this.maxCount = new DynamicProperty<>("cool-lenta-thematic-block-items-count-max", maxCount);
        this.enableWeekGeneration = new DynamicProperty<>("cool-lenta-thematic-block-week-generation-enable",
                enableWeekGeneration);
    }

    @Override
    public String generatorIdPrefix() {
        return BLOCK_ID_PREFIX;
    }

    @Override
    public boolean isAcceptableIntervalType(IntervalType intervalType) {
        return (enableWeekGeneration.get() && intervalType == IntervalType.WEEK) ||
                intervalType == IntervalType.MONTH ||
                intervalType == IntervalType.SEASON ||
                intervalType == IntervalType.YEAR ||
                intervalType == IntervalType.WHOLE_TIME;
    }

    @Override
    public ListF<BlockGeneratingResult> generateBlocksWithoutIntervalCheck(ListF<CoolLentaFileItem> allItems,
            Option<PassportUid> uidO)
    {
        return generateBlocksByRelevance(allItems, uidO.getOrThrow("No uid provided for thematic block generator"));
    }

    @Override
    public ListF<String> generateAllPossibleIds(IntervalType intervalType, DateTime intervalStart) {
        if (!isAcceptableIntervalType(intervalType)) {
            return Cf.list();
        }
        return themeDefinitionRegistry.getAll().map(ThemeDefinition::getName)
                .map(theme -> generateBlockId(intervalType, intervalStart, Option.of(theme)));
    }

    @Override
    public int getMinSizeForBlock() {
        return minCount.get();
    }

    @Override
    public double getBeautyLimit() {
        double defaultBeautyLimit = beautyLimit.get();
        return Math.min(defaultBeautyLimit, themeDefinitionRegistry.getAll().filter(ThemeDefinition::isEnabled)
                .filter(themeDefinition -> themeDefinition.getMinBeauty().isPresent())
                .map(themeDefinition -> themeDefinition.getMinBeauty().get()).minO().getOrElse(defaultBeautyLimit));
    }

    private ListF<BlockGeneratingResult> generateBlocksByRelevance(ListF<CoolLentaFileItem> allItems, PassportUid uid) {
        ListF<CoolLentaFileItem> filteredByNearGroupsItems = BlockGeneratorUtils.filterByNearGroups(allItems, getJoinIntervalSeconds());
        MapF<String, ProcessingResult> itemsWithRelevanceProcessingResult = getThemesAvailableForUser(uid)
                .toMap(ThemeDefinition::getName, theme -> getItemsWithRelevanceForTheme(theme, filteredByNearGroupsItems));
        logger.debug("CPU usage statistics during relevance calculation per theme {} itemsCount={}",
                itemsWithRelevanceProcessingResult.mapValues(ProcessingResult::getCpuUsagePercentage), filteredByNearGroupsItems.size());
        MapF<String, ProcessingResult> sortingByRelevanceProcessingResult = itemsWithRelevanceProcessingResult
                .mapValues(ProcessingResult::getItems).filterValues(items -> items.length() >= minCount.get())
                .mapValues(this::sortByRelevance);
        logger.debug("CPU usage statistics during relevance sorting per theme {}",
                sortingByRelevanceProcessingResult.mapValues(ProcessingResult::getCpuUsagePercentage));
        return sortingByRelevanceProcessingResult.mapValues(ProcessingResult::getItems)
                .mapValues(itemWithRelevance -> itemWithRelevance.map(ItemWithRelevance::getItem))
                .mapValues(items -> BlockGeneratorUtils.smartJoinGroups(items, maxCount.get()))
                .mapValues(items -> BlockGeneratorUtils.sortByEtime(items).take(maxCount.get()))
                .mapEntries((theme, items) -> new BlockGeneratingResult(Option.of(theme), items));
    }

    private CollectionF<ThemeDefinition> getThemesAvailableForUser(PassportUid uid) {
        boolean isExperimentMember = experimentMembersByOverrideManager
                .isExperimentMember(uid, EXPERIMENTAL_THEMES_EXPERIMENT_NAME);
        return themeDefinitionRegistry.getAll()
                .filter(ThemeDefinition::isEnabled)
                .filter(themeDefinition -> !themeDefinition.isExperimentalForGenerating() || isExperimentMember);
    }

    private ProcessingResult getItemsWithRelevanceForTheme(ThemeDefinition theme, ListF<CoolLentaFileItem> allItems) {
        if (theme.getMinBeauty().isPresent()) {
            allItems = allItems.filter(item -> item.getNewBeauty().get() >= theme.getMinBeauty().get());
        }
        MapF<String, byte[]> vectorsForWords = theme.getWords().toMap(WordMatch::getWord,
                wordMatch -> cvi2tProcessor.getTextVector(wordMatch.getWord()));
        ThreadCpuUsageCalculator cpuUsageCalculator = new ThreadCpuUsageCalculator();
        ListF<ItemWithRelevance> items = allItems.map(item -> createItemWithRelevance(item, theme.getWords(), vectorsForWords))
                .filter(itemWithRelevance -> itemWithRelevance.getRelevance().isPresent());
        return new ProcessingResult(items, cpuUsageCalculator.getCpuUsagePercentage());
    }

    private ProcessingResult sortByRelevance(ListF<ItemWithRelevance> items) {
        ThreadCpuUsageCalculator cpuUsageCalculator = new ThreadCpuUsageCalculator();
        items = items.sortedByDesc(itemWithRelevance -> itemWithRelevance.getRelevance().get())
                .take(relevanceLimit.get());
        return new ProcessingResult(items, cpuUsageCalculator.getCpuUsagePercentage());
    }

    private ItemWithRelevance createItemWithRelevance(CoolLentaFileItem item, ListF<WordMatch> words,
            MapF<String, byte[]> vectorsForWords)
    {
        MutableLong maxRelevance = new MutableLong(0);
        words.forEach(word -> setRelevanceForItem(item, word, maxRelevance, vectorsForWords));
        long relevance = maxRelevance.longValue();
        return new ItemWithRelevance(Option.when(relevance > 0, relevance), item);
    }

    private void setRelevanceForItem(CoolLentaFileItem item, WordMatch word, MutableLong currentRelevance,
            MapF<String, byte[]> vectorsForWords)
    {
        long relevance = Cvi2tProcessor
                .dotProduct(vectorsForWords.getOrThrow(word.getWord()), item.getSearchFileInfo().geti2tVector());
        if (relevance >= word.getSimilarityThreshold()) {
            currentRelevance.setValue(Math.max(currentRelevance.longValue(), relevance));
        }
    }

    @Data
    private static class ItemWithRelevance implements HasEtimeAndBeauty {
        private final Option<Long> relevance;
        private final CoolLentaFileItem item;

        @Override
        public DateTime getUserEtime() {
            return item.getUserEtime();
        }

        @Override
        public Option<Double> getOldBeauty() {
            return item.getOldBeauty();
        }

        @Override
        public Option<Double> getNewBeauty() {
            return item.getNewBeauty();
        }
    }

    @Data
    private static class ProcessingResult {
        private final ListF<ItemWithRelevance> items;
        private final int cpuUsagePercentage;
    }
}
