package ru.yandex.chemodan.app.lentaloader.cool.imageparser;

import java.io.IOException;
import java.net.URI;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.jetbrains.annotations.NotNull;

import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Instrument;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author tolmalev
 */
public class ImageparserClientImpl implements ImageparserClient, MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(ImageparserClientImpl.class);
    private static final BenderJsonParser<ImageparserResponse> responseParser = Bender.jsonParser(ImageparserResponse.class);

    private final Pattern avatarsStidPattern = Pattern.compile("^ava:([^:]+):([^:]+):([^:]+)$");

    private final String imageparserHost;
    private final HttpClient httpClient;

    @MonicaMetric
    @GroupByDefault
    private final Instrument invocations = new Instrument();

    public ImageparserClientImpl(String imageparserHost, HttpClient httpClient) {
        this.imageparserHost = imageparserHost;
        this.httpClient = httpClient;
    }

    @Override
    public ImageparserResponse parseImage(MulcaId mulcaId) {
        return parseImage(mulcaId.getStid());
    }

    @Override
    public ImageparserResponse parseImage(String mulcaId) {
        return invocations.measure(() -> {
            HttpGet request = consRequest(mulcaId);
            return ApacheHttpClientUtils.execute(request, httpClient, new Abstract200ExtendedResponseHandler<ImageparserResponse>() {
                @Override
                protected ImageparserResponse handle200Response(HttpResponse response) throws IOException {
                    return responseParser.parseJson(InputStreamSourceUtils.wrap(response.getEntity().getContent()));
                }
            });
        });
    }

    @NotNull
    private HttpGet consRequest(String mulcaId) {
        Matcher m = avatarsStidPattern.matcher(mulcaId);
        if (m.matches()) {
            String mdsKey = m.group(2) + "/" + m.group(3);

            URI url = UriBuilder.cons(imageparserHost)
                    .appendPath("process")
                    .appendPath("handler")
                    .addParam("stid", "avatars-disk/" + mdsKey + "/1280_nocrop/webp")
                    .build();

            HttpGet request = new HttpGet(url);
            request.addHeader("X-Srw-Key-Type", "AVAID");
            request.addHeader("X-Srw-Namespace", "avatars-disk");
            request.addHeader("X-Srw-Key", mdsKey);

            return request;
        } else {
            URI url = UriBuilder.cons(imageparserHost)
                    .appendPath("process")
                    .addParam("stid", mulcaId)
                    .build();

            HttpGet request = new HttpGet(url);
            request.addHeader("X-Srw-Key-Type", "STID");
            request.addHeader("X-Srw-Namespace", "disk");
            request.addHeader("X-Srw-Key", mulcaId);

            return request;
        }
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "imageparser-client",
                new MetricName("imageparser", "client"),
                "imageparser http client"
        );
    }
}
