package ru.yandex.chemodan.app.lentaloader.cool.model;

import lombok.Data;
import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaFileItem;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BlockGeneratorUtils;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author tolmalev
 */
@Data
@BenderBindAllFields
public class CoolLentaBlock {
    public final String id;
    public final String generationType;

    public final ListF<CoolLentaFileItem> items;

    public final Option<DateTime> photosliceDate;
    public final Option<DateTime> lastShowDate;
    public final Option<DateTime> legacyLentaLastShowDate;
    public final Option<DateTime> legacyLentaLastPushDate;

    public final boolean pushSent;

    public MinimalCoolLentaBlock toMinimalBlock() {
        ListF<MpfsResourceId> resourceIds = items.map(item -> item.mpfsFileInfo.get().getMeta().getResourceId().get());
        MpfsResourceId bestResourceId =
                BlockGeneratorUtils.selectOneBest(items).getMpfsFileInfo().get().getMeta().getResourceId().get();

        Option<Instant> photosliceDateO = photosliceDate.map(DateTime::toInstant);
        Option<Instant> lastShowDateO = lastShowDate.map(DateTime::toInstant);
        Option<Instant> legacyLentaLastShowDateO = legacyLentaLastShowDate.map(DateTime::toInstant);
        Option<Instant> legacyLentaLastPushDateO = legacyLentaLastPushDate.map(DateTime::toInstant);
        Instant minDate = items.map(i -> i.getUserEtime().toInstant()).min();
        Instant maxDate = items.map(i -> i.getUserEtime().toInstant()).max();

        String userTimezoneId = items.first().getUserEtime().getZone().getID();

        return new MinimalCoolLentaBlock(
                id, generationType,
                resourceIds, bestResourceId,
                userTimezoneId,
                photosliceDateO, minDate, maxDate,
                lastShowDateO, legacyLentaLastShowDateO, legacyLentaLastPushDateO,
                pushSent
        );
    }

    public CoolLentaBlock withItemsAndPhotosliceDate(ListF<CoolLentaFileItem> items, Option<DateTime> photosliceDate) {
        return new CoolLentaBlock(
                this.id,
                this.generationType,
                items,
                photosliceDate,
                this.lastShowDate,
                this.legacyLentaLastShowDate,
                this.legacyLentaLastPushDate,
                this.pushSent
        );
    }

    public CoolLentaBlock withItems(ListF<CoolLentaFileItem> items) {
        return new CoolLentaBlock(
                this.id,
                this.generationType,
                items,
                Option.when(BlockGeneratorUtils.getInterval(items)._1 != IntervalType.YEAR, items.first().getUserEtime()),
                this.lastShowDate,
                this.legacyLentaLastShowDate,
                this.legacyLentaLastPushDate,
                this.pushSent
        );
    }

    public CoolLentaBlock withLastShowAndPushDates(Option<DateTime> lastShowDate, Option<DateTime> legacyLentaLastShowDate,
            Option<DateTime> legacyLentaLastPushDate)
    {
        return new CoolLentaBlock(
                this.id,
                this.generationType,
                this.items,
                this.photosliceDate,
                lastShowDate,
                legacyLentaLastShowDate,
                legacyLentaLastPushDate,
                this.pushSent
        );
    }
}
