package ru.yandex.chemodan.app.lentaloader.cool.model;

import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaManager;
import ru.yandex.chemodan.mpfs.MpfsResourceId;

/**
 * @author tolmalev
 */
public class CoolLentaModelUtils {

    public static MinimalCoolLentaBlock recordToBlock(DataRecord record) {
        return new MinimalCoolLentaBlock(
                record.getRecordId(),
                CoolLentaBlockFields.GENERATION_TYPE.getO(record).getOrElse("unknown"),
                CoolLentaBlockFields.RESOURCE_IDS.get(record).map(MpfsResourceId::parse),
                MpfsResourceId.parse(CoolLentaBlockFields.BEST_RESOURCE_ID.get(record)),
                CoolLentaBlockFields.USER_TIMEZONE_ID.get(record),
                CoolLentaBlockFields.PHOTOSLICE_DATE.getO(record),
                CoolLentaBlockFields.MIN_DATE.get(record),
                CoolLentaBlockFields.MAX_DATE.get(record),
                CoolLentaBlockFields.LAST_SHOW_DATE.getO(record),
                CoolLentaBlockFields.LEGACY_LENTA_LAST_SHOW_DATE.getO(record),
                CoolLentaBlockFields.LEGACY_LENTA_LAST_PUSH_DATE.getO(record),
                CoolLentaBlockFields.PUSH_SENT.getO(record).getOrElse(false)
        );
    }

    public static MinimalCoolLentaMordaBlock recordToMordaBlock(DataRecord record) {
        return recordToBlock(record).toMordaBlock(record.id.recordId(), CoolLentaBlockFields.MTIME.get(record));
    }

    public static RecordChange blockToRecordChangeForAllBlocks(MinimalCoolLentaBlock block) {
        ListF<Tuple2<String, DataField>> dataFields = blockDataFields(block);

        return RecordChange.set(
                CoolLentaManager.ALL_BLOCKS_COLLECTION_ID,
                block.id,
                Cf.toMap(dataFields)
        );
    }

    public static RecordChange blockToRecordChangeForMordaBlocks(MinimalCoolLentaMordaBlock block) {
        ListF<Tuple2<String, DataField>> dataFields = getDataFieldsForMorda(block);

        return RecordChange.set(
                CoolLentaManager.MORDA_BLOCKS_COLLECTION_ID,
                block.id,
                Cf.toMap(dataFields)
        );
    }

    public static ListF<Tuple2<String, DataField>> getDataFieldsForMorda(MinimalCoolLentaMordaBlock block) {
        return blockDataFields(block)
                .plus1(CoolLentaBlockFields.ALL_BLOCKS_ID.toData(block.allBlocksId))
                .filterNot(e -> e._1.equals(CoolLentaBlockFields.MTIME.name))
                .plus1(CoolLentaBlockFields.MTIME.toData(block.mtime));
    }

    public static RecordChange removeMordaBlockChange(String mordaBlockId) {
        return RecordChange.delete(
                CoolLentaManager.MORDA_BLOCKS_COLLECTION_ID,
                mordaBlockId
        );
    }

    public static RecordChange updateLastShowDate(String blockId, Instant lastShowTime) {
        Tuple2<String, DataField> data = CoolLentaBlockFields.LAST_SHOW_DATE.toData(lastShowTime);

        return RecordChange.update(
                CoolLentaManager.ALL_BLOCKS_COLLECTION_ID,
                blockId,
                FieldChange.put(data._1, data._2)
        );
    }

    public static RecordChange updatePushSent(String blockId, boolean pushSent) {
        Tuple2<String, DataField> data = CoolLentaBlockFields.PUSH_SENT.toData(pushSent);

        return RecordChange.update(
                CoolLentaManager.ALL_BLOCKS_COLLECTION_ID,
                blockId,
                FieldChange.put(data._1, data._2)
        );
    }

    public static RecordChange updateLegacyLentaLastShowDate(String blockId, Instant lastShowTime) {
        Tuple2<String, DataField> data = CoolLentaBlockFields.LEGACY_LENTA_LAST_SHOW_DATE.toData(lastShowTime);

        return RecordChange.update(
                CoolLentaManager.ALL_BLOCKS_COLLECTION_ID,
                blockId,
                FieldChange.put(data._1, data._2)
        );
    }

    public static RecordChange updateLegacyLentaLastPushDate(String blockId, Instant lastPushTime) {
        Tuple2<String, DataField> data = CoolLentaBlockFields.LEGACY_LENTA_LAST_PUSH_DATE.toData(lastPushTime);

        return RecordChange.update(
                CoolLentaManager.ALL_BLOCKS_COLLECTION_ID,
                blockId,
                FieldChange.put(data._1, data._2)
        );
    }

    @NotNull
    public static ListF<Tuple2<String, DataField>> blockDataFields(MinimalCoolLentaBlock block) {
        ListF<Tuple2<String, DataField>> dataFields = Cf.arrayList(
                CoolLentaBlockFields.GENERATION_TYPE.toData(block.generationType),
                CoolLentaBlockFields.MTIME.toData(Instant.now()),

                CoolLentaBlockFields.RESOURCE_IDS.toData(block.resourceIds.map(MpfsResourceId::serialize)),
                CoolLentaBlockFields.BEST_RESOURCE_ID.toData(block.bestResourceId.serialize()),

                CoolLentaBlockFields.USER_TIMEZONE_ID.toData(block.userTimezoneId),
                CoolLentaBlockFields.MIN_DATE.toData(block.minDate),
                CoolLentaBlockFields.MAX_DATE.toData(block.maxDate),

                CoolLentaBlockFields.PUSH_SENT.toData(block.pushSent)
        );

        dataFields.addAll(block.photosliceDate.map(
                date -> CoolLentaBlockFields.PHOTOSLICE_DATE.toData(date.toInstant())));

        dataFields.addAll(block.legacyLentaLastShowDate.map(CoolLentaBlockFields.LEGACY_LENTA_LAST_SHOW_DATE::toData));

        dataFields.addAll(block.getLegacyLentaLastPushDate().map(CoolLentaBlockFields.LEGACY_LENTA_LAST_PUSH_DATE::toData));

        dataFields.addAll(block.lastShowDate.map(CoolLentaBlockFields.LAST_SHOW_DATE::toData));

        return dataFields;
    }
}
